import os
import json
from enum import Enum
from typing import Dict, Any, Union
from importlib.resources import files

class DateFormat: pass # pragma: no cover
class DateTimeFormat: pass # pragma: no cover
class PasswordFormat: pass # pragma: no cover
class ByteFormat: pass # pragma: no cover
class EmailFormat: pass # pragma: no cover
class UuidFormat: pass # pragma: no cover
class UrlFormat: pass # pragma: no cover
class HostnameFormat: pass # pragma: no cover
class Ipv4Format: pass # pragma: no cover
class Ipv6Format: pass # pragma: no cover
class Int32Format: pass # pragma: no cover
class Int64Format: pass # pragma: no cover
class FloatFormat: pass # pragma: no cover
class DoubleFormnat: pass # pragma: no cover

class BaseStorage:
    def __init__(self, data: dict[str, (str, int)]):
        self.data = data or {}

    def get(self, key: str, /, default: Any = None) -> Union[Dict[str, Any], Any]:
        try:
            return json.loads(self.data.get(key, default))
        except (TypeError, json.JSONDecodeError):
            return self.data.get(key, default)
    
    def keys(self):
        return list(self.data.keys())
    
    def values(self):
        return list(self.__filter_data().values())
    
    def items(self):
        return list(self.__filter_data().items())
    
    def __filter_data(self):
        try:
            return {key: json.loads(value) for key, value in self.data.items()}
        
        except (TypeError, json.JSONDecodeError):
            return self.data
    
    def __repr__(self): # pragma: no cover
        return repr(self.__filter_data())

class Colors: # pragma: no cover
    RESET = "\033[0m"
    BOLD = "\033[1m"

    # Cores básicas
    RED = "\033[91m"
    GREEN = "\033[92m"
    YELLOW = "\033[93m"
    BLUE = "\033[94m"
    MAGENTA = "\033[95m"
    CYAN = "\033[96m"
    WHITE = "\033[97m"

    # Cores em fundo
    BG_RED = "\033[41m"
    BG_GREEN = "\033[42m"
    BG_YELLOW = "\033[43m"
    BG_BLUE = "\033[44m"
    BG_MAGENTA = "\033[45m"
    BG_CYAN = "\033[46m"
    BG_WHITE = "\033[47m"

    # Cores com negrito
    BOLD_RED = "\033[1;91m"
    BOLD_GREEN = "\033[1;92m"
    BOLD_YELLOW = "\033[1;93m"
    BOLD_BLUE = "\033[1;94m"
    BOLD_MAGENTA = "\033[1;95m"
    BOLD_CYAN = "\033[1;96m"
    BOLD_WHITE = "\033[1;97m"

class JWTAlgorithms(Enum): # pragma: no cover
    HS256 = 'HS256'
    HS384 = 'HS384'
    HS512 = 'HS512'
    RS256 = 'RS256'
    RS384 = 'RS384'
    RS512 = 'RS512'
    ES256 = 'ES256'
    ES384 = 'ES384'
    ES512 = 'ES512'
    PS256 = 'PS256'
    PS384 = 'PS384'
    PS512 = 'PS512'

class OrientationType(Enum): # pragma: no cover
    LANDSCAPE_PRIMARY = "landscape-primary"
    LANDSCAPE_SECONDARY = "landscape-secondary"
    PORTRAIT_PRIMARY = "portrait-primary"
    PORTRAIT_SECONDARY = "portrait-secondary"

class ContentTypes(Enum): # pragma: no cover
    # -------- TEXT TYPES --------
    html = "text/html"
    css = "text/css"
    js = "application/javascript"
    json = "application/json"
    xml = "application/xml"
    txt = "text/plain"
    csv = "text/csv"
    rtf = "application/rtf"
    form_encode = "application/x-www-form-urlencoded"
    form_data = 'multipart/form-data'

    # Documentos de texto Office (abríveis como texto em alguns casos)
    doc = "application/msword"
    docx = "application/vnd.openxmlformats-officedocument.wordprocessingml.document"
    xls = "application/vnd.ms-excel"
    xlsx = "application/vnd.openxmlformats-officedocument.spreadsheetml.sheet"
    ppt = "application/vnd.ms-powerpoint"
    pptx = "application/vnd.openxmlformats-officedocument.presentationml.presentation"
    
    # -------- BYTE TYPES (BINÁRIOS) --------
    png = "image/png"
    jpg = "image/jpeg"
    jpeg = "image/jpeg"
    gif = "image/gif"
    svg = "image/svg+xml"
    ico = "image/x-icon"
    webp = "image/webp"
    bmp = "image/bmp"
    tif = "image/tiff"
    tiff = "image/tiff"

    # Fontes
    woff = "font/woff"
    woff2 = "font/woff2"
    ttf = "font/ttf"
    otf = "font/otf"
    eot = "application/vnd.ms-fontobject"

    # Vídeos
    mp4 = "video/mp4"
    webm = "video/webm"
    ogv = "video/ogg"
    avi = "video/x-msvideo"
    mov = "video/quicktime"
    flv = "video/x-flv"
    mkv = "video/x-matroska"
    m4v = "video/x-m4v"
    mpeg = "video/mpeg"

    # Áudios
    mp3 = "audio/mpeg"
    ogg = "audio/ogg"
    wav = "audio/wav"
    m4a = "audio/mp4"
    flac = "audio/flac"
    aac = "audio/aac"

    # PDFs e arquivos compactados
    pdf = "application/pdf"
    zip = "application/zip"
    zip7 = "application/x-7z-compressed"
    rar = "application/vnd.rar"
    tar = "application/x-tar"
    gz = "application/gzip"

    # Desconhecido (fallback)
    unkown = "application/octet-stream"

    @classmethod
    def content_list(cls) -> list[str]:
        return [value.name for value in cls]

class StaticFilePath(Enum): # pragma: no cover
    framework = 'pyweber'
    html_base = files(framework).joinpath(os.path.join('static', 'html.html'))
    html_404 = files(framework).joinpath(os.path.join('static', 'html404.html'))
    html_500 = files(framework).joinpath(os.path.join('static', 'html500.html'))
    html_401 = files(framework).joinpath(os.path.join('static', 'html401.html'))
    js_base = files(framework).joinpath(os.path.join('static', 'js.js'))
    css_base = files(framework).joinpath(os.path.join('static', 'css.css'))
    main_base = files(framework).joinpath(os.path.join('static', 'main.py'))
    favicon_path = files(framework).joinpath(os.path.join('static', 'favicon'))
    config_default = files(framework).joinpath(os.path.join('static', 'config.toml'))
    pyweber_css = files(framework).joinpath(os.path.join('static', 'pyweber.css'))
    update_file = files(framework).joinpath(os.path.join('static', 'update.py'))
    admin_page = files(framework).joinpath(os.path.join('admin', 'index.html'))
    admin_css_file = files(framework).joinpath(os.path.join('admin', 'src', 'style.css'))
    admin_js_file = files(framework).joinpath(os.path.join('admin', 'src', 'script.js'))
    pyweber_docs = files(framework).joinpath(os.path.join('static', 'docs.html'))
    pyweber_loading = files(framework).joinpath(os.path.join('static', 'loading.html'))

    @classmethod
    def all_static_files(cls):
        return [value.value for value in cls]

class WebSocketStatusCode(Enum): # pragma: no cover
    NORMAL_CLOSURE = "1000 Normal Closure"
    GOING_AWAY = "1001 Going Away"
    PROTOCOL_ERROR = "1002 Protocol Error"
    UNSUPPORTED_DATA = "1003 Unsupported Data"
    NO_STATUS_RECEIVED = "1005 No Status Received"
    ABNORMAL_CLOSURE = "1006 Abnormal Closure"
    INVALID_FRAME_PAYLOAD_DATA = "1007 Invalid Frame Payload Data"
    POLICY_VIOLATION = "1008 Policy Violation"
    MESSAGE_TOO_BIG = "1009 Message Too Big"
    MANDATORY_EXTENSION = "1010 Mandatory Extension"
    INTERNAL_SERVER_ERROR = "1011 Internal Server Error"
    SERVICE_RESTART = "1012 Service Restart"
    TRY_AGAIN_LATER = "1013 Try Again Later"
    BAD_GATEWAY = "1014 Bad Gateway"
    TLS_HANDSHAKE = "1015 TLS Handshake Failure"

    @classmethod
    def content_list(cls) -> list[int]:
        return [int(value.value.split(' ')[0]) for value in cls]
    
    @classmethod
    def search_by_code(cls, code: int):
        for value in cls:
            if str(code) in value.value:
                return value.value
        
        return cls.INTERNAL_SERVER_ERROR.value

class HTTPStatusCode(Enum): # pragma: no cover
    # Informational Responses (1xx)
    CONTINUE = "100 Continue"
    SWITCHING_PROTOCOLS = "101 Switching Protocols"
    PROCESSING = "102 Processing"

    # Successful Responses (2xx)
    OK = "200 OK"
    CREATED = "201 Created"
    ACCEPTED = "202 Accepted"
    NON_AUTHORITATIVE_INFORMATION = "203 Non-Authoritative Information"
    NO_CONTENT = "204 No Content"
    RESET_CONTENT = "205 Reset Content"
    PARTIAL_CONTENT = "206 Partial Content"

    # Redirection Messages (3xx)
    MULTIPLE_CHOICES = "300 Multiple Choices"
    MOVED_PERMANENTLY = "301 Moved Permanently"
    FOUND = "302 Found"
    SEE_OTHER = "303 See Other"
    NOT_MODIFIED = "304 Not Modified"
    USE_PROXY = "305 Use Proxy"
    TEMPORARY_REDIRECT = "307 Temporary Redirect"
    PERMANENT_REDIRECT = "308 Permanent Redirect"

    # Client Error Responses (4xx)
    BAD_REQUEST = "400 Bad Request"
    UNAUTHORIZED = "401 Unauthorized"
    FORBIDDEN = "402 Forbidden"
    NOT_FOUND = "404 Not Found"
    METHOD_NOT_ALLOWED = "405 Method Not Allowed"
    NOT_ACCEPTABLE = "406 Not Acceptable"
    PROXY_AUTHENTICATION_REQUIRED = "407 Proxy Authentication Required"
    REQUEST_TIMEOUT = "408 Request Timeout"
    CONFLICT = "409 Conflict"
    GONE = "410 Gone"
    LENGTH_REQUIRED = "411 Length Required"
    PRECONDITION_FAILED = "412 Precondition Failed"
    PAYLOAD_TOO_LARGE = "413 Payload Too Large"
    URI_TOO_LONG = "414 URI Too Long"
    UNSUPPORTED_MEDIA_TYPE = "415 Unsupported Media Type"
    RANGE_NOT_SATISFIABLE = "416 Range Not Satisfiable"
    EXPECTATION_FAILED = "417 Expectation Failed"
    IM_A_TEAPOT = "418 I'm a teapot (joke HTTP status code)"
    MISDIRECTED_REQUEST = "421 Misdirected Request"
    UNPROCESSABLE_ENTITY = "422 Unprocessable Entity"
    LOCKED = "423 Locked"
    FAILED_DEPENDENCY = "424 Failed Dependency"
    TOO_EARLY = "425 Too Early"
    UPGRADE_REQUIRED = "426 Upgrade Required"
    PRECONDITION_REQUIRED = "428 Precondition Required"
    TOO_MANY_REQUESTS = "429 Too Many Requests"
    REQUEST_HEADER_FIELDS_TOO_LARGE = "431 Request Header Fields Too Large"
    UNAVAILABLE_FOR_LEGAL_REASONS = "451 Unavailable For Legal Reasons"

    # Server Error Responses (5xx)
    INTERNAL_SERVER_ERROR = "500 Internal Server Error"
    NOT_IMPLEMENTED = "501 Not Implemented"
    BAD_GATEWAY = "502 Bad Gateway"
    SERVICE_UNAVAILABLE = "503 Service Unavailable"
    GATEWAY_TIMEOUT = "504 Gateway Timeout"
    VERSION_NOT_SUPPORTED = "505 Version Not Supported"
    VARIANT_ALSO_NEGOTIATES = "506 Variant Also Negotiates"
    INSUFFICIENT_STORAGE = "507 Insufficient Storage"
    LOOP_DETECTED = "508 Loop Detected"
    NOT_EXTENDED = "510 Not Extended"
    NETWORK_AUTHENTICATION_REQUIRED = "511 Network Authentication Required"

    @classmethod
    def code_list(cls) -> list[int]:
        return [int(value.value.split(' ')[0]) for value in cls]
    
    @classmethod
    def search_by_code(cls, code: int) -> str:
        for value in cls:
            if str(code) in value.value:
                return value.value
        
        raise Warning(f'{code} is not an httpstatus code')
    
    @classmethod
    def search_name_by_code(cls, code: int):
        for value in cls:
            if str(code) in value.value:
                return value.name
        
        raise Warning(f'{code} is not an httpstatus code')

class EventType(Enum): # pragma: no cover
    # Mouse Events
    CLICK = "onclick"
    DBLCLICK = "ondblclick"
    MOUSEDOWN = "onmousedown"
    MOUSEUP = "onmouseup"
    MOUSEMOVE = "onmousemove"
    MOUSEOVER = "onmouseover"
    MOUSEOUT = "onmouseout"
    MOUSEENTER = "onmouseenter"
    MOUSELEAVE = "onmouseleave"
    CONTEXTMENU = "oncontextmenu"
    WHEEL = "onwheel"

    # Keyboard Events
    KEYDOWN = "onkeydown"
    KEYUP = "onkeyup"
    KEYPRESS = "onkeypress"

    # Form Events
    FOCUS = "onfocus"
    BLUR = "onblur"
    CHANGE = "onchange"
    INPUT = "oninput"
    SUBMIT = "onsubmit"
    RESET = "onreset"
    SELECT = "onselect"

    # Drag & Drop Events
    DRAG = "ondrag"
    DRAGSTART = "ondragstart"
    DRAGEND = "ondragend"
    DRAGOVER = "ondragover"
    DRAGENTER = "ondragenter"
    DRAGLEAVE = "ondragleave"
    DROP = "ondrop"

    # Scroll & Resize Events
    SCROLL = "onscroll"
    RESIZE = "onresize"

    # Media Events
    PLAY = "onplay"
    PAUSE = "onpause"
    ENDED = "onended"
    VOLUMECHANGE = "onvolumechange"
    SEEKED = "onseeked"
    SEEKING = "onseeking"
    TIMEUPDATE = "ontimeupdate"

    # Network Events
    ONLINE = "ononline"
    OFFLINE = "onoffline"

    # Touch Events (Mobile)
    TOUCHSTART = "ontouchstart"
    TOUCHMOVE = "ontouchmove"
    TOUCHEND = "ontouchend"
    TOUCHCANCEL = "ontouchcancel"

class WindowEventType(Enum): # pragma: no cover
    # Eventos de Janela e Navegação
    AFTER_PRINT = "onafterprint"
    BEFORE_PRINT = "onbeforeprint"
    BEFORE_UNLOAD = "onbeforeunload"
    HASH_CHANGE = "onhashchange"
    LOAD = "onload"
    UNLOAD = "onunload"
    PAGE_SHOW = "onpageshow"
    PAGE_HIDE = "onpagehide"
    POP_STATE = "onpopstate"
    DOM_CONTENT_LOADED = "onDOMContentLoaded"

    # Eventos de Rede
    ONLINE = "ononline"
    OFFLINE = "onoffline"

    # Eventos de Armazenamento
    STORAGE = "onstorage"

    # Eventos de Mensagens e Comunicação
    MESSAGE = "onmessage"
    MESSAGE_ERROR = "onmessageerror"

    # Eventos de Animação e Transição
    ANIMATION_START = "onanimationstart"
    ANIMATION_END = "onanimationend"
    ANIMATION_ITERATION = "onanimationiteration"
    TRANSITION_START = "ontransitionstart"
    TRANSITION_END = "ontransitionend"
    TRANSITION_CANCEL = "ontransitioncancel"

    # Eventos de Fullscreen e Pointer Lock
    FULLSCREEN_CHANGE = "onfullscreenchange"
    FULLSCREEN_ERROR = "onfullscreenerror"
    POINTER_LOCK_CHANGE = "onpointerlockchange"
    POINTER_LOCK_ERROR = "onpointerlockerror"

    # Eventos de Dispositivo
    DEVICE_MOTION = "ondevicemotion"
    DEVICE_ORIENTATION = "ondeviceorientation"
    DEVICE_ORIENTATION_ABSOLUTE = "ondeviceorientationabsolute"
    ORIENTATION_CHANGE = "onorientationchange"

    # Eventos de Gamepad
    GAMEPAD_CONNECTED = "ongamepadconnected"
    GAMEPAD_DISCONNECTED = "ongamepaddisconnected"

    # Eventos de VR
    VR_DISPLAY_CONNECT = "onvrdisplayconnect"
    VR_DISPLAY_DISCONNECT = "onvrdisplaydisconnect"
    VR_DISPLAY_PRESENT_CHANGE = "onvrdisplaypresentchange"
    VR_DISPLAY_ACTIVATE = "onvrdisplayactivate"
    VR_DISPLAY_DEACTIVATE = "onvrdisplaydeactivate"
    VR_DISPLAY_BLUR = "onvrdisplayblur"
    VR_DISPLAY_FOCUS = "onvrdisplayfocus"
    VR_DISPLAY_POINTER_RESTRICTED = "onvrdisplaypointerrestricted"
    VR_DISPLAY_POINTER_UNRESTRICTED = "onvrdisplaypointerunrestricted"

    # Eventos de Service Worker e Cache
    INSTALL = "oninstall"
    ACTIVATE = "onactivate"
    FETCH = "onfetch"
    NOTIFICATION_CLICK = "onnotificationclick"
    NOTIFICATION_CLOSE = "onnotificationclose"
    PUSH = "onpush"
    PUSH_SUBSCRIPTION_CHANGE = "onpushsubscriptionchange"
    SYNC = "onsync"
    PERIODIC_SYNC = "onperiodicsync"
    BACKGROUND_FETCH_SUCCESS = "onbackgroundfetchsuccess"
    BACKGROUND_FETCH_FAILURE = "onbackgroundfetchfailure"
    BACKGROUND_FETCH_ABORT = "onbackgroundfetchabort"
    BACKGROUND_FETCH_CLICK = "onbackgroundfetchclick"
    CONTENT_DELETE = "oncontentdelete"

    # Eventos de Clipboard
    CUT = "oncut"
    COPY = "oncopy"
    PASTE = "onpaste"

    # Eventos de Seleção de Texto
    SELECTION_CHANGE = "onselectionchange"

    # Eventos de Visibilidade
    VISIBILITY_CHANGE = "onvisibilitychange"

    # Eventos de Rejeição de Promises
    REJECTION_HANDLED = "onrejectionhandled"
    UNHANDLED_REJECTION = "onunhandledrejection"

    # Eventos de Segurança
    SECURITY_POLICY_VIOLATION = "onsecuritypolicyviolation"

    # Eventos do pointer e mouse
    POINTEROVER = "onpointerover"
    POINTERENTER = "onpointerenter"
    POINTERDOWN = "onpointerdown"
    POINTERMOVE = "onpointermove"
    POINTERUP = "onpointerup"
    POINTERCANCEL = "onpointercancel"
    POINTEROUT = "onpointerout"
    POINTERLEAVE = "onpointerleave"
    GOTPOINTERCAPTURE = "ongotpointercapture"
    LOSTPOINTERCAPTURE = "onlostpointercapture"

class HTMLTag(Enum): # pragma: no cover
    # Tags Semânticas e de Estruturação
    html = "html"
    head = "head"
    body = "body"
    title = "title"
    script = "script"
    noscript = "noscript"
    section = "section"
    nav = "nav"
    article = "article"
    aside = "aside"
    header = "header"
    footer = "footer"
    address = "address"
    main = "main"
    div = "div"
    span = "span"
    
    # Tags de Formatação de Texto
    p = "p"
    h1 = "h1"
    h2 = "h2"
    h3 = "h3"
    h4 = "h4"
    h5 = "h5"
    h6 = "h6"
    b = "b"
    strong = "strong"
    i = "i"
    em = "em"
    mark = "mark"
    small = "small"
    DEL = "del"
    ins = "ins"
    sub = "sub"
    sup = "sup"
    blockquote = "blockquote"
    cite = "cite"
    q = "q"
    code = "code"
    pre = "pre"
    kbd = "kbd"
    samp = "samp"
    var = "var"
    
    # Tags de Listas
    ul = "ul"
    ol = "ol"
    li = "li"
    dl = "dl"
    dt = "dt"
    dd = "dd"
    
    # Tags de Tabelas
    table = "table"
    caption = "caption"
    thead = "thead"
    tbody = "tbody"
    tfoot = "tfoot"
    tr = "tr"
    th = "th"
    td = "td"
    
    # Tags de Formulário
    form = "form"
    label = "label"
    input = "input"
    textarea = "textarea"
    button = "button"
    fieldset = "fieldset"
    legend = "legend"
    select = "select"
    optgroup = "optgroup"
    option = "option"
    datalist = "datalist"
    output = "output"
    progress = "progress"
    meter = "meter"
    
    # Tags de Mídia e Integração
    audio = "audio"
    video = "video"
    source = "source"
    track = "track"
    object = "object"
    param = "param"
    embed = "embed"
    iframe = "iframe"
    canvas = "canvas"
    svg = "svg"
    math = "math"
    
    # Tags de Metadados
    base = "base"
    link = "link"
    meta = "meta"
    style = "style"
    
    # Tags de Conteúdo Interativo
    details = "details"
    summary = "summary"
    dialog = "dialog"
    menu = "menu"
    menuitem = "menuitem"

    # Tags de Comentários
    comment = "comment"

class NonSelfClosingHTMLTags(Enum): # pragma: no cover
    # Tags Semânticas e de Estruturação
    html = "html"
    head = "head"
    body = "body"
    title = "title"
    script = "script"
    style = "style"
    noscript = "noscript"
    section = "section"
    nav = "nav"
    article = "article"
    aside = "aside"
    header = "header"
    footer = "footer"
    address = "address"
    main = "main"
    div = "div"
    span = "span"
    
    # Tags de Formatação de Texto
    p = "p"
    h1 = "h1"
    h2 = "h2"
    h3 = "h3"
    h4 = "h4"
    h5 = "h5"
    h6 = "h6"
    b = "b"
    strong = "strong"
    i = "i"
    em = "em"
    mark = "mark"
    small = "small"
    DEL = "del"
    ins = "ins"
    sub = "sub"
    sup = "sup"
    blockquote = "blockquote"
    cite = "cite"
    q = "q"
    code = "code"
    pre = "pre"
    kbd = "kbd"
    samp = "samp"
    var = "var"
    
    # Tags de Listas
    ul = "ul"
    ol = "ol"
    li = "li"
    dl = "dl"
    dt = "dt"
    dd = "dd"
    
    # Tags de Tabelas
    table = "table"
    caption = "caption"
    thead = "thead"
    tbody = "tbody"
    tfoot = "tfoot"
    tr = "tr"
    th = "th"
    td = "td"
    
    # Tags de Formulário
    form = "form"
    label = "label"
    textarea = "textarea"
    button = "button"
    fieldset = "fieldset"
    legend = "legend"
    select = "select"
    optgroup = "optgroup"
    option = "option"
    datalist = "datalist"
    output = "output"
    progress = "progress"
    meter = "meter"
    
    # Tags de Mídia e Integração
    audio = "audio"
    video = "video"
    object = "object"
    iframe = "iframe"
    canvas = "canvas"
    svg = "svg"
    math = "math"
    
    # Tags de Conteúdo Interativo
    details = "details"
    summary = "summary"
    dialog = "dialog"
    menu = "menu"
    menuitem = "menuitem"
    
    # Tags de Comentários
    comment = "comment"

    @classmethod
    def non_autoclosing_tags(cls) -> list[str]:
        return [val.value for val in cls]

class GetBy(Enum): # pragma: no cover
    tag = 'tag'
    classes = 'classes'
    id = 'id'
    content = 'content'
    value = 'value'
    attrs = 'attrs'
    style = 'style'
    uuid = 'uuid'

class Icons(Enum): # pragma: no cover
    _0_CIRCLE_FILL = 'bi bi-0-circle-fill'
    _0_CIRCLE = 'bi bi-0-circle'
    _0_SQUARE_FILL = 'bi bi-0-square-fill'
    _0_SQUARE = 'bi bi-0-square'
    _1_CIRCLE_FILL = 'bi bi-1-circle-fill'
    _1_CIRCLE = 'bi bi-1-circle'
    _1_SQUARE_FILL = 'bi bi-1-square-fill'
    _1_SQUARE = 'bi bi-1-square'
    _123 = 'bi bi-123'
    _2_CIRCLE_FILL = 'bi bi-2-circle-fill'
    _2_CIRCLE = 'bi bi-2-circle'
    _2_SQUARE_FILL = 'bi bi-2-square-fill'
    _2_SQUARE = 'bi bi-2-square'
    _3_CIRCLE_FILL = 'bi bi-3-circle-fill'
    _3_CIRCLE = 'bi bi-3-circle'
    _3_SQUARE_FILL = 'bi bi-3-square-fill'
    _3_SQUARE = 'bi bi-3-square'
    _4_CIRCLE_FILL = 'bi bi-4-circle-fill'
    _4_CIRCLE = 'bi bi-4-circle'
    _4_SQUARE_FILL = 'bi bi-4-square-fill'
    _4_SQUARE = 'bi bi-4-square'
    _5_CIRCLE_FILL = 'bi bi-5-circle-fill'
    _5_CIRCLE = 'bi bi-5-circle'
    _5_SQUARE_FILL = 'bi bi-5-square-fill'
    _5_SQUARE = 'bi bi-5-square'
    _6_CIRCLE_FILL = 'bi bi-6-circle-fill'
    _6_CIRCLE = 'bi bi-6-circle'
    _6_SQUARE_FILL = 'bi bi-6-square-fill'
    _6_SQUARE = 'bi bi-6-square'
    _7_CIRCLE_FILL = 'bi bi-7-circle-fill'
    _7_CIRCLE = 'bi bi-7-circle'
    _7_SQUARE_FILL = 'bi bi-7-square-fill'
    _7_SQUARE = 'bi bi-7-square'
    _8_CIRCLE_FILL = 'bi bi-8-circle-fill'
    _8_CIRCLE = 'bi bi-8-circle'
    _8_SQUARE_FILL = 'bi bi-8-square-fill'
    _8_SQUARE = 'bi bi-8-square'
    _9_CIRCLE_FILL = 'bi bi-9-circle-fill'
    _9_CIRCLE = 'bi bi-9-circle'
    _9_SQUARE_FILL = 'bi bi-9-square-fill'
    _9_SQUARE = 'bi bi-9-square'
    ACTIVITY = 'bi bi-activity'
    AIRPLANE_ENGINES_FILL = 'bi bi-airplane-engines-fill'
    AIRPLANE_ENGINES = 'bi bi-airplane-engines'
    AIRPLANE_FILL = 'bi bi-airplane-fill'
    AIRPLANE = 'bi bi-airplane'
    ALARM_FILL = 'bi bi-alarm-fill'
    ALARM = 'bi bi-alarm'
    ALEXA = 'bi bi-alexa'
    ALIGN_BOTTOM = 'bi bi-align-bottom'
    ALIGN_CENTER = 'bi bi-align-center'
    ALIGN_END = 'bi bi-align-end'
    ALIGN_MIDDLE = 'bi bi-align-middle'
    ALIGN_START = 'bi bi-align-start'
    ALIGN_TOP = 'bi bi-align-top'
    ALIPAY = 'bi bi-alipay'
    ALPHABET_UPPERCASE = 'bi bi-alphabet-uppercase'
    ALPHABET = 'bi bi-alphabet'
    ALT = 'bi bi-alt'
    AMAZON = 'bi bi-amazon'
    AMD = 'bi bi-amd'
    ANDROID = 'bi bi-android'
    ANDROID2 = 'bi bi-android2'
    APP_INDICATOR = 'bi bi-app-indicator'
    APP = 'bi bi-app'
    APPLE = 'bi bi-apple'
    ARCHIVE_FILL = 'bi bi-archive-fill'
    ARCHIVE = 'bi bi-archive'
    ARROW_90DEG_DOWN = 'bi bi-arrow-90deg-down'
    ARROW_90DEG_LEFT = 'bi bi-arrow-90deg-left'
    ARROW_90DEG_RIGHT = 'bi bi-arrow-90deg-right'
    ARROW_90DEG_UP = 'bi bi-arrow-90deg-up'
    ARROW_BAR_DOWN = 'bi bi-arrow-bar-down'
    ARROW_BAR_LEFT = 'bi bi-arrow-bar-left'
    ARROW_BAR_RIGHT = 'bi bi-arrow-bar-right'
    ARROW_BAR_UP = 'bi bi-arrow-bar-up'
    ARROW_CLOCKWISE = 'bi bi-arrow-clockwise'
    ARROW_COUNTERCLOCKWISE = 'bi bi-arrow-counterclockwise'
    ARROW_DOWN_CIRCLE_FILL = 'bi bi-arrow-down-circle-fill'
    ARROW_DOWN_CIRCLE = 'bi bi-arrow-down-circle'
    ARROW_DOWN_LEFT_CIRCLE_FILL = 'bi bi-arrow-down-left-circle-fill'
    ARROW_DOWN_LEFT_CIRCLE = 'bi bi-arrow-down-left-circle'
    ARROW_DOWN_LEFT_SQUARE_FILL = 'bi bi-arrow-down-left-square-fill'
    ARROW_DOWN_LEFT_SQUARE = 'bi bi-arrow-down-left-square'
    ARROW_DOWN_LEFT = 'bi bi-arrow-down-left'
    ARROW_DOWN_RIGHT_CIRCLE_FILL = 'bi bi-arrow-down-right-circle-fill'
    ARROW_DOWN_RIGHT_CIRCLE = 'bi bi-arrow-down-right-circle'
    ARROW_DOWN_RIGHT_SQUARE_FILL = 'bi bi-arrow-down-right-square-fill'
    ARROW_DOWN_RIGHT_SQUARE = 'bi bi-arrow-down-right-square'
    ARROW_DOWN_RIGHT = 'bi bi-arrow-down-right'
    ARROW_DOWN_SHORT = 'bi bi-arrow-down-short'
    ARROW_DOWN_SQUARE_FILL = 'bi bi-arrow-down-square-fill'
    ARROW_DOWN_SQUARE = 'bi bi-arrow-down-square'
    ARROW_DOWN_UP = 'bi bi-arrow-down-up'
    ARROW_DOWN = 'bi bi-arrow-down'
    ARROW_LEFT_CIRCLE_FILL = 'bi bi-arrow-left-circle-fill'
    ARROW_LEFT_CIRCLE = 'bi bi-arrow-left-circle'
    ARROW_LEFT_RIGHT = 'bi bi-arrow-left-right'
    ARROW_LEFT_SHORT = 'bi bi-arrow-left-short'
    ARROW_LEFT_SQUARE_FILL = 'bi bi-arrow-left-square-fill'
    ARROW_LEFT_SQUARE = 'bi bi-arrow-left-square'
    ARROW_LEFT = 'bi bi-arrow-left'
    ARROW_REPEAT = 'bi bi-arrow-repeat'
    ARROW_RETURN_LEFT = 'bi bi-arrow-return-left'
    ARROW_RETURN_RIGHT = 'bi bi-arrow-return-right'
    ARROW_RIGHT_CIRCLE_FILL = 'bi bi-arrow-right-circle-fill'
    ARROW_RIGHT_CIRCLE = 'bi bi-arrow-right-circle'
    ARROW_RIGHT_SHORT = 'bi bi-arrow-right-short'
    ARROW_RIGHT_SQUARE_FILL = 'bi bi-arrow-right-square-fill'
    ARROW_RIGHT_SQUARE = 'bi bi-arrow-right-square'
    ARROW_RIGHT = 'bi bi-arrow-right'
    ARROW_THROUGH_HEART_FILL = 'bi bi-arrow-through-heart-fill'
    ARROW_THROUGH_HEART = 'bi bi-arrow-through-heart'
    ARROW_UP_CIRCLE_FILL = 'bi bi-arrow-up-circle-fill'
    ARROW_UP_CIRCLE = 'bi bi-arrow-up-circle'
    ARROW_UP_LEFT_CIRCLE_FILL = 'bi bi-arrow-up-left-circle-fill'
    ARROW_UP_LEFT_CIRCLE = 'bi bi-arrow-up-left-circle'
    ARROW_UP_LEFT_SQUARE_FILL = 'bi bi-arrow-up-left-square-fill'
    ARROW_UP_LEFT_SQUARE = 'bi bi-arrow-up-left-square'
    ARROW_UP_LEFT = 'bi bi-arrow-up-left'
    ARROW_UP_RIGHT_CIRCLE_FILL = 'bi bi-arrow-up-right-circle-fill'
    ARROW_UP_RIGHT_CIRCLE = 'bi bi-arrow-up-right-circle'
    ARROW_UP_RIGHT_SQUARE_FILL = 'bi bi-arrow-up-right-square-fill'
    ARROW_UP_RIGHT_SQUARE = 'bi bi-arrow-up-right-square'
    ARROW_UP_RIGHT = 'bi bi-arrow-up-right'
    ARROW_UP_SHORT = 'bi bi-arrow-up-short'
    ARROW_UP_SQUARE_FILL = 'bi bi-arrow-up-square-fill'
    ARROW_UP_SQUARE = 'bi bi-arrow-up-square'
    ARROW_UP = 'bi bi-arrow-up'
    ARROWS_ANGLE_CONTRACT = 'bi bi-arrows-angle-contract'
    ARROWS_ANGLE_EXPAND = 'bi bi-arrows-angle-expand'
    ARROWS_COLLAPSE_VERTICAL = 'bi bi-arrows-collapse-vertical'
    ARROWS_COLLAPSE = 'bi bi-arrows-collapse'
    ARROWS_EXPAND_VERTICAL = 'bi bi-arrows-expand-vertical'
    ARROWS_EXPAND = 'bi bi-arrows-expand'
    ARROWS_FULLSCREEN = 'bi bi-arrows-fullscreen'
    ARROWS_MOVE = 'bi bi-arrows-move'
    ARROWS_VERTICAL = 'bi bi-arrows-vertical'
    ARROWS = 'bi bi-arrows'
    ASPECT_RATIO_FILL = 'bi bi-aspect-ratio-fill'
    ASPECT_RATIO = 'bi bi-aspect-ratio'
    ASTERISK = 'bi bi-asterisk'
    AT = 'bi bi-at'
    AWARD_FILL = 'bi bi-award-fill'
    AWARD = 'bi bi-award'
    BACK = 'bi bi-back'
    BACKPACK_FILL = 'bi bi-backpack-fill'
    BACKPACK = 'bi bi-backpack'
    BACKPACK2_FILL = 'bi bi-backpack2-fill'
    BACKPACK2 = 'bi bi-backpack2'
    BACKPACK3_FILL = 'bi bi-backpack3-fill'
    BACKPACK3 = 'bi bi-backpack3'
    BACKPACK4_FILL = 'bi bi-backpack4-fill'
    BACKPACK4 = 'bi bi-backpack4'
    BACKSPACE_FILL = 'bi bi-backspace-fill'
    BACKSPACE_REVERSE_FILL = 'bi bi-backspace-reverse-fill'
    BACKSPACE_REVERSE = 'bi bi-backspace-reverse'
    BACKSPACE = 'bi bi-backspace'
    BADGE_3D_FILL = 'bi bi-badge-3d-fill'
    BADGE_3D = 'bi bi-badge-3d'
    BADGE_4K_FILL = 'bi bi-badge-4k-fill'
    BADGE_4K = 'bi bi-badge-4k'
    BADGE_8K_FILL = 'bi bi-badge-8k-fill'
    BADGE_8K = 'bi bi-badge-8k'
    BADGE_AD_FILL = 'bi bi-badge-ad-fill'
    BADGE_AD = 'bi bi-badge-ad'
    BADGE_AR_FILL = 'bi bi-badge-ar-fill'
    BADGE_AR = 'bi bi-badge-ar'
    BADGE_CC_FILL = 'bi bi-badge-cc-fill'
    BADGE_CC = 'bi bi-badge-cc'
    BADGE_HD_FILL = 'bi bi-badge-hd-fill'
    BADGE_HD = 'bi bi-badge-hd'
    BADGE_SD_FILL = 'bi bi-badge-sd-fill'
    BADGE_SD = 'bi bi-badge-sd'
    BADGE_TM_FILL = 'bi bi-badge-tm-fill'
    BADGE_TM = 'bi bi-badge-tm'
    BADGE_VO_FILL = 'bi bi-badge-vo-fill'
    BADGE_VO = 'bi bi-badge-vo'
    BADGE_VR_FILL = 'bi bi-badge-vr-fill'
    BADGE_VR = 'bi bi-badge-vr'
    BADGE_WC_FILL = 'bi bi-badge-wc-fill'
    BADGE_WC = 'bi bi-badge-wc'
    BAG_CHECK_FILL = 'bi bi-bag-check-fill'
    BAG_CHECK = 'bi bi-bag-check'
    BAG_DASH_FILL = 'bi bi-bag-dash-fill'
    BAG_DASH = 'bi bi-bag-dash'
    BAG_FILL = 'bi bi-bag-fill'
    BAG_HEART_FILL = 'bi bi-bag-heart-fill'
    BAG_HEART = 'bi bi-bag-heart'
    BAG_PLUS_FILL = 'bi bi-bag-plus-fill'
    BAG_PLUS = 'bi bi-bag-plus'
    BAG_X_FILL = 'bi bi-bag-x-fill'
    BAG_X = 'bi bi-bag-x'
    BAG = 'bi bi-bag'
    BALLOON_FILL = 'bi bi-balloon-fill'
    BALLOON_HEART_FILL = 'bi bi-balloon-heart-fill'
    BALLOON_HEART = 'bi bi-balloon-heart'
    BALLOON = 'bi bi-balloon'
    BAN_FILL = 'bi bi-ban-fill'
    BAN = 'bi bi-ban'
    BANDAID_FILL = 'bi bi-bandaid-fill'
    BANDAID = 'bi bi-bandaid'
    BANK = 'bi bi-bank'
    BANK2 = 'bi bi-bank2'
    BAR_CHART_FILL = 'bi bi-bar-chart-fill'
    BAR_CHART_LINE_FILL = 'bi bi-bar-chart-line-fill'
    BAR_CHART_LINE = 'bi bi-bar-chart-line'
    BAR_CHART_STEPS = 'bi bi-bar-chart-steps'
    BAR_CHART = 'bi bi-bar-chart'
    BASKET_FILL = 'bi bi-basket-fill'
    BASKET = 'bi bi-basket'
    BASKET2_FILL = 'bi bi-basket2-fill'
    BASKET2 = 'bi bi-basket2'
    BASKET3_FILL = 'bi bi-basket3-fill'
    BASKET3 = 'bi bi-basket3'
    BATTERY_CHARGING = 'bi bi-battery-charging'
    BATTERY_FULL = 'bi bi-battery-full'
    BATTERY_HALF = 'bi bi-battery-half'
    BATTERY = 'bi bi-battery'
    BEHANCE = 'bi bi-behance'
    BELL_FILL = 'bi bi-bell-fill'
    BELL_SLASH_FILL = 'bi bi-bell-slash-fill'
    BELL_SLASH = 'bi bi-bell-slash'
    BELL = 'bi bi-bell'
    BEZIER = 'bi bi-bezier'
    BEZIER2 = 'bi bi-bezier2'
    BICYCLE = 'bi bi-bicycle'
    BING = 'bi bi-bing'
    BINOCULARS_FILL = 'bi bi-binoculars-fill'
    BINOCULARS = 'bi bi-binoculars'
    BLOCKQUOTE_LEFT = 'bi bi-blockquote-left'
    BLOCKQUOTE_RIGHT = 'bi bi-blockquote-right'
    BLUETOOTH = 'bi bi-bluetooth'
    BODY_TEXT = 'bi bi-body-text'
    BOOK_FILL = 'bi bi-book-fill'
    BOOK_HALF = 'bi bi-book-half'
    BOOK = 'bi bi-book'
    BOOKMARK_CHECK_FILL = 'bi bi-bookmark-check-fill'
    BOOKMARK_CHECK = 'bi bi-bookmark-check'
    BOOKMARK_DASH_FILL = 'bi bi-bookmark-dash-fill'
    BOOKMARK_DASH = 'bi bi-bookmark-dash'
    BOOKMARK_FILL = 'bi bi-bookmark-fill'
    BOOKMARK_HEART_FILL = 'bi bi-bookmark-heart-fill'
    BOOKMARK_HEART = 'bi bi-bookmark-heart'
    BOOKMARK_PLUS_FILL = 'bi bi-bookmark-plus-fill'
    BOOKMARK_PLUS = 'bi bi-bookmark-plus'
    BOOKMARK_STAR_FILL = 'bi bi-bookmark-star-fill'
    BOOKMARK_STAR = 'bi bi-bookmark-star'
    BOOKMARK_X_FILL = 'bi bi-bookmark-x-fill'
    BOOKMARK_X = 'bi bi-bookmark-x'
    BOOKMARK = 'bi bi-bookmark'
    BOOKMARKS_FILL = 'bi bi-bookmarks-fill'
    BOOKMARKS = 'bi bi-bookmarks'
    BOOKSHELF = 'bi bi-bookshelf'
    BOOMBOX_FILL = 'bi bi-boombox-fill'
    BOOMBOX = 'bi bi-boombox'
    BOOTSTRAP_FILL = 'bi bi-bootstrap-fill'
    BOOTSTRAP_ICONS = 'bi bi-bootstrap-icons'
    BOOTSTRAP_REBOOT = 'bi bi-bootstrap-reboot'
    BOOTSTRAP = 'bi bi-bootstrap'
    BORDER_ALL = 'bi bi-border-all'
    BORDER_BOTTOM = 'bi bi-border-bottom'
    BORDER_CENTER = 'bi bi-border-center'
    BORDER_INNER = 'bi bi-border-inner'
    BORDER_LEFT = 'bi bi-border-left'
    BORDER_MIDDLE = 'bi bi-border-middle'
    BORDER_OUTER = 'bi bi-border-outer'
    BORDER_RIGHT = 'bi bi-border-right'
    BORDER_STYLE = 'bi bi-border-style'
    BORDER_TOP = 'bi bi-border-top'
    BORDER_WIDTH = 'bi bi-border-width'
    BORDER = 'bi bi-border'
    BOUNDING_BOX_CIRCLES = 'bi bi-bounding-box-circles'
    BOUNDING_BOX = 'bi bi-bounding-box'
    BOX_ARROW_DOWN_LEFT = 'bi bi-box-arrow-down-left'
    BOX_ARROW_DOWN_RIGHT = 'bi bi-box-arrow-down-right'
    BOX_ARROW_DOWN = 'bi bi-box-arrow-down'
    BOX_ARROW_IN_DOWN_LEFT = 'bi bi-box-arrow-in-down-left'
    BOX_ARROW_IN_DOWN_RIGHT = 'bi bi-box-arrow-in-down-right'
    BOX_ARROW_IN_DOWN = 'bi bi-box-arrow-in-down'
    BOX_ARROW_IN_LEFT = 'bi bi-box-arrow-in-left'
    BOX_ARROW_IN_RIGHT = 'bi bi-box-arrow-in-right'
    BOX_ARROW_IN_UP_LEFT = 'bi bi-box-arrow-in-up-left'
    BOX_ARROW_IN_UP_RIGHT = 'bi bi-box-arrow-in-up-right'
    BOX_ARROW_IN_UP = 'bi bi-box-arrow-in-up'
    BOX_ARROW_LEFT = 'bi bi-box-arrow-left'
    BOX_ARROW_RIGHT = 'bi bi-box-arrow-right'
    BOX_ARROW_UP_LEFT = 'bi bi-box-arrow-up-left'
    BOX_ARROW_UP_RIGHT = 'bi bi-box-arrow-up-right'
    BOX_ARROW_UP = 'bi bi-box-arrow-up'
    BOX_FILL = 'bi bi-box-fill'
    BOX_SEAM_FILL = 'bi bi-box-seam-fill'
    BOX_SEAM = 'bi bi-box-seam'
    BOX = 'bi bi-box'
    BOX2_FILL = 'bi bi-box2-fill'
    BOX2_HEART_FILL = 'bi bi-box2-heart-fill'
    BOX2_HEART = 'bi bi-box2-heart'
    BOX2 = 'bi bi-box2'
    BOXES = 'bi bi-boxes'
    BRACES_ASTERISK = 'bi bi-braces-asterisk'
    BRACES = 'bi bi-braces'
    BRICKS = 'bi bi-bricks'
    BRIEFCASE_FILL = 'bi bi-briefcase-fill'
    BRIEFCASE = 'bi bi-briefcase'
    BRIGHTNESS_ALT_HIGH_FILL = 'bi bi-brightness-alt-high-fill'
    BRIGHTNESS_ALT_HIGH = 'bi bi-brightness-alt-high'
    BRIGHTNESS_ALT_LOW_FILL = 'bi bi-brightness-alt-low-fill'
    BRIGHTNESS_ALT_LOW = 'bi bi-brightness-alt-low'
    BRIGHTNESS_HIGH_FILL = 'bi bi-brightness-high-fill'
    BRIGHTNESS_HIGH = 'bi bi-brightness-high'
    BRIGHTNESS_LOW_FILL = 'bi bi-brightness-low-fill'
    BRIGHTNESS_LOW = 'bi bi-brightness-low'
    BRILLIANCE = 'bi bi-brilliance'
    BROADCAST_PIN = 'bi bi-broadcast-pin'
    BROADCAST = 'bi bi-broadcast'
    BROWSER_CHROME = 'bi bi-browser-chrome'
    BROWSER_EDGE = 'bi bi-browser-edge'
    BROWSER_FIREFOX = 'bi bi-browser-firefox'
    BROWSER_SAFARI = 'bi bi-browser-safari'
    BRUSH_FILL = 'bi bi-brush-fill'
    BRUSH = 'bi bi-brush'
    BUCKET_FILL = 'bi bi-bucket-fill'
    BUCKET = 'bi bi-bucket'
    BUG_FILL = 'bi bi-bug-fill'
    BUG = 'bi bi-bug'
    BUILDING_ADD = 'bi bi-building-add'
    BUILDING_CHECK = 'bi bi-building-check'
    BUILDING_DASH = 'bi bi-building-dash'
    BUILDING_DOWN = 'bi bi-building-down'
    BUILDING_EXCLAMATION = 'bi bi-building-exclamation'
    BUILDING_FILL_ADD = 'bi bi-building-fill-add'
    BUILDING_FILL_CHECK = 'bi bi-building-fill-check'
    BUILDING_FILL_DASH = 'bi bi-building-fill-dash'
    BUILDING_FILL_DOWN = 'bi bi-building-fill-down'
    BUILDING_FILL_EXCLAMATION = 'bi bi-building-fill-exclamation'
    BUILDING_FILL_GEAR = 'bi bi-building-fill-gear'
    BUILDING_FILL_LOCK = 'bi bi-building-fill-lock'
    BUILDING_FILL_SLASH = 'bi bi-building-fill-slash'
    BUILDING_FILL_UP = 'bi bi-building-fill-up'
    BUILDING_FILL_X = 'bi bi-building-fill-x'
    BUILDING_FILL = 'bi bi-building-fill'
    BUILDING_GEAR = 'bi bi-building-gear'
    BUILDING_LOCK = 'bi bi-building-lock'
    BUILDING_SLASH = 'bi bi-building-slash'
    BUILDING_UP = 'bi bi-building-up'
    BUILDING_X = 'bi bi-building-x'
    BUILDING = 'bi bi-building'
    BUILDINGS_FILL = 'bi bi-buildings-fill'
    BUILDINGS = 'bi bi-buildings'
    BULLSEYE = 'bi bi-bullseye'
    BUS_FRONT_FILL = 'bi bi-bus-front-fill'
    BUS_FRONT = 'bi bi-bus-front'
    C_CIRCLE_FILL = 'bi bi-c-circle-fill'
    C_CIRCLE = 'bi bi-c-circle'
    C_SQUARE_FILL = 'bi bi-c-square-fill'
    C_SQUARE = 'bi bi-c-square'
    CAKE_FILL = 'bi bi-cake-fill'
    CAKE = 'bi bi-cake'
    CAKE2_FILL = 'bi bi-cake2-fill'
    CAKE2 = 'bi bi-cake2'
    CALCULATOR_FILL = 'bi bi-calculator-fill'
    CALCULATOR = 'bi bi-calculator'
    CALENDAR_CHECK_FILL = 'bi bi-calendar-check-fill'
    CALENDAR_CHECK = 'bi bi-calendar-check'
    CALENDAR_DATE_FILL = 'bi bi-calendar-date-fill'
    CALENDAR_DATE = 'bi bi-calendar-date'
    CALENDAR_DAY_FILL = 'bi bi-calendar-day-fill'
    CALENDAR_DAY = 'bi bi-calendar-day'
    CALENDAR_EVENT_FILL = 'bi bi-calendar-event-fill'
    CALENDAR_EVENT = 'bi bi-calendar-event'
    CALENDAR_FILL = 'bi bi-calendar-fill'
    CALENDAR_HEART_FILL = 'bi bi-calendar-heart-fill'
    CALENDAR_HEART = 'bi bi-calendar-heart'
    CALENDAR_MINUS_FILL = 'bi bi-calendar-minus-fill'
    CALENDAR_MINUS = 'bi bi-calendar-minus'
    CALENDAR_MONTH_FILL = 'bi bi-calendar-month-fill'
    CALENDAR_MONTH = 'bi bi-calendar-month'
    CALENDAR_PLUS_FILL = 'bi bi-calendar-plus-fill'
    CALENDAR_PLUS = 'bi bi-calendar-plus'
    CALENDAR_RANGE_FILL = 'bi bi-calendar-range-fill'
    CALENDAR_RANGE = 'bi bi-calendar-range'
    CALENDAR_WEEK_FILL = 'bi bi-calendar-week-fill'
    CALENDAR_WEEK = 'bi bi-calendar-week'
    CALENDAR_X_FILL = 'bi bi-calendar-x-fill'
    CALENDAR_X = 'bi bi-calendar-x'
    CALENDAR = 'bi bi-calendar'
    CALENDAR2_CHECK_FILL = 'bi bi-calendar2-check-fill'
    CALENDAR2_CHECK = 'bi bi-calendar2-check'
    CALENDAR2_DATE_FILL = 'bi bi-calendar2-date-fill'
    CALENDAR2_DATE = 'bi bi-calendar2-date'
    CALENDAR2_DAY_FILL = 'bi bi-calendar2-day-fill'
    CALENDAR2_DAY = 'bi bi-calendar2-day'
    CALENDAR2_EVENT_FILL = 'bi bi-calendar2-event-fill'
    CALENDAR2_EVENT = 'bi bi-calendar2-event'
    CALENDAR2_FILL = 'bi bi-calendar2-fill'
    CALENDAR2_HEART_FILL = 'bi bi-calendar2-heart-fill'
    CALENDAR2_HEART = 'bi bi-calendar2-heart'
    CALENDAR2_MINUS_FILL = 'bi bi-calendar2-minus-fill'
    CALENDAR2_MINUS = 'bi bi-calendar2-minus'
    CALENDAR2_MONTH_FILL = 'bi bi-calendar2-month-fill'
    CALENDAR2_MONTH = 'bi bi-calendar2-month'
    CALENDAR2_PLUS_FILL = 'bi bi-calendar2-plus-fill'
    CALENDAR2_PLUS = 'bi bi-calendar2-plus'
    CALENDAR2_RANGE_FILL = 'bi bi-calendar2-range-fill'
    CALENDAR2_RANGE = 'bi bi-calendar2-range'
    CALENDAR2_WEEK_FILL = 'bi bi-calendar2-week-fill'
    CALENDAR2_WEEK = 'bi bi-calendar2-week'
    CALENDAR2_X_FILL = 'bi bi-calendar2-x-fill'
    CALENDAR2_X = 'bi bi-calendar2-x'
    CALENDAR2 = 'bi bi-calendar2'
    CALENDAR3_EVENT_FILL = 'bi bi-calendar3-event-fill'
    CALENDAR3_EVENT = 'bi bi-calendar3-event'
    CALENDAR3_FILL = 'bi bi-calendar3-fill'
    CALENDAR3_RANGE_FILL = 'bi bi-calendar3-range-fill'
    CALENDAR3_RANGE = 'bi bi-calendar3-range'
    CALENDAR3_WEEK_FILL = 'bi bi-calendar3-week-fill'
    CALENDAR3_WEEK = 'bi bi-calendar3-week'
    CALENDAR3 = 'bi bi-calendar3'
    CALENDAR4_EVENT = 'bi bi-calendar4-event'
    CALENDAR4_RANGE = 'bi bi-calendar4-range'
    CALENDAR4_WEEK = 'bi bi-calendar4-week'
    CALENDAR4 = 'bi bi-calendar4'
    CAMERA_FILL = 'bi bi-camera-fill'
    CAMERA_REELS_FILL = 'bi bi-camera-reels-fill'
    CAMERA_REELS = 'bi bi-camera-reels'
    CAMERA_VIDEO_FILL = 'bi bi-camera-video-fill'
    CAMERA_VIDEO_OFF_FILL = 'bi bi-camera-video-off-fill'
    CAMERA_VIDEO_OFF = 'bi bi-camera-video-off'
    CAMERA_VIDEO = 'bi bi-camera-video'
    CAMERA = 'bi bi-camera'
    CAMERA2 = 'bi bi-camera2'
    CAPSLOCK_FILL = 'bi bi-capslock-fill'
    CAPSLOCK = 'bi bi-capslock'
    CAPSULE_PILL = 'bi bi-capsule-pill'
    CAPSULE = 'bi bi-capsule'
    CAR_FRONT_FILL = 'bi bi-car-front-fill'
    CAR_FRONT = 'bi bi-car-front'
    CARD_CHECKLIST = 'bi bi-card-checklist'
    CARD_HEADING = 'bi bi-card-heading'
    CARD_IMAGE = 'bi bi-card-image'
    CARD_LIST = 'bi bi-card-list'
    CARD_TEXT = 'bi bi-card-text'
    CARET_DOWN_FILL = 'bi bi-caret-down-fill'
    CARET_DOWN_SQUARE_FILL = 'bi bi-caret-down-square-fill'
    CARET_DOWN_SQUARE = 'bi bi-caret-down-square'
    CARET_DOWN = 'bi bi-caret-down'
    CARET_LEFT_FILL = 'bi bi-caret-left-fill'
    CARET_LEFT_SQUARE_FILL = 'bi bi-caret-left-square-fill'
    CARET_LEFT_SQUARE = 'bi bi-caret-left-square'
    CARET_LEFT = 'bi bi-caret-left'
    CARET_RIGHT_FILL = 'bi bi-caret-right-fill'
    CARET_RIGHT_SQUARE_FILL = 'bi bi-caret-right-square-fill'
    CARET_RIGHT_SQUARE = 'bi bi-caret-right-square'
    CARET_RIGHT = 'bi bi-caret-right'
    CARET_UP_FILL = 'bi bi-caret-up-fill'
    CARET_UP_SQUARE_FILL = 'bi bi-caret-up-square-fill'
    CARET_UP_SQUARE = 'bi bi-caret-up-square'
    CARET_UP = 'bi bi-caret-up'
    CART_CHECK_FILL = 'bi bi-cart-check-fill'
    CART_CHECK = 'bi bi-cart-check'
    CART_DASH_FILL = 'bi bi-cart-dash-fill'
    CART_DASH = 'bi bi-cart-dash'
    CART_FILL = 'bi bi-cart-fill'
    CART_PLUS_FILL = 'bi bi-cart-plus-fill'
    CART_PLUS = 'bi bi-cart-plus'
    CART_X_FILL = 'bi bi-cart-x-fill'
    CART_X = 'bi bi-cart-x'
    CART = 'bi bi-cart'
    CART2 = 'bi bi-cart2'
    CART3 = 'bi bi-cart3'
    CART4 = 'bi bi-cart4'
    CASH_COIN = 'bi bi-cash-coin'
    CASH_STACK = 'bi bi-cash-stack'
    CASH = 'bi bi-cash'
    CASSETTE_FILL = 'bi bi-cassette-fill'
    CASSETTE = 'bi bi-cassette'
    CAST = 'bi bi-cast'
    CC_CIRCLE_FILL = 'bi bi-cc-circle-fill'
    CC_CIRCLE = 'bi bi-cc-circle'
    CC_SQUARE_FILL = 'bi bi-cc-square-fill'
    CC_SQUARE = 'bi bi-cc-square'
    CHAT_DOTS_FILL = 'bi bi-chat-dots-fill'
    CHAT_DOTS = 'bi bi-chat-dots'
    CHAT_FILL = 'bi bi-chat-fill'
    CHAT_HEART_FILL = 'bi bi-chat-heart-fill'
    CHAT_HEART = 'bi bi-chat-heart'
    CHAT_LEFT_DOTS_FILL = 'bi bi-chat-left-dots-fill'
    CHAT_LEFT_DOTS = 'bi bi-chat-left-dots'
    CHAT_LEFT_FILL = 'bi bi-chat-left-fill'
    CHAT_LEFT_HEART_FILL = 'bi bi-chat-left-heart-fill'
    CHAT_LEFT_HEART = 'bi bi-chat-left-heart'
    CHAT_LEFT_QUOTE_FILL = 'bi bi-chat-left-quote-fill'
    CHAT_LEFT_QUOTE = 'bi bi-chat-left-quote'
    CHAT_LEFT_TEXT_FILL = 'bi bi-chat-left-text-fill'
    CHAT_LEFT_TEXT = 'bi bi-chat-left-text'
    CHAT_LEFT = 'bi bi-chat-left'
    CHAT_QUOTE_FILL = 'bi bi-chat-quote-fill'
    CHAT_QUOTE = 'bi bi-chat-quote'
    CHAT_RIGHT_DOTS_FILL = 'bi bi-chat-right-dots-fill'
    CHAT_RIGHT_DOTS = 'bi bi-chat-right-dots'
    CHAT_RIGHT_FILL = 'bi bi-chat-right-fill'
    CHAT_RIGHT_HEART_FILL = 'bi bi-chat-right-heart-fill'
    CHAT_RIGHT_HEART = 'bi bi-chat-right-heart'
    CHAT_RIGHT_QUOTE_FILL = 'bi bi-chat-right-quote-fill'
    CHAT_RIGHT_QUOTE = 'bi bi-chat-right-quote'
    CHAT_RIGHT_TEXT_FILL = 'bi bi-chat-right-text-fill'
    CHAT_RIGHT_TEXT = 'bi bi-chat-right-text'
    CHAT_RIGHT = 'bi bi-chat-right'
    CHAT_SQUARE_DOTS_FILL = 'bi bi-chat-square-dots-fill'
    CHAT_SQUARE_DOTS = 'bi bi-chat-square-dots'
    CHAT_SQUARE_FILL = 'bi bi-chat-square-fill'
    CHAT_SQUARE_HEART_FILL = 'bi bi-chat-square-heart-fill'
    CHAT_SQUARE_HEART = 'bi bi-chat-square-heart'
    CHAT_SQUARE_QUOTE_FILL = 'bi bi-chat-square-quote-fill'
    CHAT_SQUARE_QUOTE = 'bi bi-chat-square-quote'
    CHAT_SQUARE_TEXT_FILL = 'bi bi-chat-square-text-fill'
    CHAT_SQUARE_TEXT = 'bi bi-chat-square-text'
    CHAT_SQUARE = 'bi bi-chat-square'
    CHAT_TEXT_FILL = 'bi bi-chat-text-fill'
    CHAT_TEXT = 'bi bi-chat-text'
    CHAT = 'bi bi-chat'
    CHECK_ALL = 'bi bi-check-all'
    CHECK_CIRCLE_FILL = 'bi bi-check-circle-fill'
    CHECK_CIRCLE = 'bi bi-check-circle'
    CHECK_LG = 'bi bi-check-lg'
    CHECK_SQUARE_FILL = 'bi bi-check-square-fill'
    CHECK_SQUARE = 'bi bi-check-square'
    CHECK = 'bi bi-check'
    CHECK2_ALL = 'bi bi-check2-all'
    CHECK2_CIRCLE = 'bi bi-check2-circle'
    CHECK2_SQUARE = 'bi bi-check2-square'
    CHECK2 = 'bi bi-check2'
    CHEVRON_BAR_CONTRACT = 'bi bi-chevron-bar-contract'
    CHEVRON_BAR_DOWN = 'bi bi-chevron-bar-down'
    CHEVRON_BAR_EXPAND = 'bi bi-chevron-bar-expand'
    CHEVRON_BAR_LEFT = 'bi bi-chevron-bar-left'
    CHEVRON_BAR_RIGHT = 'bi bi-chevron-bar-right'
    CHEVRON_BAR_UP = 'bi bi-chevron-bar-up'
    CHEVRON_COMPACT_DOWN = 'bi bi-chevron-compact-down'
    CHEVRON_COMPACT_LEFT = 'bi bi-chevron-compact-left'
    CHEVRON_COMPACT_RIGHT = 'bi bi-chevron-compact-right'
    CHEVRON_COMPACT_UP = 'bi bi-chevron-compact-up'
    CHEVRON_CONTRACT = 'bi bi-chevron-contract'
    CHEVRON_DOUBLE_DOWN = 'bi bi-chevron-double-down'
    CHEVRON_DOUBLE_LEFT = 'bi bi-chevron-double-left'
    CHEVRON_DOUBLE_RIGHT = 'bi bi-chevron-double-right'
    CHEVRON_DOUBLE_UP = 'bi bi-chevron-double-up'
    CHEVRON_DOWN = 'bi bi-chevron-down'
    CHEVRON_EXPAND = 'bi bi-chevron-expand'
    CHEVRON_LEFT = 'bi bi-chevron-left'
    CHEVRON_RIGHT = 'bi bi-chevron-right'
    CHEVRON_UP = 'bi bi-chevron-up'
    CIRCLE_FILL = 'bi bi-circle-fill'
    CIRCLE_HALF = 'bi bi-circle-half'
    CIRCLE_SQUARE = 'bi bi-circle-square'
    CIRCLE = 'bi bi-circle'
    CLIPBOARD_CHECK_FILL = 'bi bi-clipboard-check-fill'
    CLIPBOARD_CHECK = 'bi bi-clipboard-check'
    CLIPBOARD_DATA_FILL = 'bi bi-clipboard-data-fill'
    CLIPBOARD_DATA = 'bi bi-clipboard-data'
    CLIPBOARD_FILL = 'bi bi-clipboard-fill'
    CLIPBOARD_HEART_FILL = 'bi bi-clipboard-heart-fill'
    CLIPBOARD_HEART = 'bi bi-clipboard-heart'
    CLIPBOARD_MINUS_FILL = 'bi bi-clipboard-minus-fill'
    CLIPBOARD_MINUS = 'bi bi-clipboard-minus'
    CLIPBOARD_PLUS_FILL = 'bi bi-clipboard-plus-fill'
    CLIPBOARD_PLUS = 'bi bi-clipboard-plus'
    CLIPBOARD_PULSE = 'bi bi-clipboard-pulse'
    CLIPBOARD_X_FILL = 'bi bi-clipboard-x-fill'
    CLIPBOARD_X = 'bi bi-clipboard-x'
    CLIPBOARD = 'bi bi-clipboard'
    CLIPBOARD2_CHECK_FILL = 'bi bi-clipboard2-check-fill'
    CLIPBOARD2_CHECK = 'bi bi-clipboard2-check'
    CLIPBOARD2_DATA_FILL = 'bi bi-clipboard2-data-fill'
    CLIPBOARD2_DATA = 'bi bi-clipboard2-data'
    CLIPBOARD2_FILL = 'bi bi-clipboard2-fill'
    CLIPBOARD2_HEART_FILL = 'bi bi-clipboard2-heart-fill'
    CLIPBOARD2_HEART = 'bi bi-clipboard2-heart'
    CLIPBOARD2_MINUS_FILL = 'bi bi-clipboard2-minus-fill'
    CLIPBOARD2_MINUS = 'bi bi-clipboard2-minus'
    CLIPBOARD2_PLUS_FILL = 'bi bi-clipboard2-plus-fill'
    CLIPBOARD2_PLUS = 'bi bi-clipboard2-plus'
    CLIPBOARD2_PULSE_FILL = 'bi bi-clipboard2-pulse-fill'
    CLIPBOARD2_PULSE = 'bi bi-clipboard2-pulse'
    CLIPBOARD2_X_FILL = 'bi bi-clipboard2-x-fill'
    CLIPBOARD2_X = 'bi bi-clipboard2-x'
    CLIPBOARD2 = 'bi bi-clipboard2'
    CLOCK_FILL = 'bi bi-clock-fill'
    CLOCK_HISTORY = 'bi bi-clock-history'
    CLOCK = 'bi bi-clock'
    CLOUD_ARROW_DOWN_FILL = 'bi bi-cloud-arrow-down-fill'
    CLOUD_ARROW_DOWN = 'bi bi-cloud-arrow-down'
    CLOUD_ARROW_UP_FILL = 'bi bi-cloud-arrow-up-fill'
    CLOUD_ARROW_UP = 'bi bi-cloud-arrow-up'
    CLOUD_CHECK_FILL = 'bi bi-cloud-check-fill'
    CLOUD_CHECK = 'bi bi-cloud-check'
    CLOUD_DOWNLOAD_FILL = 'bi bi-cloud-download-fill'
    CLOUD_DOWNLOAD = 'bi bi-cloud-download'
    CLOUD_DRIZZLE_FILL = 'bi bi-cloud-drizzle-fill'
    CLOUD_DRIZZLE = 'bi bi-cloud-drizzle'
    CLOUD_FILL = 'bi bi-cloud-fill'
    CLOUD_FOG_FILL = 'bi bi-cloud-fog-fill'
    CLOUD_FOG = 'bi bi-cloud-fog'
    CLOUD_FOG2_FILL = 'bi bi-cloud-fog2-fill'
    CLOUD_FOG2 = 'bi bi-cloud-fog2'
    CLOUD_HAIL_FILL = 'bi bi-cloud-hail-fill'
    CLOUD_HAIL = 'bi bi-cloud-hail'
    CLOUD_HAZE_FILL = 'bi bi-cloud-haze-fill'
    CLOUD_HAZE = 'bi bi-cloud-haze'
    CLOUD_HAZE2_FILL = 'bi bi-cloud-haze2-fill'
    CLOUD_HAZE2 = 'bi bi-cloud-haze2'
    CLOUD_LIGHTNING_FILL = 'bi bi-cloud-lightning-fill'
    CLOUD_LIGHTNING_RAIN_FILL = 'bi bi-cloud-lightning-rain-fill'
    CLOUD_LIGHTNING_RAIN = 'bi bi-cloud-lightning-rain'
    CLOUD_LIGHTNING = 'bi bi-cloud-lightning'
    CLOUD_MINUS_FILL = 'bi bi-cloud-minus-fill'
    CLOUD_MINUS = 'bi bi-cloud-minus'
    CLOUD_MOON_FILL = 'bi bi-cloud-moon-fill'
    CLOUD_MOON = 'bi bi-cloud-moon'
    CLOUD_PLUS_FILL = 'bi bi-cloud-plus-fill'
    CLOUD_PLUS = 'bi bi-cloud-plus'
    CLOUD_RAIN_FILL = 'bi bi-cloud-rain-fill'
    CLOUD_RAIN_HEAVY_FILL = 'bi bi-cloud-rain-heavy-fill'
    CLOUD_RAIN_HEAVY = 'bi bi-cloud-rain-heavy'
    CLOUD_RAIN = 'bi bi-cloud-rain'
    CLOUD_SLASH_FILL = 'bi bi-cloud-slash-fill'
    CLOUD_SLASH = 'bi bi-cloud-slash'
    CLOUD_SLEET_FILL = 'bi bi-cloud-sleet-fill'
    CLOUD_SLEET = 'bi bi-cloud-sleet'
    CLOUD_SNOW_FILL = 'bi bi-cloud-snow-fill'
    CLOUD_SNOW = 'bi bi-cloud-snow'
    CLOUD_SUN_FILL = 'bi bi-cloud-sun-fill'
    CLOUD_SUN = 'bi bi-cloud-sun'
    CLOUD_UPLOAD_FILL = 'bi bi-cloud-upload-fill'
    CLOUD_UPLOAD = 'bi bi-cloud-upload'
    CLOUD = 'bi bi-cloud'
    CLOUDS_FILL = 'bi bi-clouds-fill'
    CLOUDS = 'bi bi-clouds'
    CLOUDY_FILL = 'bi bi-cloudy-fill'
    CLOUDY = 'bi bi-cloudy'
    CODE_SLASH = 'bi bi-code-slash'
    CODE_SQUARE = 'bi bi-code-square'
    CODE = 'bi bi-code'
    COIN = 'bi bi-coin'
    COLLECTION_FILL = 'bi bi-collection-fill'
    COLLECTION_PLAY_FILL = 'bi bi-collection-play-fill'
    COLLECTION_PLAY = 'bi bi-collection-play'
    COLLECTION = 'bi bi-collection'
    COLUMNS_GAP = 'bi bi-columns-gap'
    COLUMNS = 'bi bi-columns'
    COMMAND = 'bi bi-command'
    COMPASS_FILL = 'bi bi-compass-fill'
    COMPASS = 'bi bi-compass'
    CONE_STRIPED = 'bi bi-cone-striped'
    CONE = 'bi bi-cone'
    CONTROLLER = 'bi bi-controller'
    COOKIE = 'bi bi-cookie'
    COPY = 'bi bi-copy'
    CPU_FILL = 'bi bi-cpu-fill'
    CPU = 'bi bi-cpu'
    CREDIT_CARD_2_BACK_FILL = 'bi bi-credit-card-2-back-fill'
    CREDIT_CARD_2_BACK = 'bi bi-credit-card-2-back'
    CREDIT_CARD_2_FRONT_FILL = 'bi bi-credit-card-2-front-fill'
    CREDIT_CARD_2_FRONT = 'bi bi-credit-card-2-front'
    CREDIT_CARD_FILL = 'bi bi-credit-card-fill'
    CREDIT_CARD = 'bi bi-credit-card'
    CROP = 'bi bi-crop'
    CROSSHAIR = 'bi bi-crosshair'
    CROSSHAIR2 = 'bi bi-crosshair2'
    CUP_FILL = 'bi bi-cup-fill'
    CUP_HOT_FILL = 'bi bi-cup-hot-fill'
    CUP_HOT = 'bi bi-cup-hot'
    CUP_STRAW = 'bi bi-cup-straw'
    CUP = 'bi bi-cup'
    CURRENCY_BITCOIN = 'bi bi-currency-bitcoin'
    CURRENCY_DOLLAR = 'bi bi-currency-dollar'
    CURRENCY_EURO = 'bi bi-currency-euro'
    CURRENCY_EXCHANGE = 'bi bi-currency-exchange'
    CURRENCY_POUND = 'bi bi-currency-pound'
    CURRENCY_RUPEE = 'bi bi-currency-rupee'
    CURRENCY_YEN = 'bi bi-currency-yen'
    CURSOR_FILL = 'bi bi-cursor-fill'
    CURSOR_TEXT = 'bi bi-cursor-text'
    CURSOR = 'bi bi-cursor'
    DASH_CIRCLE_DOTTED = 'bi bi-dash-circle-dotted'
    DASH_CIRCLE_FILL = 'bi bi-dash-circle-fill'
    DASH_CIRCLE = 'bi bi-dash-circle'
    DASH_LG = 'bi bi-dash-lg'
    DASH_SQUARE_DOTTED = 'bi bi-dash-square-dotted'
    DASH_SQUARE_FILL = 'bi bi-dash-square-fill'
    DASH_SQUARE = 'bi bi-dash-square'
    DASH = 'bi bi-dash'
    DATABASE_ADD = 'bi bi-database-add'
    DATABASE_CHECK = 'bi bi-database-check'
    DATABASE_DASH = 'bi bi-database-dash'
    DATABASE_DOWN = 'bi bi-database-down'
    DATABASE_EXCLAMATION = 'bi bi-database-exclamation'
    DATABASE_FILL_ADD = 'bi bi-database-fill-add'
    DATABASE_FILL_CHECK = 'bi bi-database-fill-check'
    DATABASE_FILL_DASH = 'bi bi-database-fill-dash'
    DATABASE_FILL_DOWN = 'bi bi-database-fill-down'
    DATABASE_FILL_EXCLAMATION = 'bi bi-database-fill-exclamation'
    DATABASE_FILL_GEAR = 'bi bi-database-fill-gear'
    DATABASE_FILL_LOCK = 'bi bi-database-fill-lock'
    DATABASE_FILL_SLASH = 'bi bi-database-fill-slash'
    DATABASE_FILL_UP = 'bi bi-database-fill-up'
    DATABASE_FILL_X = 'bi bi-database-fill-x'
    DATABASE_FILL = 'bi bi-database-fill'
    DATABASE_GEAR = 'bi bi-database-gear'
    DATABASE_LOCK = 'bi bi-database-lock'
    DATABASE_SLASH = 'bi bi-database-slash'
    DATABASE_UP = 'bi bi-database-up'
    DATABASE_X = 'bi bi-database-x'
    DATABASE = 'bi bi-database'
    DEVICE_HDD_FILL = 'bi bi-device-hdd-fill'
    DEVICE_HDD = 'bi bi-device-hdd'
    DEVICE_SSD_FILL = 'bi bi-device-ssd-fill'
    DEVICE_SSD = 'bi bi-device-ssd'
    DIAGRAM_2_FILL = 'bi bi-diagram-2-fill'
    DIAGRAM_2 = 'bi bi-diagram-2'
    DIAGRAM_3_FILL = 'bi bi-diagram-3-fill'
    DIAGRAM_3 = 'bi bi-diagram-3'
    DIAMOND_FILL = 'bi bi-diamond-fill'
    DIAMOND_HALF = 'bi bi-diamond-half'
    DIAMOND = 'bi bi-diamond'
    DICE_1_FILL = 'bi bi-dice-1-fill'
    DICE_1 = 'bi bi-dice-1'
    DICE_2_FILL = 'bi bi-dice-2-fill'
    DICE_2 = 'bi bi-dice-2'
    DICE_3_FILL = 'bi bi-dice-3-fill'
    DICE_3 = 'bi bi-dice-3'
    DICE_4_FILL = 'bi bi-dice-4-fill'
    DICE_4 = 'bi bi-dice-4'
    DICE_5_FILL = 'bi bi-dice-5-fill'
    DICE_5 = 'bi bi-dice-5'
    DICE_6_FILL = 'bi bi-dice-6-fill'
    DICE_6 = 'bi bi-dice-6'
    DISC_FILL = 'bi bi-disc-fill'
    DISC = 'bi bi-disc'
    DISCORD = 'bi bi-discord'
    DISPLAY_FILL = 'bi bi-display-fill'
    DISPLAY = 'bi bi-display'
    DISPLAYPORT_FILL = 'bi bi-displayport-fill'
    DISPLAYPORT = 'bi bi-displayport'
    DISTRIBUTE_HORIZONTAL = 'bi bi-distribute-horizontal'
    DISTRIBUTE_VERTICAL = 'bi bi-distribute-vertical'
    DOOR_CLOSED_FILL = 'bi bi-door-closed-fill'
    DOOR_CLOSED = 'bi bi-door-closed'
    DOOR_OPEN_FILL = 'bi bi-door-open-fill'
    DOOR_OPEN = 'bi bi-door-open'
    DOT = 'bi bi-dot'
    DOWNLOAD = 'bi bi-download'
    DPAD_FILL = 'bi bi-dpad-fill'
    DPAD = 'bi bi-dpad'
    DRIBBBLE = 'bi bi-dribbble'
    DROPBOX = 'bi bi-dropbox'
    DROPLET_FILL = 'bi bi-droplet-fill'
    DROPLET_HALF = 'bi bi-droplet-half'
    DROPLET = 'bi bi-droplet'
    DUFFLE_FILL = 'bi bi-duffle-fill'
    DUFFLE = 'bi bi-duffle'
    EAR_FILL = 'bi bi-ear-fill'
    EAR = 'bi bi-ear'
    EARBUDS = 'bi bi-earbuds'
    EASEL_FILL = 'bi bi-easel-fill'
    EASEL = 'bi bi-easel'
    EASEL2_FILL = 'bi bi-easel2-fill'
    EASEL2 = 'bi bi-easel2'
    EASEL3_FILL = 'bi bi-easel3-fill'
    EASEL3 = 'bi bi-easel3'
    EGG_FILL = 'bi bi-egg-fill'
    EGG_FRIED = 'bi bi-egg-fried'
    EGG = 'bi bi-egg'
    EJECT_FILL = 'bi bi-eject-fill'
    EJECT = 'bi bi-eject'
    EMOJI_ANGRY_FILL = 'bi bi-emoji-angry-fill'
    EMOJI_ANGRY = 'bi bi-emoji-angry'
    EMOJI_ASTONISHED_FILL = 'bi bi-emoji-astonished-fill'
    EMOJI_ASTONISHED = 'bi bi-emoji-astonished'
    EMOJI_DIZZY_FILL = 'bi bi-emoji-dizzy-fill'
    EMOJI_DIZZY = 'bi bi-emoji-dizzy'
    EMOJI_EXPRESSIONLESS_FILL = 'bi bi-emoji-expressionless-fill'
    EMOJI_EXPRESSIONLESS = 'bi bi-emoji-expressionless'
    EMOJI_FROWN_FILL = 'bi bi-emoji-frown-fill'
    EMOJI_FROWN = 'bi bi-emoji-frown'
    EMOJI_GRIMACE_FILL = 'bi bi-emoji-grimace-fill'
    EMOJI_GRIMACE = 'bi bi-emoji-grimace'
    EMOJI_GRIN_FILL = 'bi bi-emoji-grin-fill'
    EMOJI_GRIN = 'bi bi-emoji-grin'
    EMOJI_HEART_EYES_FILL = 'bi bi-emoji-heart-eyes-fill'
    EMOJI_HEART_EYES = 'bi bi-emoji-heart-eyes'
    EMOJI_KISS_FILL = 'bi bi-emoji-kiss-fill'
    EMOJI_KISS = 'bi bi-emoji-kiss'
    EMOJI_LAUGHING_FILL = 'bi bi-emoji-laughing-fill'
    EMOJI_LAUGHING = 'bi bi-emoji-laughing'
    EMOJI_NEUTRAL_FILL = 'bi bi-emoji-neutral-fill'
    EMOJI_NEUTRAL = 'bi bi-emoji-neutral'
    EMOJI_SMILE_FILL = 'bi bi-emoji-smile-fill'
    EMOJI_SMILE_UPSIDE_DOWN_FILL = 'bi bi-emoji-smile-upside-down-fill'
    EMOJI_SMILE_UPSIDE_DOWN = 'bi bi-emoji-smile-upside-down'
    EMOJI_SMILE = 'bi bi-emoji-smile'
    EMOJI_SUNGLASSES_FILL = 'bi bi-emoji-sunglasses-fill'
    EMOJI_SUNGLASSES = 'bi bi-emoji-sunglasses'
    EMOJI_SURPRISE_FILL = 'bi bi-emoji-surprise-fill'
    EMOJI_SURPRISE = 'bi bi-emoji-surprise'
    EMOJI_TEAR_FILL = 'bi bi-emoji-tear-fill'
    EMOJI_TEAR = 'bi bi-emoji-tear'
    EMOJI_WINK_FILL = 'bi bi-emoji-wink-fill'
    EMOJI_WINK = 'bi bi-emoji-wink'
    ENVELOPE_ARROW_DOWN_FILL = 'bi bi-envelope-arrow-down-fill'
    ENVELOPE_ARROW_DOWN = 'bi bi-envelope-arrow-down'
    ENVELOPE_ARROW_UP_FILL = 'bi bi-envelope-arrow-up-fill'
    ENVELOPE_ARROW_UP = 'bi bi-envelope-arrow-up'
    ENVELOPE_AT_FILL = 'bi bi-envelope-at-fill'
    ENVELOPE_AT = 'bi bi-envelope-at'
    ENVELOPE_CHECK_FILL = 'bi bi-envelope-check-fill'
    ENVELOPE_CHECK = 'bi bi-envelope-check'
    ENVELOPE_DASH_FILL = 'bi bi-envelope-dash-fill'
    ENVELOPE_DASH = 'bi bi-envelope-dash'
    ENVELOPE_EXCLAMATION_FILL = 'bi bi-envelope-exclamation-fill'
    ENVELOPE_EXCLAMATION = 'bi bi-envelope-exclamation'
    ENVELOPE_FILL = 'bi bi-envelope-fill'
    ENVELOPE_HEART_FILL = 'bi bi-envelope-heart-fill'
    ENVELOPE_HEART = 'bi bi-envelope-heart'
    ENVELOPE_OPEN_FILL = 'bi bi-envelope-open-fill'
    ENVELOPE_OPEN_HEART_FILL = 'bi bi-envelope-open-heart-fill'
    ENVELOPE_OPEN_HEART = 'bi bi-envelope-open-heart'
    ENVELOPE_OPEN = 'bi bi-envelope-open'
    ENVELOPE_PAPER_FILL = 'bi bi-envelope-paper-fill'
    ENVELOPE_PAPER_HEART_FILL = 'bi bi-envelope-paper-heart-fill'
    ENVELOPE_PAPER_HEART = 'bi bi-envelope-paper-heart'
    ENVELOPE_PAPER = 'bi bi-envelope-paper'
    ENVELOPE_PLUS_FILL = 'bi bi-envelope-plus-fill'
    ENVELOPE_PLUS = 'bi bi-envelope-plus'
    ENVELOPE_SLASH_FILL = 'bi bi-envelope-slash-fill'
    ENVELOPE_SLASH = 'bi bi-envelope-slash'
    ENVELOPE_X_FILL = 'bi bi-envelope-x-fill'
    ENVELOPE_X = 'bi bi-envelope-x'
    ENVELOPE = 'bi bi-envelope'
    ERASER_FILL = 'bi bi-eraser-fill'
    ERASER = 'bi bi-eraser'
    ESCAPE = 'bi bi-escape'
    ETHERNET = 'bi bi-ethernet'
    EV_FRONT_FILL = 'bi bi-ev-front-fill'
    EV_FRONT = 'bi bi-ev-front'
    EV_STATION_FILL = 'bi bi-ev-station-fill'
    EV_STATION = 'bi bi-ev-station'
    EXCLAMATION_CIRCLE_FILL = 'bi bi-exclamation-circle-fill'
    EXCLAMATION_CIRCLE = 'bi bi-exclamation-circle'
    EXCLAMATION_DIAMOND_FILL = 'bi bi-exclamation-diamond-fill'
    EXCLAMATION_DIAMOND = 'bi bi-exclamation-diamond'
    EXCLAMATION_LG = 'bi bi-exclamation-lg'
    EXCLAMATION_OCTAGON_FILL = 'bi bi-exclamation-octagon-fill'
    EXCLAMATION_OCTAGON = 'bi bi-exclamation-octagon'
    EXCLAMATION_SQUARE_FILL = 'bi bi-exclamation-square-fill'
    EXCLAMATION_SQUARE = 'bi bi-exclamation-square'
    EXCLAMATION_TRIANGLE_FILL = 'bi bi-exclamation-triangle-fill'
    EXCLAMATION_TRIANGLE = 'bi bi-exclamation-triangle'
    EXCLAMATION = 'bi bi-exclamation'
    EXCLUDE = 'bi bi-exclude'
    EXPLICIT_FILL = 'bi bi-explicit-fill'
    EXPLICIT = 'bi bi-explicit'
    EXPOSURE = 'bi bi-exposure'
    EYE_FILL = 'bi bi-eye-fill'
    EYE_SLASH_FILL = 'bi bi-eye-slash-fill'
    EYE_SLASH = 'bi bi-eye-slash'
    EYE = 'bi bi-eye'
    EYEDROPPER = 'bi bi-eyedropper'
    EYEGLASSES = 'bi bi-eyeglasses'
    FACEBOOK = 'bi bi-facebook'
    FAN = 'bi bi-fan'
    FAST_FORWARD_BTN_FILL = 'bi bi-fast-forward-btn-fill'
    FAST_FORWARD_BTN = 'bi bi-fast-forward-btn'
    FAST_FORWARD_CIRCLE_FILL = 'bi bi-fast-forward-circle-fill'
    FAST_FORWARD_CIRCLE = 'bi bi-fast-forward-circle'
    FAST_FORWARD_FILL = 'bi bi-fast-forward-fill'
    FAST_FORWARD = 'bi bi-fast-forward'
    FEATHER = 'bi bi-feather'
    FEATHER2 = 'bi bi-feather2'
    FILE_ARROW_DOWN_FILL = 'bi bi-file-arrow-down-fill'
    FILE_ARROW_DOWN = 'bi bi-file-arrow-down'
    FILE_ARROW_UP_FILL = 'bi bi-file-arrow-up-fill'
    FILE_ARROW_UP = 'bi bi-file-arrow-up'
    FILE_BAR_GRAPH_FILL = 'bi bi-file-bar-graph-fill'
    FILE_BAR_GRAPH = 'bi bi-file-bar-graph'
    FILE_BINARY_FILL = 'bi bi-file-binary-fill'
    FILE_BINARY = 'bi bi-file-binary'
    FILE_BREAK_FILL = 'bi bi-file-break-fill'
    FILE_BREAK = 'bi bi-file-break'
    FILE_CHECK_FILL = 'bi bi-file-check-fill'
    FILE_CHECK = 'bi bi-file-check'
    FILE_CODE_FILL = 'bi bi-file-code-fill'
    FILE_CODE = 'bi bi-file-code'
    FILE_DIFF_FILL = 'bi bi-file-diff-fill'
    FILE_DIFF = 'bi bi-file-diff'
    FILE_EARMARK_ARROW_DOWN_FILL = 'bi bi-file-earmark-arrow-down-fill'
    FILE_EARMARK_ARROW_DOWN = 'bi bi-file-earmark-arrow-down'
    FILE_EARMARK_ARROW_UP_FILL = 'bi bi-file-earmark-arrow-up-fill'
    FILE_EARMARK_ARROW_UP = 'bi bi-file-earmark-arrow-up'
    FILE_EARMARK_BAR_GRAPH_FILL = 'bi bi-file-earmark-bar-graph-fill'
    FILE_EARMARK_BAR_GRAPH = 'bi bi-file-earmark-bar-graph'
    FILE_EARMARK_BINARY_FILL = 'bi bi-file-earmark-binary-fill'
    FILE_EARMARK_BINARY = 'bi bi-file-earmark-binary'
    FILE_EARMARK_BREAK_FILL = 'bi bi-file-earmark-break-fill'
    FILE_EARMARK_BREAK = 'bi bi-file-earmark-break'
    FILE_EARMARK_CHECK_FILL = 'bi bi-file-earmark-check-fill'
    FILE_EARMARK_CHECK = 'bi bi-file-earmark-check'
    FILE_EARMARK_CODE_FILL = 'bi bi-file-earmark-code-fill'
    FILE_EARMARK_CODE = 'bi bi-file-earmark-code'
    FILE_EARMARK_DIFF_FILL = 'bi bi-file-earmark-diff-fill'
    FILE_EARMARK_DIFF = 'bi bi-file-earmark-diff'
    FILE_EARMARK_EASEL_FILL = 'bi bi-file-earmark-easel-fill'
    FILE_EARMARK_EASEL = 'bi bi-file-earmark-easel'
    FILE_EARMARK_EXCEL_FILL = 'bi bi-file-earmark-excel-fill'
    FILE_EARMARK_EXCEL = 'bi bi-file-earmark-excel'
    FILE_EARMARK_FILL = 'bi bi-file-earmark-fill'
    FILE_EARMARK_FONT_FILL = 'bi bi-file-earmark-font-fill'
    FILE_EARMARK_FONT = 'bi bi-file-earmark-font'
    FILE_EARMARK_IMAGE_FILL = 'bi bi-file-earmark-image-fill'
    FILE_EARMARK_IMAGE = 'bi bi-file-earmark-image'
    FILE_EARMARK_LOCK_FILL = 'bi bi-file-earmark-lock-fill'
    FILE_EARMARK_LOCK = 'bi bi-file-earmark-lock'
    FILE_EARMARK_LOCK2_FILL = 'bi bi-file-earmark-lock2-fill'
    FILE_EARMARK_LOCK2 = 'bi bi-file-earmark-lock2'
    FILE_EARMARK_MEDICAL_FILL = 'bi bi-file-earmark-medical-fill'
    FILE_EARMARK_MEDICAL = 'bi bi-file-earmark-medical'
    FILE_EARMARK_MINUS_FILL = 'bi bi-file-earmark-minus-fill'
    FILE_EARMARK_MINUS = 'bi bi-file-earmark-minus'
    FILE_EARMARK_MUSIC_FILL = 'bi bi-file-earmark-music-fill'
    FILE_EARMARK_MUSIC = 'bi bi-file-earmark-music'
    FILE_EARMARK_PDF_FILL = 'bi bi-file-earmark-pdf-fill'
    FILE_EARMARK_PDF = 'bi bi-file-earmark-pdf'
    FILE_EARMARK_PERSON_FILL = 'bi bi-file-earmark-person-fill'
    FILE_EARMARK_PERSON = 'bi bi-file-earmark-person'
    FILE_EARMARK_PLAY_FILL = 'bi bi-file-earmark-play-fill'
    FILE_EARMARK_PLAY = 'bi bi-file-earmark-play'
    FILE_EARMARK_PLUS_FILL = 'bi bi-file-earmark-plus-fill'
    FILE_EARMARK_PLUS = 'bi bi-file-earmark-plus'
    FILE_EARMARK_POST_FILL = 'bi bi-file-earmark-post-fill'
    FILE_EARMARK_POST = 'bi bi-file-earmark-post'
    FILE_EARMARK_PPT_FILL = 'bi bi-file-earmark-ppt-fill'
    FILE_EARMARK_PPT = 'bi bi-file-earmark-ppt'
    FILE_EARMARK_RICHTEXT_FILL = 'bi bi-file-earmark-richtext-fill'
    FILE_EARMARK_RICHTEXT = 'bi bi-file-earmark-richtext'
    FILE_EARMARK_RULED_FILL = 'bi bi-file-earmark-ruled-fill'
    FILE_EARMARK_RULED = 'bi bi-file-earmark-ruled'
    FILE_EARMARK_SLIDES_FILL = 'bi bi-file-earmark-slides-fill'
    FILE_EARMARK_SLIDES = 'bi bi-file-earmark-slides'
    FILE_EARMARK_SPREADSHEET_FILL = 'bi bi-file-earmark-spreadsheet-fill'
    FILE_EARMARK_SPREADSHEET = 'bi bi-file-earmark-spreadsheet'
    FILE_EARMARK_TEXT_FILL = 'bi bi-file-earmark-text-fill'
    FILE_EARMARK_TEXT = 'bi bi-file-earmark-text'
    FILE_EARMARK_WORD_FILL = 'bi bi-file-earmark-word-fill'
    FILE_EARMARK_WORD = 'bi bi-file-earmark-word'
    FILE_EARMARK_X_FILL = 'bi bi-file-earmark-x-fill'
    FILE_EARMARK_X = 'bi bi-file-earmark-x'
    FILE_EARMARK_ZIP_FILL = 'bi bi-file-earmark-zip-fill'
    FILE_EARMARK_ZIP = 'bi bi-file-earmark-zip'
    FILE_EARMARK = 'bi bi-file-earmark'
    FILE_EASEL_FILL = 'bi bi-file-easel-fill'
    FILE_EASEL = 'bi bi-file-easel'
    FILE_EXCEL_FILL = 'bi bi-file-excel-fill'
    FILE_EXCEL = 'bi bi-file-excel'
    FILE_FILL = 'bi bi-file-fill'
    FILE_FONT_FILL = 'bi bi-file-font-fill'
    FILE_FONT = 'bi bi-file-font'
    FILE_IMAGE_FILL = 'bi bi-file-image-fill'
    FILE_IMAGE = 'bi bi-file-image'
    FILE_LOCK_FILL = 'bi bi-file-lock-fill'
    FILE_LOCK = 'bi bi-file-lock'
    FILE_LOCK2_FILL = 'bi bi-file-lock2-fill'
    FILE_LOCK2 = 'bi bi-file-lock2'
    FILE_MEDICAL_FILL = 'bi bi-file-medical-fill'
    FILE_MEDICAL = 'bi bi-file-medical'
    FILE_MINUS_FILL = 'bi bi-file-minus-fill'
    FILE_MINUS = 'bi bi-file-minus'
    FILE_MUSIC_FILL = 'bi bi-file-music-fill'
    FILE_MUSIC = 'bi bi-file-music'
    FILE_PDF_FILL = 'bi bi-file-pdf-fill'
    FILE_PDF = 'bi bi-file-pdf'
    FILE_PERSON_FILL = 'bi bi-file-person-fill'
    FILE_PERSON = 'bi bi-file-person'
    FILE_PLAY_FILL = 'bi bi-file-play-fill'
    FILE_PLAY = 'bi bi-file-play'
    FILE_PLUS_FILL = 'bi bi-file-plus-fill'
    FILE_PLUS = 'bi bi-file-plus'
    FILE_POST_FILL = 'bi bi-file-post-fill'
    FILE_POST = 'bi bi-file-post'
    FILE_PPT_FILL = 'bi bi-file-ppt-fill'
    FILE_PPT = 'bi bi-file-ppt'
    FILE_RICHTEXT_FILL = 'bi bi-file-richtext-fill'
    FILE_RICHTEXT = 'bi bi-file-richtext'
    FILE_RULED_FILL = 'bi bi-file-ruled-fill'
    FILE_RULED = 'bi bi-file-ruled'
    FILE_SLIDES_FILL = 'bi bi-file-slides-fill'
    FILE_SLIDES = 'bi bi-file-slides'
    FILE_SPREADSHEET_FILL = 'bi bi-file-spreadsheet-fill'
    FILE_SPREADSHEET = 'bi bi-file-spreadsheet'
    FILE_TEXT_FILL = 'bi bi-file-text-fill'
    FILE_TEXT = 'bi bi-file-text'
    FILE_WORD_FILL = 'bi bi-file-word-fill'
    FILE_WORD = 'bi bi-file-word'
    FILE_X_FILL = 'bi bi-file-x-fill'
    FILE_X = 'bi bi-file-x'
    FILE_ZIP_FILL = 'bi bi-file-zip-fill'
    FILE_ZIP = 'bi bi-file-zip'
    FILE = 'bi bi-file'
    FILES_ALT = 'bi bi-files-alt'
    FILES = 'bi bi-files'
    FILETYPE_AAC = 'bi bi-filetype-aac'
    FILETYPE_AI = 'bi bi-filetype-ai'
    FILETYPE_BMP = 'bi bi-filetype-bmp'
    FILETYPE_CS = 'bi bi-filetype-cs'
    FILETYPE_CSS = 'bi bi-filetype-css'
    FILETYPE_CSV = 'bi bi-filetype-csv'
    FILETYPE_DOC = 'bi bi-filetype-doc'
    FILETYPE_DOCX = 'bi bi-filetype-docx'
    FILETYPE_EXE = 'bi bi-filetype-exe'
    FILETYPE_GIF = 'bi bi-filetype-gif'
    FILETYPE_HEIC = 'bi bi-filetype-heic'
    FILETYPE_HTML = 'bi bi-filetype-html'
    FILETYPE_JAVA = 'bi bi-filetype-java'
    FILETYPE_JPG = 'bi bi-filetype-jpg'
    FILETYPE_JS = 'bi bi-filetype-js'
    FILETYPE_JSON = 'bi bi-filetype-json'
    FILETYPE_JSX = 'bi bi-filetype-jsx'
    FILETYPE_KEY = 'bi bi-filetype-key'
    FILETYPE_M4P = 'bi bi-filetype-m4p'
    FILETYPE_MD = 'bi bi-filetype-md'
    FILETYPE_MDX = 'bi bi-filetype-mdx'
    FILETYPE_MOV = 'bi bi-filetype-mov'
    FILETYPE_MP3 = 'bi bi-filetype-mp3'
    FILETYPE_MP4 = 'bi bi-filetype-mp4'
    FILETYPE_OTF = 'bi bi-filetype-otf'
    FILETYPE_PDF = 'bi bi-filetype-pdf'
    FILETYPE_PHP = 'bi bi-filetype-php'
    FILETYPE_PNG = 'bi bi-filetype-png'
    FILETYPE_PPT = 'bi bi-filetype-ppt'
    FILETYPE_PPTX = 'bi bi-filetype-pptx'
    FILETYPE_PSD = 'bi bi-filetype-psd'
    FILETYPE_PY = 'bi bi-filetype-py'
    FILETYPE_RAW = 'bi bi-filetype-raw'
    FILETYPE_RB = 'bi bi-filetype-rb'
    FILETYPE_SASS = 'bi bi-filetype-sass'
    FILETYPE_SCSS = 'bi bi-filetype-scss'
    FILETYPE_SH = 'bi bi-filetype-sh'
    FILETYPE_SQL = 'bi bi-filetype-sql'
    FILETYPE_SVG = 'bi bi-filetype-svg'
    FILETYPE_TIFF = 'bi bi-filetype-tiff'
    FILETYPE_TSX = 'bi bi-filetype-tsx'
    FILETYPE_TTF = 'bi bi-filetype-ttf'
    FILETYPE_TXT = 'bi bi-filetype-txt'
    FILETYPE_WAV = 'bi bi-filetype-wav'
    FILETYPE_WOFF = 'bi bi-filetype-woff'
    FILETYPE_XLS = 'bi bi-filetype-xls'
    FILETYPE_XLSX = 'bi bi-filetype-xlsx'
    FILETYPE_XML = 'bi bi-filetype-xml'
    FILETYPE_YML = 'bi bi-filetype-yml'
    FILM = 'bi bi-film'
    FILTER_CIRCLE_FILL = 'bi bi-filter-circle-fill'
    FILTER_CIRCLE = 'bi bi-filter-circle'
    FILTER_LEFT = 'bi bi-filter-left'
    FILTER_RIGHT = 'bi bi-filter-right'
    FILTER_SQUARE_FILL = 'bi bi-filter-square-fill'
    FILTER_SQUARE = 'bi bi-filter-square'
    FILTER = 'bi bi-filter'
    FINGERPRINT = 'bi bi-fingerprint'
    FIRE = 'bi bi-fire'
    FLAG_FILL = 'bi bi-flag-fill'
    FLAG = 'bi bi-flag'
    FLOPPY_FILL = 'bi bi-floppy-fill'
    FLOPPY = 'bi bi-floppy'
    FLOPPY2_FILL = 'bi bi-floppy2-fill'
    FLOPPY2 = 'bi bi-floppy2'
    FLOWER1 = 'bi bi-flower1'
    FLOWER2 = 'bi bi-flower2'
    FLOWER3 = 'bi bi-flower3'
    FOLDER_CHECK = 'bi bi-folder-check'
    FOLDER_FILL = 'bi bi-folder-fill'
    FOLDER_MINUS = 'bi bi-folder-minus'
    FOLDER_PLUS = 'bi bi-folder-plus'
    FOLDER_SYMLINK_FILL = 'bi bi-folder-symlink-fill'
    FOLDER_SYMLINK = 'bi bi-folder-symlink'
    FOLDER_X = 'bi bi-folder-x'
    FOLDER = 'bi bi-folder'
    FOLDER2_OPEN = 'bi bi-folder2-open'
    FOLDER2 = 'bi bi-folder2'
    FONTS = 'bi bi-fonts'
    FORWARD_FILL = 'bi bi-forward-fill'
    FORWARD = 'bi bi-forward'
    FRONT = 'bi bi-front'
    FUEL_PUMP_DIESEL_FILL = 'bi bi-fuel-pump-diesel-fill'
    FUEL_PUMP_DIESEL = 'bi bi-fuel-pump-diesel'
    FUEL_PUMP_FILL = 'bi bi-fuel-pump-fill'
    FUEL_PUMP = 'bi bi-fuel-pump'
    FULLSCREEN_EXIT = 'bi bi-fullscreen-exit'
    FULLSCREEN = 'bi bi-fullscreen'
    FUNNEL_FILL = 'bi bi-funnel-fill'
    FUNNEL = 'bi bi-funnel'
    GEAR_FILL = 'bi bi-gear-fill'
    GEAR_WIDE_CONNECTED = 'bi bi-gear-wide-connected'
    GEAR_WIDE = 'bi bi-gear-wide'
    GEAR = 'bi bi-gear'
    GEM = 'bi bi-gem'
    GENDER_AMBIGUOUS = 'bi bi-gender-ambiguous'
    GENDER_FEMALE = 'bi bi-gender-female'
    GENDER_MALE = 'bi bi-gender-male'
    GENDER_NEUTER = 'bi bi-gender-neuter'
    GENDER_TRANS = 'bi bi-gender-trans'
    GEO_ALT_FILL = 'bi bi-geo-alt-fill'
    GEO_ALT = 'bi bi-geo-alt'
    GEO_FILL = 'bi bi-geo-fill'
    GEO = 'bi bi-geo'
    GIFT_FILL = 'bi bi-gift-fill'
    GIFT = 'bi bi-gift'
    GIT = 'bi bi-git'
    GITHUB = 'bi bi-github'
    GITLAB = 'bi bi-gitlab'
    GLOBE_AMERICAS = 'bi bi-globe-americas'
    GLOBE_ASIA_AUSTRALIA = 'bi bi-globe-asia-australia'
    GLOBE_CENTRAL_SOUTH_ASIA = 'bi bi-globe-central-south-asia'
    GLOBE_EUROPE_AFRICA = 'bi bi-globe-europe-africa'
    GLOBE = 'bi bi-globe'
    GLOBE2 = 'bi bi-globe2'
    GOOGLE_PLAY = 'bi bi-google-play'
    GOOGLE = 'bi bi-google'
    GPU_CARD = 'bi bi-gpu-card'
    GRAPH_DOWN_ARROW = 'bi bi-graph-down-arrow'
    GRAPH_DOWN = 'bi bi-graph-down'
    GRAPH_UP_ARROW = 'bi bi-graph-up-arrow'
    GRAPH_UP = 'bi bi-graph-up'
    GRID_1X2_FILL = 'bi bi-grid-1x2-fill'
    GRID_1X2 = 'bi bi-grid-1x2'
    GRID_3X2_GAP_FILL = 'bi bi-grid-3x2-gap-fill'
    GRID_3X2_GAP = 'bi bi-grid-3x2-gap'
    GRID_3X2 = 'bi bi-grid-3x2'
    GRID_3X3_GAP_FILL = 'bi bi-grid-3x3-gap-fill'
    GRID_3X3_GAP = 'bi bi-grid-3x3-gap'
    GRID_3X3 = 'bi bi-grid-3x3'
    GRID_FILL = 'bi bi-grid-fill'
    GRID = 'bi bi-grid'
    GRIP_HORIZONTAL = 'bi bi-grip-horizontal'
    GRIP_VERTICAL = 'bi bi-grip-vertical'
    H_CIRCLE_FILL = 'bi bi-h-circle-fill'
    H_CIRCLE = 'bi bi-h-circle'
    H_SQUARE_FILL = 'bi bi-h-square-fill'
    H_SQUARE = 'bi bi-h-square'
    HAMMER = 'bi bi-hammer'
    HAND_INDEX_FILL = 'bi bi-hand-index-fill'
    HAND_INDEX_THUMB_FILL = 'bi bi-hand-index-thumb-fill'
    HAND_INDEX_THUMB = 'bi bi-hand-index-thumb'
    HAND_INDEX = 'bi bi-hand-index'
    HAND_THUMBS_DOWN_FILL = 'bi bi-hand-thumbs-down-fill'
    HAND_THUMBS_DOWN = 'bi bi-hand-thumbs-down'
    HAND_THUMBS_UP_FILL = 'bi bi-hand-thumbs-up-fill'
    HAND_THUMBS_UP = 'bi bi-hand-thumbs-up'
    HANDBAG_FILL = 'bi bi-handbag-fill'
    HANDBAG = 'bi bi-handbag'
    HASH = 'bi bi-hash'
    HDD_FILL = 'bi bi-hdd-fill'
    HDD_NETWORK_FILL = 'bi bi-hdd-network-fill'
    HDD_NETWORK = 'bi bi-hdd-network'
    HDD_RACK_FILL = 'bi bi-hdd-rack-fill'
    HDD_RACK = 'bi bi-hdd-rack'
    HDD_STACK_FILL = 'bi bi-hdd-stack-fill'
    HDD_STACK = 'bi bi-hdd-stack'
    HDD = 'bi bi-hdd'
    HDMI_FILL = 'bi bi-hdmi-fill'
    HDMI = 'bi bi-hdmi'
    HEADPHONES = 'bi bi-headphones'
    HEADSET_VR = 'bi bi-headset-vr'
    HEADSET = 'bi bi-headset'
    HEART_ARROW = 'bi bi-heart-arrow'
    HEART_FILL = 'bi bi-heart-fill'
    HEART_HALF = 'bi bi-heart-half'
    HEART_PULSE_FILL = 'bi bi-heart-pulse-fill'
    HEART_PULSE = 'bi bi-heart-pulse'
    HEART = 'bi bi-heart'
    HEARTBREAK_FILL = 'bi bi-heartbreak-fill'
    HEARTBREAK = 'bi bi-heartbreak'
    HEARTS = 'bi bi-hearts'
    HEPTAGON_FILL = 'bi bi-heptagon-fill'
    HEPTAGON_HALF = 'bi bi-heptagon-half'
    HEPTAGON = 'bi bi-heptagon'
    HEXAGON_FILL = 'bi bi-hexagon-fill'
    HEXAGON_HALF = 'bi bi-hexagon-half'
    HEXAGON = 'bi bi-hexagon'
    HIGHLIGHTER = 'bi bi-highlighter'
    HIGHLIGHTS = 'bi bi-highlights'
    HOSPITAL_FILL = 'bi bi-hospital-fill'
    HOSPITAL = 'bi bi-hospital'
    HOURGLASS_BOTTOM = 'bi bi-hourglass-bottom'
    HOURGLASS_SPLIT = 'bi bi-hourglass-split'
    HOURGLASS_TOP = 'bi bi-hourglass-top'
    HOURGLASS = 'bi bi-hourglass'
    HOUSE_ADD_FILL = 'bi bi-house-add-fill'
    HOUSE_ADD = 'bi bi-house-add'
    HOUSE_CHECK_FILL = 'bi bi-house-check-fill'
    HOUSE_CHECK = 'bi bi-house-check'
    HOUSE_DASH_FILL = 'bi bi-house-dash-fill'
    HOUSE_DASH = 'bi bi-house-dash'
    HOUSE_DOOR_FILL = 'bi bi-house-door-fill'
    HOUSE_DOOR = 'bi bi-house-door'
    HOUSE_DOWN_FILL = 'bi bi-house-down-fill'
    HOUSE_DOWN = 'bi bi-house-down'
    HOUSE_EXCLAMATION_FILL = 'bi bi-house-exclamation-fill'
    HOUSE_EXCLAMATION = 'bi bi-house-exclamation'
    HOUSE_FILL = 'bi bi-house-fill'
    HOUSE_GEAR_FILL = 'bi bi-house-gear-fill'
    HOUSE_GEAR = 'bi bi-house-gear'
    HOUSE_HEART_FILL = 'bi bi-house-heart-fill'
    HOUSE_HEART = 'bi bi-house-heart'
    HOUSE_LOCK_FILL = 'bi bi-house-lock-fill'
    HOUSE_LOCK = 'bi bi-house-lock'
    HOUSE_SLASH_FILL = 'bi bi-house-slash-fill'
    HOUSE_SLASH = 'bi bi-house-slash'
    HOUSE_UP_FILL = 'bi bi-house-up-fill'
    HOUSE_UP = 'bi bi-house-up'
    HOUSE_X_FILL = 'bi bi-house-x-fill'
    HOUSE_X = 'bi bi-house-x'
    HOUSE = 'bi bi-house'
    HOUSES_FILL = 'bi bi-houses-fill'
    HOUSES = 'bi bi-houses'
    HR = 'bi bi-hr'
    HURRICANE = 'bi bi-hurricane'
    HYPNOTIZE = 'bi bi-hypnotize'
    IMAGE_ALT = 'bi bi-image-alt'
    IMAGE_FILL = 'bi bi-image-fill'
    IMAGE = 'bi bi-image'
    IMAGES = 'bi bi-images'
    INBOX_FILL = 'bi bi-inbox-fill'
    INBOX = 'bi bi-inbox'
    INBOXES_FILL = 'bi bi-inboxes-fill'
    INBOXES = 'bi bi-inboxes'
    INCOGNITO = 'bi bi-incognito'
    INDENT = 'bi bi-indent'
    INFINITY = 'bi bi-infinity'
    INFO_CIRCLE_FILL = 'bi bi-info-circle-fill'
    INFO_CIRCLE = 'bi bi-info-circle'
    INFO_LG = 'bi bi-info-lg'
    INFO_SQUARE_FILL = 'bi bi-info-square-fill'
    INFO_SQUARE = 'bi bi-info-square'
    INFO = 'bi bi-info'
    INPUT_CURSOR_TEXT = 'bi bi-input-cursor-text'
    INPUT_CURSOR = 'bi bi-input-cursor'
    INSTAGRAM = 'bi bi-instagram'
    INTERSECT = 'bi bi-intersect'
    JOURNAL_ALBUM = 'bi bi-journal-album'
    JOURNAL_ARROW_DOWN = 'bi bi-journal-arrow-down'
    JOURNAL_ARROW_UP = 'bi bi-journal-arrow-up'
    JOURNAL_BOOKMARK_FILL = 'bi bi-journal-bookmark-fill'
    JOURNAL_BOOKMARK = 'bi bi-journal-bookmark'
    JOURNAL_CHECK = 'bi bi-journal-check'
    JOURNAL_CODE = 'bi bi-journal-code'
    JOURNAL_MEDICAL = 'bi bi-journal-medical'
    JOURNAL_MINUS = 'bi bi-journal-minus'
    JOURNAL_PLUS = 'bi bi-journal-plus'
    JOURNAL_RICHTEXT = 'bi bi-journal-richtext'
    JOURNAL_TEXT = 'bi bi-journal-text'
    JOURNAL_X = 'bi bi-journal-x'
    JOURNAL = 'bi bi-journal'
    JOURNALS = 'bi bi-journals'
    JOYSTICK = 'bi bi-joystick'
    JUSTIFY_LEFT = 'bi bi-justify-left'
    JUSTIFY_RIGHT = 'bi bi-justify-right'
    JUSTIFY = 'bi bi-justify'
    KANBAN_FILL = 'bi bi-kanban-fill'
    KANBAN = 'bi bi-kanban'
    KEY_FILL = 'bi bi-key-fill'
    KEY = 'bi bi-key'
    KEYBOARD_FILL = 'bi bi-keyboard-fill'
    KEYBOARD = 'bi bi-keyboard'
    LADDER = 'bi bi-ladder'
    LAMP_FILL = 'bi bi-lamp-fill'
    LAMP = 'bi bi-lamp'
    LAPTOP_FILL = 'bi bi-laptop-fill'
    LAPTOP = 'bi bi-laptop'
    LAYER_BACKWARD = 'bi bi-layer-backward'
    LAYER_FORWARD = 'bi bi-layer-forward'
    LAYERS_FILL = 'bi bi-layers-fill'
    LAYERS_HALF = 'bi bi-layers-half'
    LAYERS = 'bi bi-layers'
    LAYOUT_SIDEBAR_INSET_REVERSE = 'bi bi-layout-sidebar-inset-reverse'
    LAYOUT_SIDEBAR_INSET = 'bi bi-layout-sidebar-inset'
    LAYOUT_SIDEBAR_REVERSE = 'bi bi-layout-sidebar-reverse'
    LAYOUT_SIDEBAR = 'bi bi-layout-sidebar'
    LAYOUT_SPLIT = 'bi bi-layout-split'
    LAYOUT_TEXT_SIDEBAR_REVERSE = 'bi bi-layout-text-sidebar-reverse'
    LAYOUT_TEXT_SIDEBAR = 'bi bi-layout-text-sidebar'
    LAYOUT_TEXT_WINDOW_REVERSE = 'bi bi-layout-text-window-reverse'
    LAYOUT_TEXT_WINDOW = 'bi bi-layout-text-window'
    LAYOUT_THREE_COLUMNS = 'bi bi-layout-three-columns'
    LAYOUT_WTF = 'bi bi-layout-wtf'
    LIFE_PRESERVER = 'bi bi-life-preserver'
    LIGHTBULB_FILL = 'bi bi-lightbulb-fill'
    LIGHTBULB_OFF_FILL = 'bi bi-lightbulb-off-fill'
    LIGHTBULB_OFF = 'bi bi-lightbulb-off'
    LIGHTBULB = 'bi bi-lightbulb'
    LIGHTNING_CHARGE_FILL = 'bi bi-lightning-charge-fill'
    LIGHTNING_CHARGE = 'bi bi-lightning-charge'
    LIGHTNING_FILL = 'bi bi-lightning-fill'
    LIGHTNING = 'bi bi-lightning'
    LINE = 'bi bi-line'
    LINK_45DEG = 'bi bi-link-45deg'
    LINK = 'bi bi-link'
    LINKEDIN = 'bi bi-linkedin'
    LIST_CHECK = 'bi bi-list-check'
    LIST_COLUMNS_REVERSE = 'bi bi-list-columns-reverse'
    LIST_COLUMNS = 'bi bi-list-columns'
    LIST_NESTED = 'bi bi-list-nested'
    LIST_OL = 'bi bi-list-ol'
    LIST_STARS = 'bi bi-list-stars'
    LIST_TASK = 'bi bi-list-task'
    LIST_UL = 'bi bi-list-ul'
    LIST = 'bi bi-list'
    LOCK_FILL = 'bi bi-lock-fill'
    LOCK = 'bi bi-lock'
    LUGGAGE_FILL = 'bi bi-luggage-fill'
    LUGGAGE = 'bi bi-luggage'
    LUNGS_FILL = 'bi bi-lungs-fill'
    LUNGS = 'bi bi-lungs'
    MAGIC = 'bi bi-magic'
    MAGNET_FILL = 'bi bi-magnet-fill'
    MAGNET = 'bi bi-magnet'
    MAILBOX_FLAG = 'bi bi-mailbox-flag'
    MAILBOX = 'bi bi-mailbox'
    MAILBOX2_FLAG = 'bi bi-mailbox2-flag'
    MAILBOX2 = 'bi bi-mailbox2'
    MAP_FILL = 'bi bi-map-fill'
    MAP = 'bi bi-map'
    MARKDOWN_FILL = 'bi bi-markdown-fill'
    MARKDOWN = 'bi bi-markdown'
    MARKER_TIP = 'bi bi-marker-tip'
    MASK = 'bi bi-mask'
    MASTODON = 'bi bi-mastodon'
    MEDIUM = 'bi bi-medium'
    MEGAPHONE_FILL = 'bi bi-megaphone-fill'
    MEGAPHONE = 'bi bi-megaphone'
    MEMORY = 'bi bi-memory'
    MENU_APP_FILL = 'bi bi-menu-app-fill'
    MENU_APP = 'bi bi-menu-app'
    MENU_BUTTON_FILL = 'bi bi-menu-button-fill'
    MENU_BUTTON_WIDE_FILL = 'bi bi-menu-button-wide-fill'
    MENU_BUTTON_WIDE = 'bi bi-menu-button-wide'
    MENU_BUTTON = 'bi bi-menu-button'
    MENU_DOWN = 'bi bi-menu-down'
    MENU_UP = 'bi bi-menu-up'
    MESSENGER = 'bi bi-messenger'
    META = 'bi bi-meta'
    MIC_FILL = 'bi bi-mic-fill'
    MIC_MUTE_FILL = 'bi bi-mic-mute-fill'
    MIC_MUTE = 'bi bi-mic-mute'
    MIC = 'bi bi-mic'
    MICROSOFT_TEAMS = 'bi bi-microsoft-teams'
    MICROSOFT = 'bi bi-microsoft'
    MINECART_LOADED = 'bi bi-minecart-loaded'
    MINECART = 'bi bi-minecart'
    MODEM_FILL = 'bi bi-modem-fill'
    MODEM = 'bi bi-modem'
    MOISTURE = 'bi bi-moisture'
    MOON_FILL = 'bi bi-moon-fill'
    MOON_STARS_FILL = 'bi bi-moon-stars-fill'
    MOON_STARS = 'bi bi-moon-stars'
    MOON = 'bi bi-moon'
    MORTARBOARD_FILL = 'bi bi-mortarboard-fill'
    MORTARBOARD = 'bi bi-mortarboard'
    MOTHERBOARD_FILL = 'bi bi-motherboard-fill'
    MOTHERBOARD = 'bi bi-motherboard'
    MOUSE_FILL = 'bi bi-mouse-fill'
    MOUSE = 'bi bi-mouse'
    MOUSE2_FILL = 'bi bi-mouse2-fill'
    MOUSE2 = 'bi bi-mouse2'
    MOUSE3_FILL = 'bi bi-mouse3-fill'
    MOUSE3 = 'bi bi-mouse3'
    MUSIC_NOTE_BEAMED = 'bi bi-music-note-beamed'
    MUSIC_NOTE_LIST = 'bi bi-music-note-list'
    MUSIC_NOTE = 'bi bi-music-note'
    MUSIC_PLAYER_FILL = 'bi bi-music-player-fill'
    MUSIC_PLAYER = 'bi bi-music-player'
    NEWSPAPER = 'bi bi-newspaper'
    NINTENDO_SWITCH = 'bi bi-nintendo-switch'
    NODE_MINUS_FILL = 'bi bi-node-minus-fill'
    NODE_MINUS = 'bi bi-node-minus'
    NODE_PLUS_FILL = 'bi bi-node-plus-fill'
    NODE_PLUS = 'bi bi-node-plus'
    NOISE_REDUCTION = 'bi bi-noise-reduction'
    NUT_FILL = 'bi bi-nut-fill'
    NUT = 'bi bi-nut'
    NVIDIA = 'bi bi-nvidia'
    NVME_FILL = 'bi bi-nvme-fill'
    NVME = 'bi bi-nvme'
    OCTAGON_FILL = 'bi bi-octagon-fill'
    OCTAGON_HALF = 'bi bi-octagon-half'
    OCTAGON = 'bi bi-octagon'
    OPENCOLLECTIVE = 'bi bi-opencollective'
    OPTICAL_AUDIO_FILL = 'bi bi-optical-audio-fill'
    OPTICAL_AUDIO = 'bi bi-optical-audio'
    OPTION = 'bi bi-option'
    OUTLET = 'bi bi-outlet'
    P_CIRCLE_FILL = 'bi bi-p-circle-fill'
    P_CIRCLE = 'bi bi-p-circle'
    P_SQUARE_FILL = 'bi bi-p-square-fill'
    P_SQUARE = 'bi bi-p-square'
    PAINT_BUCKET = 'bi bi-paint-bucket'
    PALETTE_FILL = 'bi bi-palette-fill'
    PALETTE = 'bi bi-palette'
    PALETTE2 = 'bi bi-palette2'
    PAPERCLIP = 'bi bi-paperclip'
    PARAGRAPH = 'bi bi-paragraph'
    PASS_FILL = 'bi bi-pass-fill'
    PASS = 'bi bi-pass'
    PASSPORT_FILL = 'bi bi-passport-fill'
    PASSPORT = 'bi bi-passport'
    PATCH_CHECK_FILL = 'bi bi-patch-check-fill'
    PATCH_CHECK = 'bi bi-patch-check'
    PATCH_EXCLAMATION_FILL = 'bi bi-patch-exclamation-fill'
    PATCH_EXCLAMATION = 'bi bi-patch-exclamation'
    PATCH_MINUS_FILL = 'bi bi-patch-minus-fill'
    PATCH_MINUS = 'bi bi-patch-minus'
    PATCH_PLUS_FILL = 'bi bi-patch-plus-fill'
    PATCH_PLUS = 'bi bi-patch-plus'
    PATCH_QUESTION_FILL = 'bi bi-patch-question-fill'
    PATCH_QUESTION = 'bi bi-patch-question'
    PAUSE_BTN_FILL = 'bi bi-pause-btn-fill'
    PAUSE_BTN = 'bi bi-pause-btn'
    PAUSE_CIRCLE_FILL = 'bi bi-pause-circle-fill'
    PAUSE_CIRCLE = 'bi bi-pause-circle'
    PAUSE_FILL = 'bi bi-pause-fill'
    PAUSE = 'bi bi-pause'
    PAYPAL = 'bi bi-paypal'
    PC_DISPLAY_HORIZONTAL = 'bi bi-pc-display-horizontal'
    PC_DISPLAY = 'bi bi-pc-display'
    PC_HORIZONTAL = 'bi bi-pc-horizontal'
    PC = 'bi bi-pc'
    PCI_CARD_NETWORK = 'bi bi-pci-card-network'
    PCI_CARD_SOUND = 'bi bi-pci-card-sound'
    PCI_CARD = 'bi bi-pci-card'
    PEACE_FILL = 'bi bi-peace-fill'
    PEACE = 'bi bi-peace'
    PEN_FILL = 'bi bi-pen-fill'
    PEN = 'bi bi-pen'
    PENCIL_FILL = 'bi bi-pencil-fill'
    PENCIL_SQUARE = 'bi bi-pencil-square'
    PENCIL = 'bi bi-pencil'
    PENTAGON_FILL = 'bi bi-pentagon-fill'
    PENTAGON_HALF = 'bi bi-pentagon-half'
    PENTAGON = 'bi bi-pentagon'
    PEOPLE_FILL = 'bi bi-people-fill'
    PEOPLE = 'bi bi-people'
    PERCENT = 'bi bi-percent'
    PERSON_ADD = 'bi bi-person-add'
    PERSON_ARMS_UP = 'bi bi-person-arms-up'
    PERSON_BADGE_FILL = 'bi bi-person-badge-fill'
    PERSON_BADGE = 'bi bi-person-badge'
    PERSON_BOUNDING_BOX = 'bi bi-person-bounding-box'
    PERSON_CHECK_FILL = 'bi bi-person-check-fill'
    PERSON_CHECK = 'bi bi-person-check'
    PERSON_CIRCLE = 'bi bi-person-circle'
    PERSON_DASH_FILL = 'bi bi-person-dash-fill'
    PERSON_DASH = 'bi bi-person-dash'
    PERSON_DOWN = 'bi bi-person-down'
    PERSON_EXCLAMATION = 'bi bi-person-exclamation'
    PERSON_FILL_ADD = 'bi bi-person-fill-add'
    PERSON_FILL_CHECK = 'bi bi-person-fill-check'
    PERSON_FILL_DASH = 'bi bi-person-fill-dash'
    PERSON_FILL_DOWN = 'bi bi-person-fill-down'
    PERSON_FILL_EXCLAMATION = 'bi bi-person-fill-exclamation'
    PERSON_FILL_GEAR = 'bi bi-person-fill-gear'
    PERSON_FILL_LOCK = 'bi bi-person-fill-lock'
    PERSON_FILL_SLASH = 'bi bi-person-fill-slash'
    PERSON_FILL_UP = 'bi bi-person-fill-up'
    PERSON_FILL_X = 'bi bi-person-fill-x'
    PERSON_FILL = 'bi bi-person-fill'
    PERSON_GEAR = 'bi bi-person-gear'
    PERSON_HEART = 'bi bi-person-heart'
    PERSON_HEARTS = 'bi bi-person-hearts'
    PERSON_LINES_FILL = 'bi bi-person-lines-fill'
    PERSON_LOCK = 'bi bi-person-lock'
    PERSON_PLUS_FILL = 'bi bi-person-plus-fill'
    PERSON_PLUS = 'bi bi-person-plus'
    PERSON_RAISED_HAND = 'bi bi-person-raised-hand'
    PERSON_ROLODEX = 'bi bi-person-rolodex'
    PERSON_SLASH = 'bi bi-person-slash'
    PERSON_SQUARE = 'bi bi-person-square'
    PERSON_STANDING_DRESS = 'bi bi-person-standing-dress'
    PERSON_STANDING = 'bi bi-person-standing'
    PERSON_UP = 'bi bi-person-up'
    PERSON_VCARD_FILL = 'bi bi-person-vcard-fill'
    PERSON_VCARD = 'bi bi-person-vcard'
    PERSON_VIDEO = 'bi bi-person-video'
    PERSON_VIDEO2 = 'bi bi-person-video2'
    PERSON_VIDEO3 = 'bi bi-person-video3'
    PERSON_WALKING = 'bi bi-person-walking'
    PERSON_WHEELCHAIR = 'bi bi-person-wheelchair'
    PERSON_WORKSPACE = 'bi bi-person-workspace'
    PERSON_X_FILL = 'bi bi-person-x-fill'
    PERSON_X = 'bi bi-person-x'
    PERSON = 'bi bi-person'
    PHONE_FILL = 'bi bi-phone-fill'
    PHONE_FLIP = 'bi bi-phone-flip'
    PHONE_LANDSCAPE_FILL = 'bi bi-phone-landscape-fill'
    PHONE_LANDSCAPE = 'bi bi-phone-landscape'
    PHONE_VIBRATE_FILL = 'bi bi-phone-vibrate-fill'
    PHONE_VIBRATE = 'bi bi-phone-vibrate'
    PHONE = 'bi bi-phone'
    PIE_CHART_FILL = 'bi bi-pie-chart-fill'
    PIE_CHART = 'bi bi-pie-chart'
    PIGGY_BANK_FILL = 'bi bi-piggy-bank-fill'
    PIGGY_BANK = 'bi bi-piggy-bank'
    PIN_ANGLE_FILL = 'bi bi-pin-angle-fill'
    PIN_ANGLE = 'bi bi-pin-angle'
    PIN_FILL = 'bi bi-pin-fill'
    PIN_MAP_FILL = 'bi bi-pin-map-fill'
    PIN_MAP = 'bi bi-pin-map'
    PIN = 'bi bi-pin'
    PINTEREST = 'bi bi-pinterest'
    PIP_FILL = 'bi bi-pip-fill'
    PIP = 'bi bi-pip'
    PLAY_BTN_FILL = 'bi bi-play-btn-fill'
    PLAY_BTN = 'bi bi-play-btn'
    PLAY_CIRCLE_FILL = 'bi bi-play-circle-fill'
    PLAY_CIRCLE = 'bi bi-play-circle'
    PLAY_FILL = 'bi bi-play-fill'
    PLAY = 'bi bi-play'
    PLAYSTATION = 'bi bi-playstation'
    PLUG_FILL = 'bi bi-plug-fill'
    PLUG = 'bi bi-plug'
    PLUGIN = 'bi bi-plugin'
    PLUS_CIRCLE_DOTTED = 'bi bi-plus-circle-dotted'
    PLUS_CIRCLE_FILL = 'bi bi-plus-circle-fill'
    PLUS_CIRCLE = 'bi bi-plus-circle'
    PLUS_LG = 'bi bi-plus-lg'
    PLUS_SLASH_MINUS = 'bi bi-plus-slash-minus'
    PLUS_SQUARE_DOTTED = 'bi bi-plus-square-dotted'
    PLUS_SQUARE_FILL = 'bi bi-plus-square-fill'
    PLUS_SQUARE = 'bi bi-plus-square'
    PLUS = 'bi bi-plus'
    POSTAGE_FILL = 'bi bi-postage-fill'
    POSTAGE_HEART_FILL = 'bi bi-postage-heart-fill'
    POSTAGE_HEART = 'bi bi-postage-heart'
    POSTAGE = 'bi bi-postage'
    POSTCARD_FILL = 'bi bi-postcard-fill'
    POSTCARD_HEART_FILL = 'bi bi-postcard-heart-fill'
    POSTCARD_HEART = 'bi bi-postcard-heart'
    POSTCARD = 'bi bi-postcard'
    POWER = 'bi bi-power'
    PRESCRIPTION = 'bi bi-prescription'
    PRESCRIPTION2 = 'bi bi-prescription2'
    PRINTER_FILL = 'bi bi-printer-fill'
    PRINTER = 'bi bi-printer'
    PROJECTOR_FILL = 'bi bi-projector-fill'
    PROJECTOR = 'bi bi-projector'
    PUZZLE_FILL = 'bi bi-puzzle-fill'
    PUZZLE = 'bi bi-puzzle'
    QR_CODE_SCAN = 'bi bi-qr-code-scan'
    QR_CODE = 'bi bi-qr-code'
    QUESTION_CIRCLE_FILL = 'bi bi-question-circle-fill'
    QUESTION_CIRCLE = 'bi bi-question-circle'
    QUESTION_DIAMOND_FILL = 'bi bi-question-diamond-fill'
    QUESTION_DIAMOND = 'bi bi-question-diamond'
    QUESTION_LG = 'bi bi-question-lg'
    QUESTION_OCTAGON_FILL = 'bi bi-question-octagon-fill'
    QUESTION_OCTAGON = 'bi bi-question-octagon'
    QUESTION_SQUARE_FILL = 'bi bi-question-square-fill'
    QUESTION_SQUARE = 'bi bi-question-square'
    QUESTION = 'bi bi-question'
    QUORA = 'bi bi-quora'
    QUOTE = 'bi bi-quote'
    R_CIRCLE_FILL = 'bi bi-r-circle-fill'
    R_CIRCLE = 'bi bi-r-circle'
    R_SQUARE_FILL = 'bi bi-r-square-fill'
    R_SQUARE = 'bi bi-r-square'
    RADAR = 'bi bi-radar'
    RADIOACTIVE = 'bi bi-radioactive'
    RAINBOW = 'bi bi-rainbow'
    RECEIPT_CUTOFF = 'bi bi-receipt-cutoff'
    RECEIPT = 'bi bi-receipt'
    RECEPTION_0 = 'bi bi-reception-0'
    RECEPTION_1 = 'bi bi-reception-1'
    RECEPTION_2 = 'bi bi-reception-2'
    RECEPTION_3 = 'bi bi-reception-3'
    RECEPTION_4 = 'bi bi-reception-4'
    RECORD_BTN_FILL = 'bi bi-record-btn-fill'
    RECORD_BTN = 'bi bi-record-btn'
    RECORD_CIRCLE_FILL = 'bi bi-record-circle-fill'
    RECORD_CIRCLE = 'bi bi-record-circle'
    RECORD_FILL = 'bi bi-record-fill'
    RECORD = 'bi bi-record'
    RECORD2_FILL = 'bi bi-record2-fill'
    RECORD2 = 'bi bi-record2'
    RECYCLE = 'bi bi-recycle'
    REDDIT = 'bi bi-reddit'
    REGEX = 'bi bi-regex'
    REPEAT_1 = 'bi bi-repeat-1'
    REPEAT = 'bi bi-repeat'
    REPLY_ALL_FILL = 'bi bi-reply-all-fill'
    REPLY_ALL = 'bi bi-reply-all'
    REPLY_FILL = 'bi bi-reply-fill'
    REPLY = 'bi bi-reply'
    REWIND_BTN_FILL = 'bi bi-rewind-btn-fill'
    REWIND_BTN = 'bi bi-rewind-btn'
    REWIND_CIRCLE_FILL = 'bi bi-rewind-circle-fill'
    REWIND_CIRCLE = 'bi bi-rewind-circle'
    REWIND_FILL = 'bi bi-rewind-fill'
    REWIND = 'bi bi-rewind'
    ROBOT = 'bi bi-robot'
    ROCKET_FILL = 'bi bi-rocket-fill'
    ROCKET_TAKEOFF_FILL = 'bi bi-rocket-takeoff-fill'
    ROCKET_TAKEOFF = 'bi bi-rocket-takeoff'
    ROCKET = 'bi bi-rocket'
    ROUTER_FILL = 'bi bi-router-fill'
    ROUTER = 'bi bi-router'
    RSS_FILL = 'bi bi-rss-fill'
    RSS = 'bi bi-rss'
    RULERS = 'bi bi-rulers'
    SAFE_FILL = 'bi bi-safe-fill'
    SAFE = 'bi bi-safe'
    SAFE2_FILL = 'bi bi-safe2-fill'
    SAFE2 = 'bi bi-safe2'
    SAVE_FILL = 'bi bi-save-fill'
    SAVE = 'bi bi-save'
    SAVE2_FILL = 'bi bi-save2-fill'
    SAVE2 = 'bi bi-save2'
    SCISSORS = 'bi bi-scissors'
    SCOOTER = 'bi bi-scooter'
    SCREWDRIVER = 'bi bi-screwdriver'
    SD_CARD_FILL = 'bi bi-sd-card-fill'
    SD_CARD = 'bi bi-sd-card'
    SEARCH_HEART_FILL = 'bi bi-search-heart-fill'
    SEARCH_HEART = 'bi bi-search-heart'
    SEARCH = 'bi bi-search'
    SEGMENTED_NAV = 'bi bi-segmented-nav'
    SEND_ARROW_DOWN_FILL = 'bi bi-send-arrow-down-fill'
    SEND_ARROW_DOWN = 'bi bi-send-arrow-down'
    SEND_ARROW_UP_FILL = 'bi bi-send-arrow-up-fill'
    SEND_ARROW_UP = 'bi bi-send-arrow-up'
    SEND_CHECK_FILL = 'bi bi-send-check-fill'
    SEND_CHECK = 'bi bi-send-check'
    SEND_DASH_FILL = 'bi bi-send-dash-fill'
    SEND_DASH = 'bi bi-send-dash'
    SEND_EXCLAMATION_FILL = 'bi bi-send-exclamation-fill'
    SEND_EXCLAMATION = 'bi bi-send-exclamation'
    SEND_FILL = 'bi bi-send-fill'
    SEND_PLUS_FILL = 'bi bi-send-plus-fill'
    SEND_PLUS = 'bi bi-send-plus'
    SEND_SLASH_FILL = 'bi bi-send-slash-fill'
    SEND_SLASH = 'bi bi-send-slash'
    SEND_X_FILL = 'bi bi-send-x-fill'
    SEND_X = 'bi bi-send-x'
    SEND = 'bi bi-send'
    SERVER = 'bi bi-server'
    SHADOWS = 'bi bi-shadows'
    SHARE_FILL = 'bi bi-share-fill'
    SHARE = 'bi bi-share'
    SHIELD_CHECK = 'bi bi-shield-check'
    SHIELD_EXCLAMATION = 'bi bi-shield-exclamation'
    SHIELD_FILL_CHECK = 'bi bi-shield-fill-check'
    SHIELD_FILL_EXCLAMATION = 'bi bi-shield-fill-exclamation'
    SHIELD_FILL_MINUS = 'bi bi-shield-fill-minus'
    SHIELD_FILL_PLUS = 'bi bi-shield-fill-plus'
    SHIELD_FILL_X = 'bi bi-shield-fill-x'
    SHIELD_FILL = 'bi bi-shield-fill'
    SHIELD_LOCK_FILL = 'bi bi-shield-lock-fill'
    SHIELD_LOCK = 'bi bi-shield-lock'
    SHIELD_MINUS = 'bi bi-shield-minus'
    SHIELD_PLUS = 'bi bi-shield-plus'
    SHIELD_SHADED = 'bi bi-shield-shaded'
    SHIELD_SLASH_FILL = 'bi bi-shield-slash-fill'
    SHIELD_SLASH = 'bi bi-shield-slash'
    SHIELD_X = 'bi bi-shield-x'
    SHIELD = 'bi bi-shield'
    SHIFT_FILL = 'bi bi-shift-fill'
    SHIFT = 'bi bi-shift'
    SHOP_WINDOW = 'bi bi-shop-window'
    SHOP = 'bi bi-shop'
    SHUFFLE = 'bi bi-shuffle'
    SIGN_DEAD_END_FILL = 'bi bi-sign-dead-end-fill'
    SIGN_DEAD_END = 'bi bi-sign-dead-end'
    SIGN_DO_NOT_ENTER_FILL = 'bi bi-sign-do-not-enter-fill'
    SIGN_DO_NOT_ENTER = 'bi bi-sign-do-not-enter'
    SIGN_INTERSECTION_FILL = 'bi bi-sign-intersection-fill'
    SIGN_INTERSECTION_SIDE_FILL = 'bi bi-sign-intersection-side-fill'
    SIGN_INTERSECTION_SIDE = 'bi bi-sign-intersection-side'
    SIGN_INTERSECTION_T_FILL = 'bi bi-sign-intersection-t-fill'
    SIGN_INTERSECTION_T = 'bi bi-sign-intersection-t'
    SIGN_INTERSECTION_Y_FILL = 'bi bi-sign-intersection-y-fill'
    SIGN_INTERSECTION_Y = 'bi bi-sign-intersection-y'
    SIGN_INTERSECTION = 'bi bi-sign-intersection'
    SIGN_MERGE_LEFT_FILL = 'bi bi-sign-merge-left-fill'
    SIGN_MERGE_LEFT = 'bi bi-sign-merge-left'
    SIGN_MERGE_RIGHT_FILL = 'bi bi-sign-merge-right-fill'
    SIGN_MERGE_RIGHT = 'bi bi-sign-merge-right'
    SIGN_NO_LEFT_TURN_FILL = 'bi bi-sign-no-left-turn-fill'
    SIGN_NO_LEFT_TURN = 'bi bi-sign-no-left-turn'
    SIGN_NO_PARKING_FILL = 'bi bi-sign-no-parking-fill'
    SIGN_NO_PARKING = 'bi bi-sign-no-parking'
    SIGN_NO_RIGHT_TURN_FILL = 'bi bi-sign-no-right-turn-fill'
    SIGN_NO_RIGHT_TURN = 'bi bi-sign-no-right-turn'
    SIGN_RAILROAD_FILL = 'bi bi-sign-railroad-fill'
    SIGN_RAILROAD = 'bi bi-sign-railroad'
    SIGN_STOP_FILL = 'bi bi-sign-stop-fill'
    SIGN_STOP_LIGHTS_FILL = 'bi bi-sign-stop-lights-fill'
    SIGN_STOP_LIGHTS = 'bi bi-sign-stop-lights'
    SIGN_STOP = 'bi bi-sign-stop'
    SIGN_TURN_LEFT_FILL = 'bi bi-sign-turn-left-fill'
    SIGN_TURN_LEFT = 'bi bi-sign-turn-left'
    SIGN_TURN_RIGHT_FILL = 'bi bi-sign-turn-right-fill'
    SIGN_TURN_RIGHT = 'bi bi-sign-turn-right'
    SIGN_TURN_SLIGHT_LEFT_FILL = 'bi bi-sign-turn-slight-left-fill'
    SIGN_TURN_SLIGHT_LEFT = 'bi bi-sign-turn-slight-left'
    SIGN_TURN_SLIGHT_RIGHT_FILL = 'bi bi-sign-turn-slight-right-fill'
    SIGN_TURN_SLIGHT_RIGHT = 'bi bi-sign-turn-slight-right'
    SIGN_YIELD_FILL = 'bi bi-sign-yield-fill'
    SIGN_YIELD = 'bi bi-sign-yield'
    SIGNAL = 'bi bi-signal'
    SIGNPOST_2_FILL = 'bi bi-signpost-2-fill'
    SIGNPOST_2 = 'bi bi-signpost-2'
    SIGNPOST_FILL = 'bi bi-signpost-fill'
    SIGNPOST_SPLIT_FILL = 'bi bi-signpost-split-fill'
    SIGNPOST_SPLIT = 'bi bi-signpost-split'
    SIGNPOST = 'bi bi-signpost'
    SIM_FILL = 'bi bi-sim-fill'
    SIM_SLASH_FILL = 'bi bi-sim-slash-fill'
    SIM_SLASH = 'bi bi-sim-slash'
    SIM = 'bi bi-sim'
    SINA_WEIBO = 'bi bi-sina-weibo'
    SKIP_BACKWARD_BTN_FILL = 'bi bi-skip-backward-btn-fill'
    SKIP_BACKWARD_BTN = 'bi bi-skip-backward-btn'
    SKIP_BACKWARD_CIRCLE_FILL = 'bi bi-skip-backward-circle-fill'
    SKIP_BACKWARD_CIRCLE = 'bi bi-skip-backward-circle'
    SKIP_BACKWARD_FILL = 'bi bi-skip-backward-fill'
    SKIP_BACKWARD = 'bi bi-skip-backward'
    SKIP_END_BTN_FILL = 'bi bi-skip-end-btn-fill'
    SKIP_END_BTN = 'bi bi-skip-end-btn'
    SKIP_END_CIRCLE_FILL = 'bi bi-skip-end-circle-fill'
    SKIP_END_CIRCLE = 'bi bi-skip-end-circle'
    SKIP_END_FILL = 'bi bi-skip-end-fill'
    SKIP_END = 'bi bi-skip-end'
    SKIP_FORWARD_BTN_FILL = 'bi bi-skip-forward-btn-fill'
    SKIP_FORWARD_BTN = 'bi bi-skip-forward-btn'
    SKIP_FORWARD_CIRCLE_FILL = 'bi bi-skip-forward-circle-fill'
    SKIP_FORWARD_CIRCLE = 'bi bi-skip-forward-circle'
    SKIP_FORWARD_FILL = 'bi bi-skip-forward-fill'
    SKIP_FORWARD = 'bi bi-skip-forward'
    SKIP_START_BTN_FILL = 'bi bi-skip-start-btn-fill'
    SKIP_START_BTN = 'bi bi-skip-start-btn'
    SKIP_START_CIRCLE_FILL = 'bi bi-skip-start-circle-fill'
    SKIP_START_CIRCLE = 'bi bi-skip-start-circle'
    SKIP_START_FILL = 'bi bi-skip-start-fill'
    SKIP_START = 'bi bi-skip-start'
    SKYPE = 'bi bi-skype'
    SLACK = 'bi bi-slack'
    SLASH_CIRCLE_FILL = 'bi bi-slash-circle-fill'
    SLASH_CIRCLE = 'bi bi-slash-circle'
    SLASH_LG = 'bi bi-slash-lg'
    SLASH_SQUARE_FILL = 'bi bi-slash-square-fill'
    SLASH_SQUARE = 'bi bi-slash-square'
    SLASH = 'bi bi-slash'
    SLIDERS = 'bi bi-sliders'
    SLIDERS2_VERTICAL = 'bi bi-sliders2-vertical'
    SLIDERS2 = 'bi bi-sliders2'
    SMARTWATCH = 'bi bi-smartwatch'
    SNAPCHAT = 'bi bi-snapchat'
    SNOW = 'bi bi-snow'
    SNOW2 = 'bi bi-snow2'
    SNOW3 = 'bi bi-snow3'
    SORT_ALPHA_DOWN_ALT = 'bi bi-sort-alpha-down-alt'
    SORT_ALPHA_DOWN = 'bi bi-sort-alpha-down'
    SORT_ALPHA_UP_ALT = 'bi bi-sort-alpha-up-alt'
    SORT_ALPHA_UP = 'bi bi-sort-alpha-up'
    SORT_DOWN_ALT = 'bi bi-sort-down-alt'
    SORT_DOWN = 'bi bi-sort-down'
    SORT_NUMERIC_DOWN_ALT = 'bi bi-sort-numeric-down-alt'
    SORT_NUMERIC_DOWN = 'bi bi-sort-numeric-down'
    SORT_NUMERIC_UP_ALT = 'bi bi-sort-numeric-up-alt'
    SORT_NUMERIC_UP = 'bi bi-sort-numeric-up'
    SORT_UP_ALT = 'bi bi-sort-up-alt'
    SORT_UP = 'bi bi-sort-up'
    SOUNDWAVE = 'bi bi-soundwave'
    SOURCEFORGE = 'bi bi-sourceforge'
    SPEAKER_FILL = 'bi bi-speaker-fill'
    SPEAKER = 'bi bi-speaker'
    SPEEDOMETER = 'bi bi-speedometer'
    SPEEDOMETER2 = 'bi bi-speedometer2'
    SPELLCHECK = 'bi bi-spellcheck'
    SPOTIFY = 'bi bi-spotify'
    SQUARE_FILL = 'bi bi-square-fill'
    SQUARE_HALF = 'bi bi-square-half'
    SQUARE = 'bi bi-square'
    STACK_OVERFLOW = 'bi bi-stack-overflow'
    STACK = 'bi bi-stack'
    STAR_FILL = 'bi bi-star-fill'
    STAR_HALF = 'bi bi-star-half'
    STAR = 'bi bi-star'
    STARS = 'bi bi-stars'
    STEAM = 'bi bi-steam'
    STICKIES_FILL = 'bi bi-stickies-fill'
    STICKIES = 'bi bi-stickies'
    STICKY_FILL = 'bi bi-sticky-fill'
    STICKY = 'bi bi-sticky'
    STOP_BTN_FILL = 'bi bi-stop-btn-fill'
    STOP_BTN = 'bi bi-stop-btn'
    STOP_CIRCLE_FILL = 'bi bi-stop-circle-fill'
    STOP_CIRCLE = 'bi bi-stop-circle'
    STOP_FILL = 'bi bi-stop-fill'
    STOP = 'bi bi-stop'
    STOPLIGHTS_FILL = 'bi bi-stoplights-fill'
    STOPLIGHTS = 'bi bi-stoplights'
    STOPWATCH_FILL = 'bi bi-stopwatch-fill'
    STOPWATCH = 'bi bi-stopwatch'
    STRAVA = 'bi bi-strava'
    STRIPE = 'bi bi-stripe'
    SUBSCRIPT = 'bi bi-subscript'
    SUBSTACK = 'bi bi-substack'
    SUBTRACT = 'bi bi-subtract'
    SUIT_CLUB_FILL = 'bi bi-suit-club-fill'
    SUIT_CLUB = 'bi bi-suit-club'
    SUIT_DIAMOND_FILL = 'bi bi-suit-diamond-fill'
    SUIT_DIAMOND = 'bi bi-suit-diamond'
    SUIT_HEART_FILL = 'bi bi-suit-heart-fill'
    SUIT_HEART = 'bi bi-suit-heart'
    SUIT_SPADE_FILL = 'bi bi-suit-spade-fill'
    SUIT_SPADE = 'bi bi-suit-spade'
    SUITCASE_FILL = 'bi bi-suitcase-fill'
    SUITCASE_LG_FILL = 'bi bi-suitcase-lg-fill'
    SUITCASE_LG = 'bi bi-suitcase-lg'
    SUITCASE = 'bi bi-suitcase'
    SUITCASE2_FILL = 'bi bi-suitcase2-fill'
    SUITCASE2 = 'bi bi-suitcase2'
    SUN_FILL = 'bi bi-sun-fill'
    SUN = 'bi bi-sun'
    SUNGLASSES = 'bi bi-sunglasses'
    SUNRISE_FILL = 'bi bi-sunrise-fill'
    SUNRISE = 'bi bi-sunrise'
    SUNSET_FILL = 'bi bi-sunset-fill'
    SUNSET = 'bi bi-sunset'
    SUPERSCRIPT = 'bi bi-superscript'
    SYMMETRY_HORIZONTAL = 'bi bi-symmetry-horizontal'
    SYMMETRY_VERTICAL = 'bi bi-symmetry-vertical'
    TABLE = 'bi bi-table'
    TABLET_FILL = 'bi bi-tablet-fill'
    TABLET_LANDSCAPE_FILL = 'bi bi-tablet-landscape-fill'
    TABLET_LANDSCAPE = 'bi bi-tablet-landscape'
    TABLET = 'bi bi-tablet'
    TAG_FILL = 'bi bi-tag-fill'
    TAG = 'bi bi-tag'
    TAGS_FILL = 'bi bi-tags-fill'
    TAGS = 'bi bi-tags'
    TAXI_FRONT_FILL = 'bi bi-taxi-front-fill'
    TAXI_FRONT = 'bi bi-taxi-front'
    TELEGRAM = 'bi bi-telegram'
    TELEPHONE_FILL = 'bi bi-telephone-fill'
    TELEPHONE_FORWARD_FILL = 'bi bi-telephone-forward-fill'
    TELEPHONE_FORWARD = 'bi bi-telephone-forward'
    TELEPHONE_INBOUND_FILL = 'bi bi-telephone-inbound-fill'
    TELEPHONE_INBOUND = 'bi bi-telephone-inbound'
    TELEPHONE_MINUS_FILL = 'bi bi-telephone-minus-fill'
    TELEPHONE_MINUS = 'bi bi-telephone-minus'
    TELEPHONE_OUTBOUND_FILL = 'bi bi-telephone-outbound-fill'
    TELEPHONE_OUTBOUND = 'bi bi-telephone-outbound'
    TELEPHONE_PLUS_FILL = 'bi bi-telephone-plus-fill'
    TELEPHONE_PLUS = 'bi bi-telephone-plus'
    TELEPHONE_X_FILL = 'bi bi-telephone-x-fill'
    TELEPHONE_X = 'bi bi-telephone-x'
    TELEPHONE = 'bi bi-telephone'
    TENCENT_QQ = 'bi bi-tencent-qq'
    TERMINAL_DASH = 'bi bi-terminal-dash'
    TERMINAL_FILL = 'bi bi-terminal-fill'
    TERMINAL_PLUS = 'bi bi-terminal-plus'
    TERMINAL_SPLIT = 'bi bi-terminal-split'
    TERMINAL_X = 'bi bi-terminal-x'
    TERMINAL = 'bi bi-terminal'
    TEXT_CENTER = 'bi bi-text-center'
    TEXT_INDENT_LEFT = 'bi bi-text-indent-left'
    TEXT_INDENT_RIGHT = 'bi bi-text-indent-right'
    TEXT_LEFT = 'bi bi-text-left'
    TEXT_PARAGRAPH = 'bi bi-text-paragraph'
    TEXT_RIGHT = 'bi bi-text-right'
    TEXT_WRAP = 'bi bi-text-wrap'
    TEXTAREA_RESIZE = 'bi bi-textarea-resize'
    TEXTAREA_T = 'bi bi-textarea-t'
    TEXTAREA = 'bi bi-textarea'
    THERMOMETER_HALF = 'bi bi-thermometer-half'
    THERMOMETER_HIGH = 'bi bi-thermometer-high'
    THERMOMETER_LOW = 'bi bi-thermometer-low'
    THERMOMETER_SNOW = 'bi bi-thermometer-snow'
    THERMOMETER_SUN = 'bi bi-thermometer-sun'
    THERMOMETER = 'bi bi-thermometer'
    THREADS_FILL = 'bi bi-threads-fill'
    THREADS = 'bi bi-threads'
    THREE_DOTS_VERTICAL = 'bi bi-three-dots-vertical'
    THREE_DOTS = 'bi bi-three-dots'
    THUNDERBOLT_FILL = 'bi bi-thunderbolt-fill'
    THUNDERBOLT = 'bi bi-thunderbolt'
    TICKET_DETAILED_FILL = 'bi bi-ticket-detailed-fill'
    TICKET_DETAILED = 'bi bi-ticket-detailed'
    TICKET_FILL = 'bi bi-ticket-fill'
    TICKET_PERFORATED_FILL = 'bi bi-ticket-perforated-fill'
    TICKET_PERFORATED = 'bi bi-ticket-perforated'
    TICKET = 'bi bi-ticket'
    TIKTOK = 'bi bi-tiktok'
    TOGGLE_OFF = 'bi bi-toggle-off'
    TOGGLE_ON = 'bi bi-toggle-on'
    TOGGLE2_OFF = 'bi bi-toggle2-off'
    TOGGLE2_ON = 'bi bi-toggle2-on'
    TOGGLES = 'bi bi-toggles'
    TOGGLES2 = 'bi bi-toggles2'
    TOOLS = 'bi bi-tools'
    TORNADO = 'bi bi-tornado'
    TRAIN_FREIGHT_FRONT_FILL = 'bi bi-train-freight-front-fill'
    TRAIN_FREIGHT_FRONT = 'bi bi-train-freight-front'
    TRAIN_FRONT_FILL = 'bi bi-train-front-fill'
    TRAIN_FRONT = 'bi bi-train-front'
    TRAIN_LIGHTRAIL_FRONT_FILL = 'bi bi-train-lightrail-front-fill'
    TRAIN_LIGHTRAIL_FRONT = 'bi bi-train-lightrail-front'
    TRANSLATE = 'bi bi-translate'
    TRANSPARENCY = 'bi bi-transparency'
    TRASH_FILL = 'bi bi-trash-fill'
    TRASH = 'bi bi-trash'
    TRASH2_FILL = 'bi bi-trash2-fill'
    TRASH2 = 'bi bi-trash2'
    TRASH3_FILL = 'bi bi-trash3-fill'
    TRASH3 = 'bi bi-trash3'
    TREE_FILL = 'bi bi-tree-fill'
    TREE = 'bi bi-tree'
    TRELLO = 'bi bi-trello'
    TRIANGLE_FILL = 'bi bi-triangle-fill'
    TRIANGLE_HALF = 'bi bi-triangle-half'
    TRIANGLE = 'bi bi-triangle'
    TROPHY_FILL = 'bi bi-trophy-fill'
    TROPHY = 'bi bi-trophy'
    TROPICAL_STORM = 'bi bi-tropical-storm'
    TRUCK_FLATBED = 'bi bi-truck-flatbed'
    TRUCK_FRONT_FILL = 'bi bi-truck-front-fill'
    TRUCK_FRONT = 'bi bi-truck-front'
    TRUCK = 'bi bi-truck'
    TSUNAMI = 'bi bi-tsunami'
    TV_FILL = 'bi bi-tv-fill'
    TV = 'bi bi-tv'
    TWITCH = 'bi bi-twitch'
    TWITTER_X = 'bi bi-twitter-x'
    TWITTER = 'bi bi-twitter'
    TYPE_BOLD = 'bi bi-type-bold'
    TYPE_H1 = 'bi bi-type-h1'
    TYPE_H2 = 'bi bi-type-h2'
    TYPE_H3 = 'bi bi-type-h3'
    TYPE_H4 = 'bi bi-type-h4'
    TYPE_H5 = 'bi bi-type-h5'
    TYPE_H6 = 'bi bi-type-h6'
    TYPE_ITALIC = 'bi bi-type-italic'
    TYPE_STRIKETHROUGH = 'bi bi-type-strikethrough'
    TYPE_UNDERLINE = 'bi bi-type-underline'
    TYPE = 'bi bi-type'
    UBUNTU = 'bi bi-ubuntu'
    UI_CHECKS_GRID = 'bi bi-ui-checks-grid'
    UI_CHECKS = 'bi bi-ui-checks'
    UI_RADIOS_GRID = 'bi bi-ui-radios-grid'
    UI_RADIOS = 'bi bi-ui-radios'
    UMBRELLA_FILL = 'bi bi-umbrella-fill'
    UMBRELLA = 'bi bi-umbrella'
    UNINDENT = 'bi bi-unindent'
    UNION = 'bi bi-union'
    UNITY = 'bi bi-unity'
    UNIVERSAL_ACCESS_CIRCLE = 'bi bi-universal-access-circle'
    UNIVERSAL_ACCESS = 'bi bi-universal-access'
    UNLOCK_FILL = 'bi bi-unlock-fill'
    UNLOCK = 'bi bi-unlock'
    UPC_SCAN = 'bi bi-upc-scan'
    UPC = 'bi bi-upc'
    UPLOAD = 'bi bi-upload'
    USB_C_FILL = 'bi bi-usb-c-fill'
    USB_C = 'bi bi-usb-c'
    USB_DRIVE_FILL = 'bi bi-usb-drive-fill'
    USB_DRIVE = 'bi bi-usb-drive'
    USB_FILL = 'bi bi-usb-fill'
    USB_MICRO_FILL = 'bi bi-usb-micro-fill'
    USB_MICRO = 'bi bi-usb-micro'
    USB_MINI_FILL = 'bi bi-usb-mini-fill'
    USB_MINI = 'bi bi-usb-mini'
    USB_PLUG_FILL = 'bi bi-usb-plug-fill'
    USB_PLUG = 'bi bi-usb-plug'
    USB_SYMBOL = 'bi bi-usb-symbol'
    USB = 'bi bi-usb'
    VALENTINE = 'bi bi-valentine'
    VALENTINE2 = 'bi bi-valentine2'
    VECTOR_PEN = 'bi bi-vector-pen'
    VIEW_LIST = 'bi bi-view-list'
    VIEW_STACKED = 'bi bi-view-stacked'
    VIGNETTE = 'bi bi-vignette'
    VIMEO = 'bi bi-vimeo'
    VINYL_FILL = 'bi bi-vinyl-fill'
    VINYL = 'bi bi-vinyl'
    VIRUS = 'bi bi-virus'
    VIRUS2 = 'bi bi-virus2'
    VOICEMAIL = 'bi bi-voicemail'
    VOLUME_DOWN_FILL = 'bi bi-volume-down-fill'
    VOLUME_DOWN = 'bi bi-volume-down'
    VOLUME_MUTE_FILL = 'bi bi-volume-mute-fill'
    VOLUME_MUTE = 'bi bi-volume-mute'
    VOLUME_OFF_FILL = 'bi bi-volume-off-fill'
    VOLUME_OFF = 'bi bi-volume-off'
    VOLUME_UP_FILL = 'bi bi-volume-up-fill'
    VOLUME_UP = 'bi bi-volume-up'
    VR = 'bi bi-vr'
    WALLET_FILL = 'bi bi-wallet-fill'
    WALLET = 'bi bi-wallet'
    WALLET2 = 'bi bi-wallet2'
    WATCH = 'bi bi-watch'
    WATER = 'bi bi-water'
    WEBCAM_FILL = 'bi bi-webcam-fill'
    WEBCAM = 'bi bi-webcam'
    WECHAT = 'bi bi-wechat'
    WHATSAPP = 'bi bi-whatsapp'
    WIFI_1 = 'bi bi-wifi-1'
    WIFI_2 = 'bi bi-wifi-2'
    WIFI_OFF = 'bi bi-wifi-off'
    WIFI = 'bi bi-wifi'
    WIKIPEDIA = 'bi bi-wikipedia'
    WIND = 'bi bi-wind'
    WINDOW_DASH = 'bi bi-window-dash'
    WINDOW_DESKTOP = 'bi bi-window-desktop'
    WINDOW_DOCK = 'bi bi-window-dock'
    WINDOW_FULLSCREEN = 'bi bi-window-fullscreen'
    WINDOW_PLUS = 'bi bi-window-plus'
    WINDOW_SIDEBAR = 'bi bi-window-sidebar'
    WINDOW_SPLIT = 'bi bi-window-split'
    WINDOW_STACK = 'bi bi-window-stack'
    WINDOW_X = 'bi bi-window-x'
    WINDOW = 'bi bi-window'
    WINDOWS = 'bi bi-windows'
    WORDPRESS = 'bi bi-wordpress'
    WRENCH_ADJUSTABLE_CIRCLE_FILL = 'bi bi-wrench-adjustable-circle-fill'
    WRENCH_ADJUSTABLE_CIRCLE = 'bi bi-wrench-adjustable-circle'
    WRENCH_ADJUSTABLE = 'bi bi-wrench-adjustable'
    WRENCH = 'bi bi-wrench'
    X_CIRCLE_FILL = 'bi bi-x-circle-fill'
    X_CIRCLE = 'bi bi-x-circle'
    X_DIAMOND_FILL = 'bi bi-x-diamond-fill'
    X_DIAMOND = 'bi bi-x-diamond'
    X_LG = 'bi bi-x-lg'
    X_OCTAGON_FILL = 'bi bi-x-octagon-fill'
    X_OCTAGON = 'bi bi-x-octagon'
    X_SQUARE_FILL = 'bi bi-x-square-fill'
    X_SQUARE = 'bi bi-x-square'
    X = 'bi bi-x'
    XBOX = 'bi bi-xbox'
    YELP = 'bi bi-yelp'
    YIN_YANG = 'bi bi-yin-yang'
    YOUTUBE = 'bi bi-youtube'
    ZOOM_IN = 'bi bi-zoom-in'
    ZOOM_OUT = 'bi bi-zoom-out'