import { CloudAssembly } from './cloud-assembly';
import { Environment } from './environment';
import { MetadataEntry, MetadataEntryResult, SynthesisMessage } from './metadata';
/**
 * Type of cloud artifact.
 */
export declare enum ArtifactType {
    NONE = "none",
    /**
     * The artifact is an AWS CloudFormation stack.
     */
    AWS_CLOUDFORMATION_STACK = "aws:cloudformation:stack"
}
/**
 * A manifest for a single artifact within the cloud assembly.
 */
export interface ArtifactManifest {
    /**
     * The type of artifact.
     */
    readonly type: ArtifactType;
    /**
     * The environment into which this artifact is deployed.
     */
    readonly environment: string;
    /**
     * Associated metadata.
     */
    readonly metadata?: {
        [path: string]: MetadataEntry[];
    };
    /**
     * IDs of artifacts that must be deployed before this artifact.
     */
    readonly dependencies?: string[];
    /**
     * The set of properties for this artifact (depends on type)
     */
    readonly properties?: {
        [name: string]: any;
    };
}
/**
 * Artifact properties for CloudFormation stacks.
 */
export interface AwsCloudFormationStackProperties {
    /**
     * A file relative to the assembly root which contains the CloudFormation template for this stack.
     */
    readonly templateFile: string;
    /**
     * Values for CloudFormation stack parameters that should be passed when the stack is deployed.
     */
    readonly parameters?: {
        [id: string]: string;
    };
}
/**
 * Represents an artifact within a cloud assembly.
 */
export declare class CloudArtifact {
    readonly assembly: CloudAssembly;
    readonly id: string;
    /**
     * Returns a subclass of `CloudArtifact` based on the artifact type defined in the artifact manifest.
     * @param assembly The cloud assembly from which to load the artifact
     * @param id The artifact ID
     * @param artifact The artifact manifest
     */
    static from(assembly: CloudAssembly, id: string, artifact: ArtifactManifest): CloudArtifact;
    /**
     * The artifact's manifest
     */
    readonly manifest: ArtifactManifest;
    /**
     * The set of messages extracted from the artifact's metadata.
     */
    readonly messages: SynthesisMessage[];
    /**
     * The environment into which to deploy this artifact.
     */
    readonly environment: Environment;
    /**
     * Cache of resolved dependencies.
     */
    private _deps?;
    protected constructor(assembly: CloudAssembly, id: string, manifest: ArtifactManifest);
    /**
     * Returns all the artifacts that this artifact depends on.
     */
    readonly dependencies: CloudArtifact[];
    /**
     * @returns all the metadata entries of a specific type in this artifact.
     * @param type
     */
    findMetadataByType(type: string): MetadataEntryResult[];
    private renderMessages;
}
