"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const environment_1 = require("./environment");
const metadata_1 = require("./metadata");
/**
 * Type of cloud artifact.
 */
var ArtifactType;
(function (ArtifactType) {
    ArtifactType["NONE"] = "none";
    /**
     * The artifact is an AWS CloudFormation stack.
     */
    ArtifactType["AWS_CLOUDFORMATION_STACK"] = "aws:cloudformation:stack";
})(ArtifactType = exports.ArtifactType || (exports.ArtifactType = {}));
/**
 * Represents an artifact within a cloud assembly.
 */
class CloudArtifact {
    constructor(assembly, id, manifest) {
        this.assembly = assembly;
        this.id = id;
        this.manifest = manifest;
        this.environment = environment_1.EnvironmentUtils.parse(manifest.environment);
        this.messages = this.renderMessages();
        this._dependencyIDs = manifest.dependencies || [];
    }
    /**
     * Returns a subclass of `CloudArtifact` based on the artifact type defined in the artifact manifest.
     * @param assembly The cloud assembly from which to load the artifact
     * @param id The artifact ID
     * @param artifact The artifact manifest
     */
    static from(assembly, id, artifact) {
        switch (artifact.type) {
            case ArtifactType.AWS_CLOUDFORMATION_STACK:
                return new cloudformation_artifact_1.CloudFormationStackArtifact(assembly, id, artifact);
            default:
                throw new Error(`unsupported artifact type: ${artifact.type}`);
        }
    }
    /**
     * Returns all the artifacts that this artifact depends on.
     */
    get dependencies() {
        if (this._deps) {
            return this._deps;
        }
        this._deps = this._dependencyIDs.map(id => {
            const dep = this.assembly.artifacts.find(a => a.id === id);
            if (!dep) {
                throw new Error(`Artifact ${this.id} depends on non-existing artifact ${id}`);
            }
            return dep;
        });
        return this._deps;
    }
    /**
     * @returns all the metadata entries of a specific type in this artifact.
     * @param type
     */
    findMetadataByType(type) {
        const result = new Array();
        for (const path of Object.keys(this.manifest.metadata || {})) {
            for (const entry of (this.manifest.metadata || {})[path]) {
                if (entry.type === type) {
                    result.push({ path, ...entry });
                }
            }
        }
        return result;
    }
    renderMessages() {
        const messages = new Array();
        for (const [id, metadata] of Object.entries(this.manifest.metadata || {})) {
            for (const entry of metadata) {
                let level;
                switch (entry.type) {
                    case metadata_1.WARNING_METADATA_KEY:
                        level = metadata_1.SynthesisMessageLevel.WARNING;
                        break;
                    case metadata_1.ERROR_METADATA_KEY:
                        level = metadata_1.SynthesisMessageLevel.ERROR;
                        break;
                    case metadata_1.INFO_METADATA_KEY:
                        level = metadata_1.SynthesisMessageLevel.INFO;
                        break;
                    default:
                        continue;
                }
                messages.push({ level, entry, id });
            }
        }
        return messages;
    }
}
exports.CloudArtifact = CloudArtifact;
// needs to be defined at the end to avoid a cyclic dependency
const cloudformation_artifact_1 = require("./cloudformation-artifact");
//# sourceMappingURL=data:application/json;base64,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