import { ArtifactManifest, CloudArtifact } from './cloud-artifact';
import { CloudFormationStackArtifact } from './cloudformation-artifact';
/**
 * A manifest which describes the cloud assembly.
 */
export interface AssemblyManifest {
    /**
     * Protocol version
     */
    readonly version: string;
    /**
     * The set of artifacts in this assembly.
     */
    readonly artifacts?: {
        [id: string]: ArtifactManifest;
    };
    /**
     * Missing context information. If this field has values, it means that the
     * cloud assembly is not complete and should not be deployed.
     */
    readonly missing?: MissingContext[];
    /**
     * Runtime information.
     */
    readonly runtime?: RuntimeInfo;
}
/**
 * Represents a deployable cloud application.
 */
export declare class CloudAssembly {
    /**
     * The root directory of the cloud assembly.
     */
    readonly directory: string;
    /**
     * The schema version of the assembly manifest.
     */
    readonly version: string;
    /**
     * All artifacts included in this assembly.
     */
    readonly artifacts: CloudArtifact[];
    /**
     * Runtime information such as module versions used to synthesize this assembly.
     */
    readonly runtime: RuntimeInfo;
    /**
     * The raw assembly manifest.
     */
    readonly manifest: AssemblyManifest;
    /**
     * Reads a cloud assembly from the specified directory.
     * @param directory The root directory of the assembly.
     */
    constructor(directory: string);
    /**
     * Attempts to find an artifact with a specific identity.
     * @returns A `CloudArtifact` object or `undefined` if the artifact does not exist in this assembly.
     * @param id The artifact ID
     */
    tryGetArtifact(id: string): CloudArtifact | undefined;
    /**
     * Returns a CloudFormation stack artifact from this assembly.
     * @param stackName the name of the CloudFormation stack.
     * @throws if there is no stack artifact by that name
     * @returns a `CloudFormationStackArtifact` object.
     */
    getStack(stackName: string): CloudFormationStackArtifact;
    /**
     * @returns all the CloudFormation stack artifacts that are included in this assembly.
     */
    readonly stacks: CloudFormationStackArtifact[];
    private validateDeps;
    private renderArtifacts;
}
/**
 * Can be used to build a cloud assembly.
 */
export declare class CloudAssemblyBuilder {
    /**
     * The root directory of the resulting cloud assembly.
     */
    readonly outdir: string;
    private readonly artifacts;
    private readonly missing;
    /**
     * Initializes a cloud assembly builder.
     * @param outdir The output directory, uses temporary directory if undefined
     */
    constructor(outdir?: string);
    /**
     * Adds an artifact into the cloud assembly.
     * @param id The ID of the artifact.
     * @param manifest The artifact manifest
     */
    addArtifact(id: string, manifest: ArtifactManifest): void;
    /**
     * Reports that some context is missing in order for this cloud assembly to be fully synthesized.
     * @param missing Missing context information.
     */
    addMissing(missing: MissingContext): void;
    /**
     * Finalizes the cloud assembly into the output directory returns a
     * `CloudAssembly` object that can be used to inspect the assembly.
     * @param options
     */
    build(options?: AssemblyBuildOptions): CloudAssembly;
}
export interface AssemblyBuildOptions {
    /**
     * Include the specified runtime information (module versions) in manifest.
     * @default - if this option is not specified, runtime info will not be included
     */
    readonly runtimeInfo?: RuntimeInfo;
}
/**
 * Information about the application's runtime components.
 */
export interface RuntimeInfo {
    /**
     * The list of libraries loaded in the application, associated with their versions.
     */
    readonly libraries: {
        [name: string]: string;
    };
}
/**
 * Represents a missing piece of context.
 */
export interface MissingContext {
    /**
     * The missing context key.
     */
    readonly key: string;
    /**
     * The provider from which we expect this context key to be obtained.
     */
    readonly provider: string;
    /**
     * A set of provider-specific options.
     */
    readonly props: {
        account?: string;
        region?: string;
        [key: string]: any;
    };
}
