"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * Return a topological sort of all elements of xs, according to the given dependency functions
 *
 * Dependencies outside the referenced set are ignored.
 *
 * Not a stable sort, but in order to keep the order as stable as possible, we'll sort by key
 * among elements of equal precedence.
 */
function topologicalSort(xs, keyFn, depFn) {
    const remaining = new Map();
    for (const element of xs) {
        const key = keyFn(element);
        remaining.set(key, { key, element, dependencies: depFn(element) });
    }
    const ret = new Array();
    while (remaining.size > 0) {
        // All elements with no more deps in the set can be ordered
        const selectable = Array.from(remaining.values()).filter(e => e.dependencies.every(d => !remaining.has(d)));
        selectable.sort((a, b) => a.key < b.key ? -1 : b.key < a.key ? 1 : 0);
        for (const selected of selectable) {
            ret.push(selected.element);
            remaining.delete(selected.key);
        }
        // If we didn't make any progress, we got stuck
        if (selectable.length === 0) {
            throw new Error(`Could not determine ordering between: ${Array.from(remaining.keys()).join(', ')}`);
        }
    }
    return ret;
}
exports.topologicalSort = topologicalSort;
//# sourceMappingURL=data:application/json;base64,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