"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const semver = require("semver");
/**
 * Bump this to the library version if and only if the CX protocol changes.
 *
 * We could also have used 1, 2, 3, ... here to indicate protocol versions, but
 * those then still need to be mapped to software versions to be useful. So we
 * might as well use the software version as protocol version and immediately
 * generate a useful error message from this.
 *
 * Note the following:
 *
 * - The versions are not compared in a semver way, they are used as
 *    opaque ordered tokens.
 * - The version needs to be set to the NEXT releasable version when it's
 *   updated (as the current verison in package.json has already been released!)
 * - The request does not have versioning yet, only the response.
 */
exports.CLOUD_ASSEMBLY_VERSION = '0.36.0';
/**
 * Look at the type of response we get and upgrade it to the latest expected version
 */
function verifyManifestVersion(manifetVersion) {
    const frameworkVersion = parseSemver(manifetVersion);
    const toolkitVersion = parseSemver(exports.CLOUD_ASSEMBLY_VERSION);
    // if framework > cli, we require a newer cli version
    if (semver.gt(frameworkVersion, toolkitVersion)) {
        throw new Error(`CDK CLI >= ${frameworkVersion} is required to interact with this app`);
    }
    // if framework < cli, we require a newer framework version
    if (semver.lt(frameworkVersion, toolkitVersion)) {
        throw new Error(`CDK CLI can only be used with apps created by CDK >= ${exports.CLOUD_ASSEMBLY_VERSION}`);
    }
}
exports.verifyManifestVersion = verifyManifestVersion;
function parseSemver(version) {
    const ver = semver.coerce(version);
    if (!ver) {
        throw new Error(`Could not parse "${version}" as semver`);
    }
    return ver;
}
//# sourceMappingURL=data:application/json;base64,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