"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const fs = require("fs");
const os = require("os");
const path = require("path");
const lib_1 = require("../lib");
const versioning_1 = require("../lib/versioning");
test('cloud assembly builder', () => {
    // GIVEN
    const outdir = fs.mkdtempSync(path.join(os.tmpdir(), 'cloud-assembly-builder-tests'));
    const session = new lib_1.CloudAssemblyBuilder(outdir);
    const templateFile = 'foo.template.json';
    // WHEN
    session.addArtifact('my-first-artifact', {
        type: lib_1.ArtifactType.AWS_CLOUDFORMATION_STACK,
        environment: 'aws://1222344/us-east-1',
        dependencies: ['minimal-artifact'],
        metadata: {
            foo: [{ data: 123, type: 'foo', trace: [] }]
        },
        properties: {
            templateFile,
            parameters: {
                prop1: '1234',
                prop2: '555'
            }
        },
    });
    session.addMissing({
        key: 'foo',
        provider: 'context-provider',
        props: {
            a: 'A',
            b: 2
        }
    });
    session.addArtifact('minimal-artifact', {
        type: lib_1.ArtifactType.AWS_CLOUDFORMATION_STACK,
        environment: 'aws://111/helo-world',
        properties: {
            templateFile
        }
    });
    fs.writeFileSync(path.join(session.outdir, templateFile), JSON.stringify({
        Resources: {
            MyTopic: {
                Type: 'AWS::S3::Topic'
            }
        }
    }));
    const assembly = session.build();
    const manifest = assembly.manifest;
    // THEN
    // verify the manifest looks right
    expect(manifest).toStrictEqual({
        version: versioning_1.CLOUD_ASSEMBLY_VERSION,
        missing: [
            { key: 'foo', provider: 'context-provider', props: { a: 'A', b: 2 } }
        ],
        artifacts: {
            'my-first-artifact': {
                type: 'aws:cloudformation:stack',
                environment: 'aws://1222344/us-east-1',
                dependencies: ['minimal-artifact'],
                metadata: { foo: [{ data: 123, type: 'foo', trace: [] }] },
                properties: {
                    templateFile: 'foo.template.json',
                    parameters: {
                        prop1: '1234',
                        prop2: '555'
                    },
                },
            },
            'minimal-artifact': {
                type: 'aws:cloudformation:stack',
                environment: 'aws://111/helo-world',
                properties: { templateFile: 'foo.template.json' }
            }
        }
    });
    // verify we have a template file
    expect(assembly.getStack('minimal-artifact').template).toStrictEqual({
        Resources: {
            MyTopic: {
                Type: 'AWS::S3::Topic'
            }
        }
    });
});
test('outdir must be a directory', () => {
    expect(() => new lib_1.CloudAssemblyBuilder(__filename)).toThrow('must be a directory');
});
//# sourceMappingURL=data:application/json;base64,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