"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const path = require("path");
const lib_1 = require("../lib");
const versioning_1 = require("../lib/versioning");
const FIXTURES = path.join(__dirname, 'fixtures');
test('empty assembly', () => {
    const assembly = new lib_1.CloudAssembly(path.join(FIXTURES, 'empty'));
    expect(assembly.artifacts).toEqual([]);
    expect(assembly.runtime).toEqual({ libraries: {} });
    expect(assembly.stacks).toEqual([]);
    expect(assembly.version).toEqual(versioning_1.CLOUD_ASSEMBLY_VERSION);
    expect(assembly.manifest).toMatchSnapshot();
});
test('assembly a single cloudformation stack', () => {
    const assembly = new lib_1.CloudAssembly(path.join(FIXTURES, 'single-stack'));
    expect(assembly.artifacts).toHaveLength(1);
    expect(assembly.stacks).toHaveLength(1);
    expect(assembly.manifest.missing).toBeUndefined();
    expect(assembly.runtime).toEqual({ libraries: {} });
    expect(assembly.version).toEqual(versioning_1.CLOUD_ASSEMBLY_VERSION);
    expect(assembly.artifacts[0]).toEqual(assembly.stacks[0]);
    const stack = assembly.stacks[0];
    expect(stack.manifest).toMatchSnapshot();
    expect(stack.assets).toHaveLength(0);
    expect(stack.dependencies).toEqual([]);
    expect(stack.environment).toEqual({ account: '37736633', region: 'us-region-1', name: 'aws://37736633/us-region-1' });
    expect(stack.template).toEqual({ Resources: { MyBucket: { Type: "AWS::S3::Bucket" } } });
    expect(stack.messages).toEqual([]);
    expect(stack.manifest.metadata).toEqual(undefined);
    expect(stack.originalName).toEqual('MyStackName');
    expect(stack.name).toEqual('MyStackName');
});
test('assembly with missing context', () => {
    const assembly = new lib_1.CloudAssembly(path.join(FIXTURES, 'missing-context'));
    expect(assembly.manifest.missing).toMatchSnapshot();
});
test('assembly with multiple stacks', () => {
    const assembly = new lib_1.CloudAssembly(path.join(FIXTURES, 'multiple-stacks'));
    expect(assembly.stacks).toHaveLength(2);
    expect(assembly.artifacts).toHaveLength(2);
});
test('fails for invalid artifact type', () => {
    expect(() => new lib_1.CloudAssembly(path.join(FIXTURES, 'invalid-artifact-type')))
        .toThrow('unsupported artifact type: who:am:i');
});
test('fails for invalid environment format', () => {
    expect(() => new lib_1.CloudAssembly(path.join(FIXTURES, 'invalid-env-format')))
        .toThrow('Unable to parse environment specification');
});
test('fails if stack artifact does not have properties', () => {
    expect(() => new lib_1.CloudAssembly(path.join(FIXTURES, 'stack-without-params')))
        .toThrow('Invalid CloudFormation stack artifact. Missing \"templateFile\" property in cloud assembly manifest');
});
test('messages', () => {
    const assembly = new lib_1.CloudAssembly(path.join(FIXTURES, 'messages'));
    expect(assembly.stacks[0].messages).toMatchSnapshot();
});
test('assets', () => {
    const assembly = new lib_1.CloudAssembly(path.join(FIXTURES, 'assets'));
    expect(assembly.stacks[0].assets).toMatchSnapshot();
});
test('dependencies', () => {
    const assembly = new lib_1.CloudAssembly(path.join(FIXTURES, 'depends'));
    expect(assembly.stacks).toHaveLength(4);
    // expect stacks to be listed in topological order
    expect(assembly.stacks.map(s => s.name)).toEqual(['StackA', 'StackD', 'StackC', 'StackB']);
    expect(assembly.stacks[0].dependencies).toEqual([]);
    expect(assembly.stacks[1].dependencies).toEqual([]);
    expect(assembly.stacks[2].dependencies.map(x => x.id)).toEqual(['StackD']);
    expect(assembly.stacks[3].dependencies.map(x => x.id)).toEqual(['StackC', 'StackD']);
});
test('fails for invalid dependencies', () => {
    expect(() => new lib_1.CloudAssembly(path.join(FIXTURES, 'invalid-depends'))).toThrow('Artifact StackC depends on non-existing artifact StackX');
});
test('verifyManifestVersion', () => {
    versioning_1.verifyManifestVersion(versioning_1.CLOUD_ASSEMBLY_VERSION);
    expect(() => versioning_1.verifyManifestVersion('0.31.0')).toThrow(`CDK CLI can only be used with apps created by CDK >= ${versioning_1.CLOUD_ASSEMBLY_VERSION}`);
    expect(() => versioning_1.verifyManifestVersion('99.99.99')).toThrow(`CDK CLI >= 99.99.99 is required to interact with this app`);
});
//# sourceMappingURL=data:application/json;base64,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