#!/usr/bin/env python3
"""PyInstaller spec file generator for Riveter.

This module generates PyInstaller specification files that define how to
bundle Riveter with all dependencies, rule packs, and static assets.

The spec file includes:
- All Riveter modules in hiddenimports
- Rule packs directory and static assets as data files
- Platform-specific configurations
- Optimization settings for binary size and performance
"""

from pathlib import Path
from typing import Dict, List, Tuple


def generate_spec(
    entry_point: str,
    hidden_imports: List[str],
    data_files: List[Tuple[str, str]],
    target_platform: str,
    debug: bool = False,
) -> str:
    """Generate PyInstaller spec file content.

    Args:
        entry_point: Path to the main CLI entry point
        hidden_imports: List of modules to include as hidden imports
        data_files: List of (source, destination) tuples for data files
        target_platform: Target platform identifier
        debug: Enable debug mode

    Returns:
        Complete spec file content as string
    """

    # Convert data files to PyInstaller format
    datas_list = []
    for source, dest in data_files:
        datas_list.append(f"('{source}', '{dest}')")

    datas_str = "[" + ", ".join(datas_list) + "]" if datas_list else "[]"

    # Convert hidden imports to string format
    hiddenimports_str = "[" + ", ".join(f"'{imp}'" for imp in hidden_imports) + "]"

    # Platform-specific configurations
    platform_config = get_platform_config(target_platform)

    # Generate the spec file content
    spec_content = f'''# -*- mode: python ; coding: utf-8 -*-
"""
PyInstaller spec file for Riveter binary distribution.

This spec file is auto-generated by scripts/build_spec.py.
Do not edit this file manually.

Target platform: {target_platform}
Debug mode: {debug}
"""

import os
import sys
from pathlib import Path

# Add source directory to Python path
project_root = Path(os.getcwd())
src_dir = project_root / "src"
sys.path.insert(0, str(src_dir))

block_cipher = None

a = Analysis(
    ['{entry_point}'],
    pathex=[str(src_dir)],
    binaries=[],
    datas={datas_str},
    hiddenimports={hiddenimports_str},
    hookspath=[],
    hooksconfig={{}},
    runtime_hooks=[],
    excludes={platform_config['excludes']},
    win_no_prefer_redirects=False,
    win_private_assemblies=False,
    cipher=block_cipher,
    noarchive=False,
)

# Remove duplicate entries and optimize
pyz = PYZ(a.pure, a.zipped_data, cipher=block_cipher)

exe = EXE(
    pyz,
    a.scripts,
    a.binaries,
    a.zipfiles,
    a.datas,
    [],
    name='riveter',
    debug={debug},
    bootloader_ignore_signals=False,
    strip={not debug},
    upx={platform_config['use_upx']},
    upx_exclude=[],
    runtime_tmpdir=None,
    console=True,
    disable_windowed_traceback=False,
    target_arch='{platform_config['target_arch']}',
    codesign_identity=None,
    entitlements_file=None,
{platform_config['exe_options']}
)
'''

    return spec_content


def get_platform_config(target_platform: str) -> Dict:
    """Get platform-specific configuration for PyInstaller.

    Args:
        target_platform: Target platform identifier

    Returns:
        Dictionary containing platform-specific settings
    """

    base_config = {
        "excludes": [
            # Exclude unnecessary modules to reduce binary size
            "tkinter",
            "matplotlib",
            "numpy",
            "pandas",
            "scipy",
            "PIL",
            "IPython",
            "jupyter",
            "notebook",
            "webbrowser",
            "turtle",
            "curses",
            # Test-related modules
            "_pytest",
            "pytest_cov",
            "coverage",
            "nose",
            "mock",
            # Development tools
            "black",
            "isort",
            "mypy",
            "ruff",
            "bandit",
            "pre_commit",
            # Documentation tools
            "sphinx",
            "docutils",
            "jinja2",
            "markupsafe",
        ],
        "use_upx": True,  # Enable UPX compression by default
        "target_arch": None,
        "exe_options": "",
    }

    if target_platform == "macos-intel":
        base_config.update(
            {
                "target_arch": "x86_64",
                "exe_options": """    # macOS Intel specific options
    icon=None,""",
            }
        )
    elif target_platform == "macos-arm64":
        base_config.update(
            {
                "target_arch": "arm64",
                "exe_options": """    # macOS Apple Silicon specific options
    icon=None,""",
            }
        )
    elif target_platform == "linux-x86_64":
        base_config.update(
            {
                "target_arch": "x86_64",
                "use_upx": True,  # UPX works well on Linux
                "exe_options": """    # Linux x86_64 specific options
    icon=None,""",
            }
        )

    # Convert excludes list to string format
    excludes_str = "[" + ", ".join(f"'{exc}'" for exc in base_config["excludes"]) + "]"
    base_config["excludes"] = excludes_str

    return base_config


def get_riveter_modules() -> List[str]:
    """Get list of all Riveter modules for hidden imports.

    Returns:
        List of Riveter module names
    """
    modules = [
        "riveter",
        "riveter.cli",
        "riveter.scanner",
        "riveter.rules",
        "riveter.config",
        "riveter.extract_config",
        "riveter.logging",
        "riveter.performance",
        "riveter.reporter",
        "riveter.rule_distribution",
        "riveter.rule_filter",
        "riveter.rule_linter",
        "riveter.rule_packs",
        "riveter.rule_repository",
        "riveter.formatters",
        "riveter.operators",
        "riveter.exceptions",
        "riveter.cloud_parsers",
        "riveter.changelog_processor",
        "riveter.toml_handler",
        "riveter.version_manager",
    ]

    return modules


def get_third_party_imports() -> List[str]:
    """Get list of third-party modules that might need explicit inclusion.

    Returns:
        List of third-party module names
    """
    modules = [
        # Core dependencies
        "yaml",
        "hcl2",
        "click",
        "rich",
        "cryptography",
        "requests",
        # Rich dependencies
        "rich.console",
        "rich.table",
        "rich.text",
        "rich.style",
        "rich.color",
        "rich.theme",
        "rich.markup",
        "rich.measure",
        "rich.segment",
        "rich.cells",
        "rich.align",
        "rich.padding",
        "rich.panel",
        "rich.box",
        "rich.columns",
        "rich.layout",
        "rich.live",
        "rich.progress",
        "rich.spinner",
        "rich.status",
        "rich.tree",
        "rich.syntax",
        "rich.traceback",
        "rich.logging",
        # Click dependencies
        "click.core",
        "click.decorators",
        "click.options",
        "click.types",
        "click.utils",
        "click.exceptions",
        "click.formatting",
        "click.parser",
        "click.termui",
        # HCL2 dependencies
        "hcl2.api",
        "hcl2.transformer",
        "hcl2.lark_parser",
        # Cryptography dependencies
        "cryptography.hazmat",
        "cryptography.hazmat.primitives",
        "cryptography.hazmat.primitives.hashes",
        "cryptography.hazmat.primitives.serialization",
        "cryptography.hazmat.backends",
        "cryptography.hazmat.backends.openssl",
        # Requests dependencies
        "requests.adapters",
        "requests.auth",
        "requests.cookies",
        "requests.exceptions",
        "requests.models",
        "requests.sessions",
        "requests.structures",
        "requests.utils",
        "urllib3",
        "urllib3.util",
        "urllib3.util.retry",
        "urllib3.util.ssl_",
        "urllib3.util.timeout",
        "urllib3.poolmanager",
        "urllib3.connectionpool",
        # Standard library modules that might be missed
        "json",
        "pathlib",
        "subprocess",
        "tempfile",
        "shutil",
        "os",
        "sys",
        "platform",
        "logging",
        "argparse",
        "re",
        "datetime",
        "hashlib",
        "base64",
        "uuid",
        "collections",
        "collections.abc",
        "itertools",
        "functools",
        "operator",
        "typing",
        "dataclasses",
        "enum",
        "contextlib",
        "weakref",
        "copy",
        "pickle",
        "gzip",
        "tarfile",
        "zipfile",
        "io",
        "csv",
        "configparser",
        "urllib",
        "urllib.parse",
        "urllib.request",
        "http",
        "http.client",
        "ssl",
        "socket",
        "threading",
        "multiprocessing",
        "concurrent",
        "concurrent.futures",
        "queue",
        "time",
        "calendar",
        "locale",
        "gettext",
        "string",
        "textwrap",
        "unicodedata",
        "codecs",
        "encodings",
        "encodings.utf_8",
        "encodings.latin_1",
        "encodings.ascii",
    ]

    return modules


def get_all_hidden_imports() -> List[str]:
    """Get complete list of hidden imports for Riveter.

    Returns:
        Combined list of all modules to include as hidden imports
    """
    imports = []
    imports.extend(get_riveter_modules())
    imports.extend(get_third_party_imports())

    # Remove duplicates while preserving order
    seen = set()
    unique_imports = []
    for imp in imports:
        if imp not in seen:
            seen.add(imp)
            unique_imports.append(imp)

    return unique_imports


def find_rule_packs(project_root: Path) -> List[Tuple[str, str]]:
    """Find all rule pack files to include in the binary.

    Args:
        project_root: Root directory of the project

    Returns:
        List of (source_path, destination_dir) tuples
    """
    rule_packs = []
    rule_packs_dir = project_root / "rule_packs"

    if rule_packs_dir.exists():
        for rule_pack in rule_packs_dir.glob("*.yml"):
            rule_packs.append((str(rule_pack), "rule_packs"))

        for rule_pack in rule_packs_dir.glob("*.yaml"):
            rule_packs.append((str(rule_pack), "rule_packs"))

    return rule_packs


def find_static_assets(project_root: Path) -> List[Tuple[str, str]]:
    """Find static assets to include in the binary.

    Args:
        project_root: Root directory of the project

    Returns:
        List of (source_path, destination_dir) tuples
    """
    assets = []

    # Include any configuration templates
    templates_dir = project_root / "templates"
    if templates_dir.exists():
        for template in templates_dir.rglob("*"):
            if template.is_file():
                rel_path = template.relative_to(templates_dir)
                assets.append((str(template), f"templates/{rel_path.parent}"))

    # Include documentation files that might be needed at runtime
    docs_dir = project_root / "docs"
    if docs_dir.exists():
        for doc in docs_dir.glob("*.md"):
            if doc.name.lower() in ["readme.md", "license.md", "changelog.md"]:
                assets.append((str(doc), "docs"))

    # Include version information
    version_files = ["VERSION", "version.txt", "pyproject.toml"]
    for version_file in version_files:
        version_path = project_root / version_file
        if version_path.exists():
            assets.append((str(version_path), "."))

    return assets


def get_all_data_files(project_root: Path) -> List[Tuple[str, str]]:
    """Get all data files to include in the binary.

    Args:
        project_root: Root directory of the project

    Returns:
        Combined list of all data files
    """
    data_files = []
    data_files.extend(find_rule_packs(project_root))
    data_files.extend(find_static_assets(project_root))

    return data_files


def main():
    """Main entry point for testing spec generation."""
    import sys
    from pathlib import Path

    # Find project root
    current_dir = Path(__file__).parent
    project_root = current_dir.parent

    if not (project_root / "pyproject.toml").exists():
        print("Error: Could not find project root (pyproject.toml not found)")
        sys.exit(1)

    # Generate spec for current platform
    entry_point = str(project_root / "src" / "riveter" / "cli.py")
    hidden_imports = get_all_hidden_imports()
    data_files = get_all_data_files(project_root)

    spec_content = generate_spec(
        entry_point=entry_point,
        hidden_imports=hidden_imports,
        data_files=data_files,
        target_platform="linux-x86_64",  # Default for testing
        debug=False,
    )

    print("Generated PyInstaller spec file:")
    print("=" * 50)
    print(spec_content)


if __name__ == "__main__":
    main()
