from http import HTTPStatus
from typing import Any, Optional, Union
from uuid import UUID

import httpx

from ... import errors
from ...client import AuthenticatedClient, Client
from ...models.http_validation_error import HTTPValidationError
from ...models.machine_response import MachineResponse
from ...models.machine_update import MachineUpdate
from ...types import Response


def _get_kwargs(
    machine_id: UUID,
    *,
    body: MachineUpdate,
) -> dict[str, Any]:
    headers: dict[str, Any] = {}

    _kwargs: dict[str, Any] = {
        "method": "patch",
        "url": f"/v1/machines/{machine_id}",
    }

    _kwargs["json"] = body.to_dict()

    headers["Content-Type"] = "application/json"

    _kwargs["headers"] = headers
    return _kwargs


def _parse_response(
    *, client: Union[AuthenticatedClient, Client], response: httpx.Response
) -> Optional[Union[HTTPValidationError, MachineResponse]]:
    if response.status_code == 200:
        response_200 = MachineResponse.from_dict(response.json())

        return response_200
    if response.status_code == 422:
        response_422 = HTTPValidationError.from_dict(response.json())

        return response_422
    if client.raise_on_unexpected_status:
        raise errors.UnexpectedStatus(response.status_code, response.content)
    else:
        return None


def _build_response(
    *, client: Union[AuthenticatedClient, Client], response: httpx.Response
) -> Response[Union[HTTPValidationError, MachineResponse]]:
    return Response(
        status_code=HTTPStatus(response.status_code),
        content=response.content,
        headers=response.headers,
        parsed=_parse_response(client=client, response=response),
    )


def sync_detailed(
    machine_id: UUID,
    *,
    client: AuthenticatedClient,
    body: MachineUpdate,
) -> Response[Union[HTTPValidationError, MachineResponse]]:
    """Update Machine

     Update a machine's information.

    Only the fields provided in the request body will be updated.
    The machine must belong to the authenticated user.

    Args:
        machine_id (UUID):
        body (MachineUpdate): Schema for updating a machine

    Raises:
        errors.UnexpectedStatus: If the server returns an undocumented status code and Client.raise_on_unexpected_status is True.
        httpx.TimeoutException: If the request takes longer than Client.timeout.

    Returns:
        Response[Union[HTTPValidationError, MachineResponse]]
    """

    kwargs = _get_kwargs(
        machine_id=machine_id,
        body=body,
    )

    response = client.get_httpx_client().request(
        **kwargs,
    )

    return _build_response(client=client, response=response)


def sync(
    machine_id: UUID,
    *,
    client: AuthenticatedClient,
    body: MachineUpdate,
) -> Optional[Union[HTTPValidationError, MachineResponse]]:
    """Update Machine

     Update a machine's information.

    Only the fields provided in the request body will be updated.
    The machine must belong to the authenticated user.

    Args:
        machine_id (UUID):
        body (MachineUpdate): Schema for updating a machine

    Raises:
        errors.UnexpectedStatus: If the server returns an undocumented status code and Client.raise_on_unexpected_status is True.
        httpx.TimeoutException: If the request takes longer than Client.timeout.

    Returns:
        Union[HTTPValidationError, MachineResponse]
    """

    return sync_detailed(
        machine_id=machine_id,
        client=client,
        body=body,
    ).parsed


async def asyncio_detailed(
    machine_id: UUID,
    *,
    client: AuthenticatedClient,
    body: MachineUpdate,
) -> Response[Union[HTTPValidationError, MachineResponse]]:
    """Update Machine

     Update a machine's information.

    Only the fields provided in the request body will be updated.
    The machine must belong to the authenticated user.

    Args:
        machine_id (UUID):
        body (MachineUpdate): Schema for updating a machine

    Raises:
        errors.UnexpectedStatus: If the server returns an undocumented status code and Client.raise_on_unexpected_status is True.
        httpx.TimeoutException: If the request takes longer than Client.timeout.

    Returns:
        Response[Union[HTTPValidationError, MachineResponse]]
    """

    kwargs = _get_kwargs(
        machine_id=machine_id,
        body=body,
    )

    response = await client.get_async_httpx_client().request(**kwargs)

    return _build_response(client=client, response=response)


async def asyncio(
    machine_id: UUID,
    *,
    client: AuthenticatedClient,
    body: MachineUpdate,
) -> Optional[Union[HTTPValidationError, MachineResponse]]:
    """Update Machine

     Update a machine's information.

    Only the fields provided in the request body will be updated.
    The machine must belong to the authenticated user.

    Args:
        machine_id (UUID):
        body (MachineUpdate): Schema for updating a machine

    Raises:
        errors.UnexpectedStatus: If the server returns an undocumented status code and Client.raise_on_unexpected_status is True.
        httpx.TimeoutException: If the request takes longer than Client.timeout.

    Returns:
        Union[HTTPValidationError, MachineResponse]
    """

    return (
        await asyncio_detailed(
            machine_id=machine_id,
            client=client,
            body=body,
        )
    ).parsed
