import datetime
from collections.abc import Mapping
from typing import TYPE_CHECKING, Any, TypeVar, Union, cast
from uuid import UUID

from attrs import define as _attrs_define
from attrs import field as _attrs_field
from dateutil.parser import isoparse

from ..types import UNSET, Unset

if TYPE_CHECKING:
    from ..models.trajectory_response_dimensions import TrajectoryResponseDimensions
    from ..models.trajectory_response_original_input_values_type_0 import TrajectoryResponseOriginalInputValuesType0
    from ..models.trajectory_response_trajectory_data_item import TrajectoryResponseTrajectoryDataItem


T = TypeVar("T", bound="TrajectoryResponse")


@_attrs_define
class TrajectoryResponse:
    """Trajectory response schema

    Attributes:
        workflow_id (UUID):
        trajectory_data (list['TrajectoryResponseTrajectoryDataItem']):
        dimensions (TrajectoryResponseDimensions): Display dimensions when trajectory was recorded
        id (UUID):
        user_id (UUID):
        created_at (datetime.datetime):
        updated_at (datetime.datetime):
        original_input_values (Union['TrajectoryResponseOriginalInputValuesType0', None, Unset]): Original input values
            used when trajectory was created
    """

    workflow_id: UUID
    trajectory_data: list["TrajectoryResponseTrajectoryDataItem"]
    dimensions: "TrajectoryResponseDimensions"
    id: UUID
    user_id: UUID
    created_at: datetime.datetime
    updated_at: datetime.datetime
    original_input_values: Union["TrajectoryResponseOriginalInputValuesType0", None, Unset] = UNSET
    additional_properties: dict[str, Any] = _attrs_field(init=False, factory=dict)

    def to_dict(self) -> dict[str, Any]:
        from ..models.trajectory_response_original_input_values_type_0 import TrajectoryResponseOriginalInputValuesType0

        workflow_id = str(self.workflow_id)

        trajectory_data = []
        for trajectory_data_item_data in self.trajectory_data:
            trajectory_data_item = trajectory_data_item_data.to_dict()
            trajectory_data.append(trajectory_data_item)

        dimensions = self.dimensions.to_dict()

        id = str(self.id)

        user_id = str(self.user_id)

        created_at = self.created_at.isoformat()

        updated_at = self.updated_at.isoformat()

        original_input_values: Union[None, Unset, dict[str, Any]]
        if isinstance(self.original_input_values, Unset):
            original_input_values = UNSET
        elif isinstance(self.original_input_values, TrajectoryResponseOriginalInputValuesType0):
            original_input_values = self.original_input_values.to_dict()
        else:
            original_input_values = self.original_input_values

        field_dict: dict[str, Any] = {}
        field_dict.update(self.additional_properties)
        field_dict.update(
            {
                "workflow_id": workflow_id,
                "trajectory_data": trajectory_data,
                "dimensions": dimensions,
                "id": id,
                "user_id": user_id,
                "created_at": created_at,
                "updated_at": updated_at,
            }
        )
        if original_input_values is not UNSET:
            field_dict["original_input_values"] = original_input_values

        return field_dict

    @classmethod
    def from_dict(cls: type[T], src_dict: Mapping[str, Any]) -> T:
        from ..models.trajectory_response_dimensions import TrajectoryResponseDimensions
        from ..models.trajectory_response_original_input_values_type_0 import TrajectoryResponseOriginalInputValuesType0
        from ..models.trajectory_response_trajectory_data_item import TrajectoryResponseTrajectoryDataItem

        d = dict(src_dict)
        workflow_id = UUID(d.pop("workflow_id"))

        trajectory_data = []
        _trajectory_data = d.pop("trajectory_data")
        for trajectory_data_item_data in _trajectory_data:
            trajectory_data_item = TrajectoryResponseTrajectoryDataItem.from_dict(trajectory_data_item_data)

            trajectory_data.append(trajectory_data_item)

        dimensions = TrajectoryResponseDimensions.from_dict(d.pop("dimensions"))

        id = UUID(d.pop("id"))

        user_id = UUID(d.pop("user_id"))

        created_at = isoparse(d.pop("created_at"))

        updated_at = isoparse(d.pop("updated_at"))

        def _parse_original_input_values(
            data: object,
        ) -> Union["TrajectoryResponseOriginalInputValuesType0", None, Unset]:
            if data is None:
                return data
            if isinstance(data, Unset):
                return data
            try:
                if not isinstance(data, dict):
                    raise TypeError()
                original_input_values_type_0 = TrajectoryResponseOriginalInputValuesType0.from_dict(data)

                return original_input_values_type_0
            except:  # noqa: E722
                pass
            return cast(Union["TrajectoryResponseOriginalInputValuesType0", None, Unset], data)

        original_input_values = _parse_original_input_values(d.pop("original_input_values", UNSET))

        trajectory_response = cls(
            workflow_id=workflow_id,
            trajectory_data=trajectory_data,
            dimensions=dimensions,
            id=id,
            user_id=user_id,
            created_at=created_at,
            updated_at=updated_at,
            original_input_values=original_input_values,
        )

        trajectory_response.additional_properties = d
        return trajectory_response

    @property
    def additional_keys(self) -> list[str]:
        return list(self.additional_properties.keys())

    def __getitem__(self, key: str) -> Any:
        return self.additional_properties[key]

    def __setitem__(self, key: str, value: Any) -> None:
        self.additional_properties[key] = value

    def __delitem__(self, key: str) -> None:
        del self.additional_properties[key]

    def __contains__(self, key: str) -> bool:
        return key in self.additional_properties
