# Import necessary types and helpers from Python's standard library.
from collections.abc import Callable
from typing import Any, Optional

# Import custom modules from this integration.
from ...api import TISApi  # The main API for communicating with TIS devices.
from ...Protocols.udp.ProtocolHandler import (
    TISProtocolHandler,
    TISPacket,
)  # TIS protocol specifics.


class TISAPISwitch:
    """Base class for TIS switches, providing common functionality."""

    def __init__(
        self,
        tis_api: TISApi,
        *,
        channel_number: int,
        device_id: list[int],
        gateway: str,
        is_protected: bool = False,
    ) -> None:
        """Initialize the base switch attributes."""
        self.api = tis_api

        # Store device-specific information.
        self.device_id = device_id
        self.gateway = gateway
        self.channel_number = int(channel_number)
        self.is_protected = is_protected
        self._is_on: bool | None = None
        self._update_callback: Callable[[], None] | None = None

        # This avoids rebuilding the byte arrays every time a command is sent.
        self.on_packet: TISPacket = TISProtocolHandler.generate_control_on_packet(self)
        self.off_packet: TISPacket = TISProtocolHandler.generate_control_off_packet(
            self
        )
        self.update_packet: TISPacket = (
            TISProtocolHandler.generate_control_update_packet(self)
        )

    @property
    def is_on(self) -> bool | None:
        """Return true if the switch is on."""
        return self._is_on

    def register_callback(self, callback: Callable[[], None]) -> None:
        """Register a callback function to call when the state changes."""
        self._update_callback = callback

    def process_update(self, event_data: dict[str, Any]) -> None:
        """Process an incoming event from the TIS gateway and update state."""
        new_state: bool | None = self._is_on
        feedback_type = event_data.get("feedback_type")

        if feedback_type == "control_response":
            if int(event_data["channel_number"]) == self.channel_number:
                channel_value = event_data["additional_bytes"][2]
                new_state = int(channel_value) == 100

        elif feedback_type == "update_response":
            additional_bytes = event_data["additional_bytes"]
            channel_status = int(additional_bytes[self.channel_number])
            new_state = channel_status > 0

        elif feedback_type == "offline_device":
            new_state = None

        # If the state changed, update it and call the callback to notify listeners.
        if new_state != self._is_on:
            self._is_on = new_state
            if self._update_callback:
                self._update_callback()

    async def turn_switch_on(self) -> bool | None:
        """Turn the switch on by sending the on_packet."""
        # Send the pre-generated 'on' packet and wait for an acknowledgement (ack).
        return await self.api.protocol.sender.send_packet_with_ack(self.on_packet)

    async def turn_switch_off(self) -> bool | None:
        """Turn the switch off by sending the off_packet."""
        # Send the pre-generated 'off' packet and wait for an acknowledgement (ack).
        return await self.api.protocol.sender.send_packet_with_ack(self.off_packet)

    async def request_update(self) -> None:
        """Send a request to the device for its current state."""
        await self.api.protocol.sender.send_packet(self.update_packet)
