import ast
import json
from pathlib import Path
import toml
import pandas as pd
from .logging_setup import logger_ai4cehelper
from ._backend_calls import _backend_POST, _backend_PUT, _backend_GET
from .functions import load_file

def create_new_project(project_info: dict):
    """Create a new project in the backend.
    Params:
        project_info(dict):
    Returns:
        dict:
    """
    status_code, project = _backend_POST(
        endpoint=f"/v2/projects/", data=project_info)
    if status_code != 201:
        logger_ai4cehelper.error(f"(cnp) Error creating new project ({status_code}): {project}")
        raise Exception(f"Error creating new project ({status_code}): {project}")
    return project


def get_list_of_all_project_infos() -> list[tuple[int, str, str]]:
    # check for refactoring
    """A function to connect to the backend and get a list of all projects
    in the current project data base.

    Params:
        None

    Returns:
        tuple[int, str, str]: A list of tuples containing project ID, project name, and last modified date.
    """

    status_code, projects = _backend_GET(endpoint=f"/v2/projects/")
    project_ids: list[tuple[int, str, str]] = []
    if status_code == 200:
        for project in projects:
            project_ids.append((project["id"], project["project_name"], project["modified"]))
        return project_ids
    else:
        logger_ai4cehelper.error(f"(gloapi) Error fetching projects ({status_code}): {projects}")
        raise Exception(f"Error fetching projects ({status_code}): {projects}")

def get_project_info(project_id: int) -> dict:
    status_code, project_info = _backend_GET(endpoint=f"/v2/projects/{project_id}/")
    if status_code != 200:
        logger_ai4cehelper.error(f"(gpi) Error fetching project info ({status_code}): {project_info}")
        raise Exception(f"Error fetching project info ({status_code}): {project_info}")
    return project_info


def set_project_name(project_id: int, project_name: str) -> dict:
    """A function to set the project name of a specific project, identified by its ID.

    Params:
        project_id (int): The project Identification Number
        project_name(str): Name of the project/mission, eg OPS-Sat
    Returns:
        dict: json response from backend
    """

    data = {
        "id": project_id,
        "project_name": project_name,
    }

    # status, msg = _backend_put(endpoint=f"/v2/projects/{project_id}", data=data)
    # return (status, msg)
    status_code, response = _backend_PUT(endpoint=f"/v2/projects/{project_id}/", data=data)
    if status_code != 200:
        logger_ai4cehelper.error(f"(spn) Error setting project name: {response}")
        return {status_code: response}
    return response

def get_recent_project() -> tuple[int, str, str]:
    status_code, response = _backend_GET(endpoint=f"/v2/projects/recent/")
    if status_code != 200:
        logger_ai4cehelper.error(f"(grp) Error fetching recent project ({status_code}): {response}")
        raise Exception(f"Error fetching recent project ({status_code}): {response}")
    return (response["id"], response["project_name"], response["modified"])


def get_mission_orbit(project_id: int) -> dict:
    """A function to get the mission orbit info for a specific project based on its ID.
    Params:
        project_id (int): The project Identification Number

    Returns:
        dict: Keplerian elements to define the orbit
    """

    status_code, response = _backend_GET(endpoint=f"/v2/projects/{project_id}/mission/")
    if status_code != 200:
        logger_ai4cehelper.error(f"(gmo) Error fetching mission orbit ({status_code}): {response}")
        raise Exception(f"Error fetching mission orbit ({status_code}): {response}")
    if "orbit" not in response:
        logger_ai4cehelper.error(f"(gmo) No orbit information found in mission data")
        raise KeyError("No orbit information found in mission data")
    return response["orbit"]



def set_mission_orbit(project_id: int, orbit_info: dict) -> dict:
    """A function to set the orbit in the project database. Will create an empty mission parent if it does not exist.
    Params:
        project_id (int): The project Identification Number
        orbit_info(dict): Information about the satellite's orbit
    Returns:
        dict: 
     """
    # check if project exists
    status_code, project_info = _backend_GET(endpoint=f"/v2/projects/{project_id}/")
    if status_code != 200:
        logger_ai4cehelper.error(f"(smo) Error fetching project info ({status_code}): {project_info}")
        raise Exception(f"Error fetching project info ({status_code}): {project_info}")
    
    status_code, mission_info = _backend_GET(
        endpoint=f"/v2/projects/{project_id}/mission/")
    if status_code == 404:
        logger_ai4cehelper.info(f"(smo) No mission info found for project {project_id}. Creating new mission info.")
        mission_info = {
            "mission_name": f"Mission {project_info['project_name']}",
            "orbit": orbit_info,
        }
        status_code, response = _backend_POST(endpoint=f"/v2/projects/{project_id}/mission/", data=mission_info)
        if status_code != 201:
            logger_ai4cehelper.error(f"(smo) Error creating mission info ({status_code}): {response}")
            raise Exception(f"Error creating mission info ({status_code}): {response}")
        return response
    elif status_code == 200:
        # update possible mission_info["orbit"] with new orbit_info
        logger_ai4cehelper.info(f"(smo) Mission info found for project {project_id}. Updating orbit info.")
        if "orbit" not in mission_info:
            mission_info["orbit"] = {}
        mission_info["orbit"].update(orbit_info)
        status_code, response = _backend_PUT(endpoint=f"/v2/projects/{project_id}/mission/", data=mission_info)
        if status_code != 200:
            logger_ai4cehelper.error(f"(smo) Error updating mission info ({status_code}): {response}")
            raise Exception(f"Error updating mission info ({status_code}): {response}")
        return response
    else:
        logger_ai4cehelper.error(f"(smo) Error fetching mission info ({status_code}): {mission_info}")
        raise Exception(f"Error fetching mission info ({status_code}): {mission_info}")


def get_enabled_components(nested_dict, enabled_components=None) -> list:
    # check for refactoring
    """
    Recursively traverses a nested dictionary to find and return a list of components that are enabled.

    Parameters:
    nested_dict (dict): The nested dictionary to traverse.
    enabled_components (list, optional): A list to store the names of the enabled components.
                                         Defaults to None, in which case a new list is created.

    Returns:
    list: A list of the names of the enabled components.
    """

    if enabled_components is None:
        enabled_components = []

    for key, value in nested_dict.items():
        if isinstance(value, dict):
            if value.get('Enabled') == True:
                enabled_components.append(key)
                pass
            get_enabled_components(value, enabled_components)

    return enabled_components




def traverse_and_modify(d: dict, sys_config_enabled: dict):
    # check for refactoring
    for key, value in d.items():
        if isinstance(value, dict):
            component = key
            traverse_and_modify(d=value, sys_config_enabled=sys_config_enabled)
        else:
            # This block executes only if `value` is not a dict, i.e., at the deepest level
            # if d["Enabled"] is True or d["Enable"] is True or :
            try:
                if d["Enabled"] is True:
                    sys_config_enabled.update(enable_component(
                        component_name=key, data=load_file(Path("data/backend_sys_default.json"))))
            except KeyError as e: # what are you doing here -.-
                print("Error: ", e)


def enable_component(component_name: str, data: dict) -> dict:
    # check for refactoring
    """Recursively search for a component in the nested dictionary from the backend
    and set 'enabled' to True if the feature name matches any key or value (case-insensitive).

    Parameters:
        data (dict): The nested dictionary to search within.
        feature_name (str): The feature name to search for, case-insensitively.

    Returns:
        dict: edited dict
    """

    for key, value in data.items():
        #  print(key, value)
        if isinstance(value, dict):
            enable_component(component_name, value)
        elif isinstance(value, list):
            for item in value:
                if isinstance(item, dict):
                    enable_component(component_name, item)
        if key.lower() == component_name.lower() or str(value).lower() == component_name.lower():
            data['enabled'] = True
    return data


def set_sys_arch(project_id: int, sys_arch: dict) -> dict:
    """A function to set the system architecture in the project database. Updates the existing architecture if it exists.
    Params:
        project_id (int): The project Identification Number
        sys_arch(dict): Information about the satelllite's modules, which form the system architecture
    Returns:
        dict: Backend version of the system architecture or the status_code and error message.
     """

    status_code, system = _backend_GET(
        endpoint=f"/v2/projects/{project_id}/system/")
    if status_code == 404:
        logger_ai4cehelper.info(f"(ssa) No system architecture found for project {project_id}. Creating new system architecture.")
        status_code, response = _backend_POST(
            endpoint=f"/v2/projects/{project_id}/system/", data=sys_arch)
        if status_code != 201:
            logger_ai4cehelper.error(f"(ssa) Error creating system architecture ({status_code}): {response}")
            raise Exception(f"Error creating system architecture ({status_code}): {response}")
        return response
    elif status_code == 200:
        logger_ai4cehelper.info(f"(ssa) System architecture found for project {project_id}. Updating system architecture.")
        system.update(sys_arch)
        status_code, response = _backend_PUT(
            endpoint=f"/v2/projects/{project_id}/system/", data=system)
        if status_code != 200:
            logger_ai4cehelper.error(f"(ssa) Error updating system architecture ({status_code}): {response}")
            raise Exception(f"Error updating system architecture ({status_code}): {response}")
        return response


def get_sys_arch(project_id: int) -> dict:
    """A function to get the system configuration info for a specific project based on its ID.
    Params:
        project_id (int): The project Identification Number

    Returns:
        dict: System information
    """

    status_code, response = _backend_GET(
        endpoint=f"/v2/projects/{project_id}/system/")
    if status_code != 200:
        logger_ai4cehelper.error(f"(gsa) Error fetching system architecture ({status_code}): {response}")
        raise Exception(f"Error fetching system architecture ({status_code}): {response}")
    return response

#  set_comp_list(project: int, not_yet_defined: dict)
# For each selected system generator, we need a place to store the corresponding found comp lists
# Every generator can produce n sets of components


def set_sys_generator(project_id: int, sys_gen_info: dict) -> dict:
    status_code, response = _backend_POST(
        endpoint=f"/v2/projects/{project_id}/sysgen/", data=sys_gen_info)
    if status_code != 201:
        logger_ai4cehelper.error(f"(ssg) Error creating system generator info ({status_code}): {response}")
        raise Exception(f"Error creating system generator info ({status_code}): {response}")
    return status_code, response


def update_sys_generator(project_id: int, sys_gen_info: dict) -> dict:

    if "id" not in sys_gen_info:
        logger_ai4cehelper.error(f"(usg) sys_gen_info must contain an 'id' field for updating.")
        raise KeyError("sys_gen_info must contain an 'id' field for updating.")

    status_code, sys_gen = _backend_GET(
        endpoint=f"/v2/projects/{project_id}/sysgen/")
    if status_code == 404:
        logger_ai4cehelper.info(f"(usg) No system generator info found for project {project_id}. Creating new system generator info.")
        status_code, response = _backend_POST(
            endpoint=f"/v2/projects/{project_id}/sysgen/", data=sys_gen_info)
        if status_code != 201:
            logger_ai4cehelper.error(f"(usg) Error creating system generator info ({status_code}): {response}")
            raise Exception(f"Error creating system generator info ({status_code}): {response}")
        return response
    elif status_code == 200:
        logger_ai4cehelper.info(f"(usg) System generator info found for project {project_id}. Updating system generator info.")
        sys_gen.update(sys_gen_info)
        status_code, response = _backend_PUT(
            endpoint=f"/v2/projects/{project_id}/sysgen/{sys_gen_info['id']}/", data=sys_gen)
        if status_code != 200:
            logger_ai4cehelper.error(f"(usg) Error updating system generator info ({status_code}): {response}")
            raise Exception(f"Error updating system generator info ({status_code}): {response}")
        return response

def set_trained_model(project_id: int, sysgen_id: int, environment_settings: dict = {}, model_file: bytes = None) -> dict:
   
    if not model_file and not environment_settings:
        logger_ai4cehelper.error(f"(stm) Either environment_settings or model_file must be provided.")
        raise ValueError("Either environment_settings or model_file must be provided.")

    sys_gen = {
        "environment_settings": environment_settings,
        "id": sysgen_id,
    }

    sysgen_db = update_sys_generator(project_id=project_id, sys_gen_info=sys_gen)

    if model_file:
        status_code, response = _backend_GET(
            endpoint=f"/v2/projects/{project_id}/sysgen/{sysgen_id}/model/")
        if status_code == 404:
            logger_ai4cehelper.info(f"(stm) No trained model found for sysgen {sysgen_id} in project {project_id}. Creating new trained model.")
            status_code, response = _backend_POST(
                endpoint=f"/v2/projects/{project_id}/sysgen/{sysgen_id}/model/", data={"model_file": model_file})
            if status_code != 201:
                logger_ai4cehelper.error(f"(stm) Error creating trained model ({status_code}): {response}")
                raise Exception(f"Error creating trained model ({status_code}): {response}")
            return response
        elif status_code == 200:
            logger_ai4cehelper.info(f"(stm) Trained model found for sysgen {sysgen_id} in project {project_id}. Updating trained model.")
            status_code, response = _backend_PUT(
                endpoint=f"/v2/projects/{project_id}/sysgen/{sysgen_id}/model/", data={"model_file": model_file})
            if status_code != 200:
                logger_ai4cehelper.error(f"(stm) Error updating trained model ({status_code}): {response}")
                raise Exception(f"Error updating trained model ({status_code}): {response}")
            return response
        raise Exception(f"(stm) Unexpected status code ({status_code}): {response}")
    return sysgen_db

def get_prepared_system_generator_info(project_id: int) -> list:
    """A function to get all prepared system generators.
    Params:
        project_id (int): The project Identification Number

    Returns:
        list: list of dictionaries with infos for the prepared system generators
    """

    status_code, response = _backend_GET(
        endpoint=f"/v2/projects/{project_id}/sysgen/")
    if status_code != 200:
        logger_ai4cehelper.error(f"(gpsg) Error fetching prepared system generators ({status_code}): {response}")
        raise Exception(f"Error fetching prepared system generators ({status_code}): {response}")
    return response


def set_sys_arch_defaults(system: dict, project_id: int) -> dict:
    """Translate a system configuration from a TOML file dictionary previously handled by load_file() to a format that can be uploaded to the backend.

    Parameters:
    system (dict): The system configuration in TOML format.

    Returns:
    dict: The system configuration in a format that can be uploaded to the backend.
    """

    sys_arch_example = {
        "system_type": "string",
        "satellite_class": "string",
        "OAP_W": 0,
        "requirements": {},
        "aodcs": {
          "enabled": False,
          "amount_of_entities": 0,
          "description": "string",
          "eFunctions": [],
          "type": "string",
          "components": [],
          "depending_parameters": [],
          "cmg": {
            "enabled": False,
            "amount_of_entities": 0,
            "description": "string",
            "eFunctions": [],
            "requirements": {}
          },
          "magnetorquers": {
            "enabled": False,
            "amount_of_entities": 0,
            "description": "string",
            "eFunctions": [],
            "requirements": {}
          },
        },
        "tcs": {
          "enabled": False,
          "amount_of_entities": 0,
          "description": "string",
          "eFunctions": [],
          "requirements": {},
          "coatings": {
            "enabled": False,
            "amount_of_entities": 0,
            "description": "string",
            "eFunctions": [],
            "requirements": {}
          },
          "heaters": {
            "enabled": False,
            "amount_of_entities": 0,
            "description": "string",
            "eFunctions": [],
            "requirements": {}
          }
        }
    }

    sys_config_enabled = {}
    # Loading System Configuration
    for key, value in system.items():
        if isinstance(value, dict):
            traverse_and_modify(d=value, sys_config_enabled=sys_config_enabled)

    response = set_sys_arch(project_id=project_id, sys_arch=sys_config_enabled)
    pass
    return response


def set_new_project(project_info: dict) -> dict:
    """Create a new project in the backend.
    Params:
        project_info(dict):
    Returns:
        dict:
    """
    status_code, response = _backend_POST(
        endpoint="/v2/projects/", data=project_info)
    if status_code != 201:
        logger_ai4cehelper.error(f"(snp) Error creating new project ({status_code}): {response}")
        raise Exception(f"Error creating new project ({status_code}): {response}")
    return status_code, response

# def get_trained_model(project_id: int) -> mode: ASKYOUNES
#     pass


# def set_train_log(project_id: int, logs: ASKYOUNES) -> DB_response: ASKALEX
#     pass


# def get_train_logs(project_id: int) -> logs: ASKYOUNE   pass

