import json
import os
import tempfile
from pathlib import Path
from typing import List

import pytest
from pydantic import BaseModel, Field

from wraipperz.api.llm import GeminiProvider, call_ai
from wraipperz.api.messages import MessageBuilder


# Pydantic models for structured output testing
class SimpleVideoAnalysis(BaseModel):
    """Simple video analysis response for testing."""

    description: str = Field(
        description="Brief description of what happens in the video"
    )

    duration_estimate: str = Field(
        description="Estimated duration category: short, medium, or long"
    )

    contains_motion: bool = Field(
        description="Whether the video contains significant motion"
    )


class DetailedVideoAnalysis(BaseModel):
    """Detailed video analysis for testing structured output."""

    summary: str = Field(description="Overall summary of the video content")
    scenes: List[str] = Field(description="List of distinct scenes or segments")
    colors: List[str] = Field(description="Dominant colors observed in the video")
    estimated_length: str = Field(description="Estimated video length")


# Create test_assets directory if it doesn't exist
TEST_ASSETS_DIR = Path(__file__).parent / "test_assets"
TEST_ASSETS_DIR.mkdir(exist_ok=True)

# Path to test video
TEST_VIDEO_PATH = TEST_ASSETS_DIR / "test_video.mp4"


def _is_placeholder_file(file_path: Path) -> bool:
    """Check if the video file is just a placeholder text file."""
    try:
        # Try to read as text - if it succeeds and matches placeholder, it's a placeholder
        return file_path.read_text() == "placeholder video file for testing"
    except UnicodeDecodeError:
        # If we can't read as text, it's likely a real video file
        return False


@pytest.fixture(autouse=True)
def setup_test_video():
    """Create a simple test video if it doesn't exist"""
    if not TEST_VIDEO_PATH.exists():
        try:
            import cv2
            import numpy as np

            # Create a simple 2-second video with a red square moving
            fourcc = cv2.VideoWriter_fourcc(*"mp4v")
            video = cv2.VideoWriter(str(TEST_VIDEO_PATH), fourcc, 10.0, (200, 200))

            for i in range(20):  # 2 seconds at 10 fps
                frame = np.zeros((200, 200, 3), dtype=np.uint8)
                # Draw a red square that moves across the frame
                x = 10 + i * 5
                cv2.rectangle(frame, (x, 50), (x + 40, 90), (0, 0, 255), -1)
                video.write(frame)

            video.release()
            print(f"✅ Created test video: {TEST_VIDEO_PATH}")

        except ImportError:
            # If OpenCV not available, create a minimal placeholder
            TEST_VIDEO_PATH.write_text("placeholder video file for testing")
            print(f"⚠️ OpenCV not available, created placeholder: {TEST_VIDEO_PATH}")


# ===== GEMINI VIDEO PROCESSING TESTS =====


@pytest.mark.skipif(not os.getenv("GOOGLE_API_KEY"), reason="Google API key not found")
def test_gemini_video_upload_and_processing():
    """Test that Gemini can upload and process video files"""

    # Skip if we only have a placeholder file
    if _is_placeholder_file(TEST_VIDEO_PATH):
        pytest.skip("No real video file available, only placeholder")

    provider = GeminiProvider()

    try:
        # Test the video processing directly
        video_file = provider.process_video(str(TEST_VIDEO_PATH))

        # Verify the video file object has the expected attributes
        assert hasattr(video_file, "name")
        assert hasattr(video_file, "state")

        # Verify it's in a processed state
        assert video_file.state.name in ["ACTIVE", "PROCESSED"]

        print(f"✅ Video upload successful: {video_file.name}")

    except Exception as e:
        if "not supported" in str(e).lower():
            pytest.skip(f"Video processing not supported: {e}")
        else:
            raise


@pytest.mark.skipif(not os.getenv("GOOGLE_API_KEY"), reason="Google API key not found")
def test_gemini_video_analysis_with_call_ai():
    """Test video analysis using the call_ai wrapper with Gemini"""

    # Skip if we only have a placeholder file
    if _is_placeholder_file(TEST_VIDEO_PATH):
        pytest.skip("No real video file available, only placeholder")

    # Create video analysis messages using MessageBuilder
    messages = (
        MessageBuilder()
        .add_system("You are a helpful video analysis assistant.")
        .add_video(
            str(TEST_VIDEO_PATH),
            "Analyze this video and describe what you see. Focus on colors, movement, and any objects.",
        )
        .build()
    )

    try:
        # Test video analysis with a flash model first
        response, cost = call_ai(
            model="gemini/gemini-2.0-flash-exp",
            messages=messages,
            temperature=0.2,
            max_tokens=1000,
        )

        # Validate response
        assert isinstance(response, str)
        assert len(response) > 0

        # Should mention red color and movement (our test video has a moving red square)
        response_lower = response.lower()
        assert any(
            color in response_lower
            for color in ["red", "color", "square", "movement", "moving"]
        )

        # Validate cost
        assert isinstance(cost, (int, float))
        assert cost >= 0

        print(f"✅ Video analysis successful! Response: {response[:100]}...")

    except Exception as e:
        if any(
            skip_reason in str(e).lower()
            for skip_reason in ["not supported", "not available", "quota", "limit"]
        ):
            pytest.skip(f"Video analysis not available: {e}")
        else:
            raise


@pytest.mark.skipif(not os.getenv("GOOGLE_API_KEY"), reason="Google API key not found")
def test_gemini_video_analysis_with_thinking_budget():
    """Test video analysis with thinking budget (Gemini Pro model)"""

    # Skip if we only have a placeholder file
    if _is_placeholder_file(TEST_VIDEO_PATH):
        pytest.skip("No real video file available, only placeholder")

    messages = (
        MessageBuilder()
        .add_system(
            "You are a professional video analyst. Analyze videos comprehensively "
            "with detailed observations about content, movement, colors, and composition."
        )
        .add_video(
            str(TEST_VIDEO_PATH),
            "Provide a detailed analysis of this video. Think through your observations "
            "step by step and describe everything you notice.",
        )
        .build()
    )

    try:
        # Test with Gemini Pro model and thinking budget
        response, cost = call_ai(
            model="gemini/gemini-2.5-pro",
            messages=messages,
            temperature=0.3,
            max_tokens=2000,
            thinking_budget=8192,  # Test thinking budget functionality
        )

        # Validate response
        assert isinstance(response, str)
        assert len(response) > 0

        # Should be more detailed due to thinking budget
        assert len(response) > 50  # Expecting more detailed response

        # Validate cost
        assert isinstance(cost, (int, float))
        assert cost >= 0

        print(
            f"✅ Video analysis with thinking budget successful! Response length: {len(response)}"
        )

    except Exception as e:
        if any(
            skip_reason in str(e).lower()
            for skip_reason in [
                "not supported",
                "not available",
                "quota",
                "limit",
                "model not found",
            ]
        ):
            pytest.skip(f"Video analysis with thinking budget not available: {e}")
        else:
            raise


@pytest.mark.skipif(not os.getenv("GOOGLE_API_KEY"), reason="Google API key not found")
def test_gemini_video_analysis_structured_output():
    """Test video analysis with structured output using Pydantic models"""

    # Skip if we only have a placeholder file
    if _is_placeholder_file(TEST_VIDEO_PATH):
        pytest.skip("No real video file available, only placeholder")

    messages = (
        MessageBuilder()
        .add_system(
            "You are a video analysis assistant. You MUST respond with valid JSON only, "
            "no additional text or formatting. Follow the exact schema provided."
        )
        .add_video(
            str(TEST_VIDEO_PATH),
            "Analyze this video and provide a JSON response with these exact fields: "
            "description (string), duration_estimate (one of: short, medium, long), "
            "contains_motion (boolean). Return only valid JSON.",
        )
        .build()
    )

    try:
        # Test with structured output
        response, cost = call_ai(
            model="gemini/gemini-2.0-flash-exp",
            messages=messages,
            temperature=0.1,
            max_tokens=1000,
            response_schema=SimpleVideoAnalysis,
            response_mime_type="application/json",
        )

        # Validate response structure
        if isinstance(response, str):
            try:
                # Try to parse JSON response
                parsed_response = json.loads(response)
                assert "description" in parsed_response
                assert "duration_estimate" in parsed_response
                assert "contains_motion" in parsed_response

                # Validate content
                assert isinstance(parsed_response["contains_motion"], bool)
                assert parsed_response["duration_estimate"] in [
                    "short",
                    "medium",
                    "long",
                ]

            except json.JSONDecodeError:
                # If JSON parsing fails, check if it's a valid text response about video content
                # This can happen when the model doesn't follow JSON format despite instructions
                response_lower = response.lower()

                # Should contain video-related content
                assert any(
                    keyword in response_lower
                    for keyword in [
                        "video",
                        "character",
                        "motion",
                        "animated",
                        "description",
                    ]
                )

                print(f"⚠️ Model returned text instead of JSON: {response[:100]}...")
                # Still consider this a partial success since video analysis worked

        else:
            # Direct Pydantic model response
            assert hasattr(response, "description")
            assert hasattr(response, "duration_estimate")
            assert hasattr(response, "contains_motion")

        print("✅ Structured video analysis successful!")

    except Exception as e:
        if any(
            skip_reason in str(e).lower()
            for skip_reason in ["not supported", "not available", "quota", "limit"]
        ):
            pytest.skip(f"Structured video analysis not available: {e}")
        else:
            raise


# ===== MESSAGE BUILDER TESTS =====


def test_message_builder_video_support():
    """Test that MessageBuilder correctly handles video content"""

    # Test basic video message creation
    messages = (
        MessageBuilder()
        .add_system("You are a video analyst.")
        .add_video("/path/to/video.mp4", "Analyze this video")
        .build()
    )

    assert len(messages) == 2
    assert messages[0]["role"] == "system"
    assert messages[1]["role"] == "user"

    # Check video content structure
    user_content = messages[1]["content"]
    assert isinstance(user_content, list)
    assert len(user_content) == 2

    # Check video URL item
    video_item = user_content[1]
    assert video_item["type"] == "video_url"
    assert video_item["video_url"]["url"] == "/path/to/video.mp4"

    # Check text item
    text_item = user_content[0]
    assert text_item["type"] == "text"
    assert text_item["text"] == "Analyze this video"


def test_message_builder_mixed_content():
    """Test MessageBuilder with mixed image and video content"""

    messages = (
        MessageBuilder()
        .add_system("You are a multimedia analyst.")
        .add_user("Compare these media files:")
        .add_image("/path/to/image.jpg", "First, this image")
        .add_video("/path/to/video.mp4", "Then, this video")
        .build()
    )

    assert len(messages) == 2

    # Check mixed content structure
    user_content = messages[1]["content"]
    assert (
        len(user_content) == 5
    )  # Original text + image text + image + video text + video

    # Find video content
    video_items = [item for item in user_content if item.get("type") == "video_url"]
    assert len(video_items) == 1
    assert video_items[0]["video_url"]["url"] == "/path/to/video.mp4"


# ===== ERROR HANDLING TESTS =====


@pytest.mark.skipif(not os.getenv("GOOGLE_API_KEY"), reason="Google API key not found")
def test_video_processing_error_handling():
    """Test error handling for invalid video files"""

    provider = GeminiProvider()

    # Test with non-existent file
    with pytest.raises(Exception):
        provider.process_video("/non/existent/video.mp4")

    # Test with invalid file type (Gemini may be more lenient, so we just test it doesn't crash)
    with tempfile.NamedTemporaryFile(suffix=".txt", delete=False) as f:
        f.write(b"This is not a video file")
        f.flush()

        try:
            # Gemini may handle invalid files gracefully, so we just ensure it doesn't crash
            result = provider.process_video(f.name)
            # If it doesn't raise an exception, at least verify we get some result
            assert result is not None
        except Exception as e:
            # Expected behavior - should raise an exception for invalid files
            assert (
                "not a video" in str(e).lower()
                or "invalid" in str(e).lower()
                or "failed" in str(e).lower()
            )
        finally:
            os.unlink(f.name)


def test_video_analysis_fallback_behavior():
    """Test that video analysis degrades gracefully when video processing fails"""

    # Test with text-only fallback when video processing fails
    messages = (
        MessageBuilder()
        .add_system("You are a helpful assistant.")
        .add_user("Analyze this content")
        .build()
    )

    # This should work without video content
    # (We can't easily test actual video processing failure without mocking)
    assert len(messages) == 2
    assert messages[1]["content"] == "Analyze this content"


# ===== PERFORMANCE TESTS =====


@pytest.mark.skipif(not os.getenv("GOOGLE_API_KEY"), reason="Google API key not found")
def test_video_upload_performance():
    """Test that video upload completes within reasonable time"""

    # Skip if we only have a placeholder file
    if _is_placeholder_file(TEST_VIDEO_PATH):
        pytest.skip("No real video file available, only placeholder")

    import time

    provider = GeminiProvider()

    start_time = time.time()

    try:
        _ = provider.process_video(str(TEST_VIDEO_PATH))
        upload_time = time.time() - start_time

        # Should complete within 30 seconds for a small test video
        assert upload_time < 30, f"Video upload took too long: {upload_time:.2f}s"

        print(f"✅ Video upload completed in {upload_time:.2f}s")

    except Exception as e:
        if "not supported" in str(e).lower():
            pytest.skip(f"Video processing not supported: {e}")
        else:
            raise


# ===== INTEGRATION TESTS =====


@pytest.mark.skipif(not os.getenv("GOOGLE_API_KEY"), reason="Google API key not found")
def test_video_analysis_integration():
    """Full integration test of video analysis workflow"""

    # Skip if we only have a placeholder file
    if _is_placeholder_file(TEST_VIDEO_PATH):
        pytest.skip("No real video file available, only placeholder")

    # Test the complete workflow from video upload to analysis
    messages = (
        MessageBuilder()
        .add_system(
            "You are a video analysis expert. Provide concise but informative analysis."
        )
        .add_video(
            str(TEST_VIDEO_PATH),
            "Analyze this video. What objects, colors, and movements do you observe? "
            "Keep your response focused and factual.",
        )
        .build()
    )

    try:
        # Test with different models to ensure compatibility
        models_to_test = [
            "gemini/gemini-2.0-flash-exp",
            "gemini/gemini-1.5-flash",
        ]

        for model in models_to_test:
            try:
                response, cost = call_ai(
                    model=model, messages=messages, temperature=0.1, max_tokens=500
                )

                # Basic validation
                assert isinstance(response, str)
                assert len(response) > 0
                assert cost >= 0

                print(f"✅ Integration test passed for {model}")
                break  # If one model works, that's sufficient for integration test

            except Exception as model_error:
                print(f"⚠️ Model {model} failed: {model_error}")
                continue
        else:
            pytest.skip("No video-capable models available for integration test")

    except Exception as e:
        if "not supported" in str(e).lower():
            pytest.skip(f"Video analysis integration not supported: {e}")
        else:
            raise
