"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.GitlabRunnerAutoscaling = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const gcp = require("@cdktf/provider-google");
const cdktf = require("cdktf");
const constructs_1 = require("constructs");
class GitlabRunnerAutoscaling extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const defaultProps = {
            desiredCapacity: 1,
            machineType: 'e2-medium',
            tags: ['gitlab', 'cdktf', 'runner'],
            gitlabUrl: 'https://gitlab.com/',
            gitlabRunnerImage: 'public.ecr.aws/gitlab/gitlab-runner:alpine',
            preemptible: true,
            automaticRestart: false,
            concurrent: 1,
            downloadGitlabRunnerBinaryUrl: 'https://gitlab-runner-downloads.s3.amazonaws.com/latest/binaries/gitlab-runner-linux-amd64',
        };
        const runnerProps = { ...defaultProps, ...props };
        const network = runnerProps?.computeNetwork ?? new gcp.ComputeNetwork(this, 'Network', {
            name: 'cdktf-gitlabrunner-network',
        });
        const serviceAccount = new gcp.ServiceAccount(this, 'ServiceAccount', {
            accountId: 'gitlab-runner-sa',
            displayName: 'Gitlab Runner Servuce Account',
        });
        new gcp.ComputeFirewall(this, 'GitlabRunnerFirewallRule', {
            priority: 900,
            name: 'allow-ingress-from-iap',
            sourceRanges: ['35.235.240.0/20'],
            network: network.id,
            allow: [{
                    protocol: 'tcp',
                    ports: ['22'],
                }],
            targetTags: ['runner-iap'],
            dependsOn: [network],
        });
        new gcp.ProjectIamBinding(this, 'ComputeIamBinding', {
            provider: runnerProps.provider,
            role: 'roles/compute.admin',
            members: [`serviceAccount:${serviceAccount.email}`],
            project: `${runnerProps.provider.project}`,
        });
        const networkTags = ['runner-iap'];
        if (runnerProps.networkTags) {
            networkTags.push(...runnerProps.networkTags);
        }
        const compute_template = new gcp.ComputeInstanceTemplate(this, 'cdktf-gitlabrunner-instance-template', {
            machineType: runnerProps.machineType,
            disk: [
                {
                    autoDelete: true,
                    boot: true,
                    diskSizeGb: props.defaultDiskSizeGb ?? 60,
                    sourceImage: 'cos-cloud/cos-stable',
                },
            ],
            serviceAccount: runnerProps.serviceAccount ? runnerProps.serviceAccount : { email: serviceAccount.email, scopes: ['cloud-platform'] },
            canIpForward: true,
            description: 'cdktf-gitlabrunner-instance-template',
            name: 'cdktf-gitlabrunner-instance-template',
            metadata: {
                'user-data': this.createMetadataStartupScript(runnerProps).join('\n'),
                'shutdown-script': this.createShutdwonScript().join('\n'),
            },
            provider: runnerProps.provider,
            networkInterface: [
                {
                    network: network.id,
                    accessConfig: [{}],
                },
            ],
            tags: networkTags,
            lifecycle: {
                createBeforeDestroy: true,
            },
            scheduling: {
                preemptible: runnerProps.preemptible,
                automaticRestart: runnerProps.automaticRestart,
            },
        });
        new gcp.ComputeInstanceGroupManager(this, 'instance-group', {
            provider: runnerProps.provider,
            name: 'cdktf-gitlabrunner-instance-group',
            baseInstanceName: 'scaling-gitlab-runner',
            version: [{
                    instanceTemplate: compute_template.id,
                }],
            lifecycle: {
                createBeforeDestroy: true,
            },
            targetSize: runnerProps.desiredCapacity,
            dependsOn: [compute_template, serviceAccount],
        });
        new cdktf.TerraformOutput(this, 'ComputeNetworkName', {
            value: network.name,
        });
    }
    dockerVolumesList(dockerVolume) {
        let tempString = '--docker-volumes "/var/run/docker.sock:/var/run/docker.sock"';
        if (dockerVolume) {
            let tempList = [];
            dockerVolume.forEach(e => {
                tempList.push(`"${e.hostPath}:${e.containerPath}"`);
            });
            tempList.forEach(e => {
                tempString = `${tempString} --docker-volumes ${e}`;
            });
        }
        return tempString;
    }
    /**
     * @param props
     * @returns Array.
     */
    createMetadataStartupScript(props) {
        return [
            `Content-Type: multipart/mixed; boundary="MIMEBOUNDARY"
MIME-Version: 1.0

--MIMEBOUNDARY
Content-Disposition: attachment; filename="init.cfg"
Content-Transfer-Encoding: 7bit
Content-Type: text/cloud-config
Mime-Version: 1.0

`, `# cloud-config
users:
  - name: gitlab-runner
    shell: /bin/bash
    uid: 2000
    groups:
      - docker

write_files:
  - path: /etc/gitlab-runner/config.toml
    owner: root:root
    permissions: '0644'
    content: |
      # Prometheus metrics at /metrics, also used for health checks.
      listen_address = ":9252"
      concurrent = ${props.concurrent}
  - path: /var/lib/cloud/bin/firewall
    permissions: 0755
    owner: root
    content: |
      #! /bin/bash
      iptables -w -A INPUT -p tcp --dport 9252 -j ACCEPT
  - path: /etc/systemd/system/gitlab-runner-register.service
    permissions: 0644
    owner: root
    content: |
      [Unit]
      Description=GitLab Runner Registration/Unregistration
      ConditionFileIsExecutable=/var/lib/google/bin/gitlab-runner
      After=syslog.target network-online.target
      [Service]
      Type=oneshot
      RemainAfterExit=yes
      ExecStart=/var/lib/google/bin/gitlab-runner register --non-interactive --url "${props.gitlabUrl}" --executor "docker" --registration-token ${props.gitlabToken} --docker-image alpine:latest --tag-list "${props.tags?.join(',')}" --locked="false" --access-level="not_protected" ${this.dockerVolumesList(props?.dockerVolumes)} --description "A Runner on GCP GCE (${props.machineType})" --docker-privileged
      ExecStop=/var/lib/google/bin/gitlab-runner "unregister" "--config" "/etc/gitlab-runner/config.toml" "--all-runners"
      [Install]
      WantedBy=multi-user.target
  - path: /etc/systemd/system/gitlab-runner.service
    permissions: 0644
    owner: root
    content: |
      [Unit]
      Description=GitLab Runner
      ConditionFileIsExecutable=/var/lib/google/bin/gitlab-runner
      After=gitlab-runner-register.service syslog.target network-online.target
      Requires=gitlab-runner-register.service
      [Service]
      StartLimitInterval=5
      StartLimitBurst=10
      ExecStart=/var/lib/google/bin/gitlab-runner "run" "--working-directory" "/home/gitlab-runner" "--config" "/etc/gitlab-runner/config.toml" "--service" "gitlab-runner" "--syslog" "--user" "gitlab-runner"
      Restart=always
      RestartSec=120
      [Install]
      WantedBy=multi-user.target
  - path: /etc/systemd/system/firewall.service
    permissions: 0644
    owner: root
    content: |
      [Unit]
      Description=Host firewall configuration
      ConditionFileIsExecutable=/var/lib/cloud/bin/firewall
      After=network-online.target
      [Service]
      ExecStart=/var/lib/cloud/bin/firewall
      Type=oneshot
      [Install]
      WantedBy=multi-user.target

runcmd:
  - mkdir /var/lib/google/tmp
  - mkdir /var/lib/google/bin
  - curl -L --output /var/lib/google/tmp/gitlab-runner ${props.downloadGitlabRunnerBinaryUrl}
  - install -o 0 -g 0 -m 0755 /var/lib/google/tmp/gitlab-runner /var/lib/google/bin/gitlab-runner
  - systemctl daemon-reload
  - systemctl start firewall.service
  - systemctl start gitlab-runner-register.service
  - systemctl start gitlab-runner.service
  - chmod +x /var/run/docker.sock

--MIMEBOUNDARY--
`,
        ];
    }
    createShutdwonScript() {
        return ['#!/bin/bash', '/var/lib/google/bin/gitlab-runner unregister --all-runners'];
    }
}
exports.GitlabRunnerAutoscaling = GitlabRunnerAutoscaling;
_a = JSII_RTTI_SYMBOL_1;
GitlabRunnerAutoscaling[_a] = { fqn: "cdktf-gitlab-runner.GitlabRunnerAutoscaling", version: "0.0.751" };
//# sourceMappingURL=data:application/json;base64,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