import argparse
import os
import secrets
import ssl
import json
from aiohttp import web

# ---------------- 工具函数 ----------------
def load_or_create_token(token_file: str) -> str:
    if os.path.exists(token_file):
        with open(token_file, "r") as f:
            token = f.read().strip()
            if token:
                return token
    token = secrets.token_hex(16)
    with open(token_file, "w") as f:
        f.write(token)
    return token

def load_records(file_path: str) -> dict:
    if os.path.exists(file_path):
        try:
            with open(file_path, "r") as f:
                return json.load(f)
        except Exception:
            return {}
    return {}

def save_records(file_path: str, data: dict):
    with open(file_path, "w") as f:
        json.dump(data, f, indent=2, ensure_ascii=False)

# ---------------- 登录相关 ----------------
async def login_page(request):
    return web.Response(text="""
        <html><body>
        <h2>Login</h2>
        <form method="post" action="/login">
            Token: <input type="password" name="token"/>
            <input type="submit" value="Login"/>
        </form>
        </body></html>
    """, content_type="text/html")

async def login_handler(request):
    data = await request.post()
    token = data.get("token", "")
    if token == request.app["token"]:
        resp = web.HTTPFound("/")
        resp.set_cookie("auth", token, httponly=True)
        return resp
    return web.Response(text="Invalid token", status=401)

def check_cookie(request):
    return request.cookies.get("auth") == request.app["token"]

def check_token(request):
    auth = request.headers.get("Authorization", "")
    return auth.startswith("Bearer ") and auth.split(" ", 1)[1] == request.app["token"]

# ---------------- 页面操作（Cookie 验证） ----------------
async def list_handler(request):
    if not check_cookie(request):
        return web.HTTPFound("/login")

    records = load_records(request.app['data_file'])
    html = ["<html><body><h2>URL List</h2><ul>"]
    for name, url in records.items():
        html.append(
            f'<li>{name}: <a href="/{name}" target="_blank">{url}</a> '
            f'[<a href="/delete/{name}">删除</a>]</li>'
        )
    html.append("</ul></body></html>")
    return web.Response(text="".join(html), content_type="text/html")

async def delete_handler(request):
    if not check_cookie(request):
        return web.HTTPFound("/login")

    records = load_records(request.app['data_file'])
    name = request.match_info.get("name")
    if name in records:
        del records[name]
        save_records(request.app["data_file"], records)
        return web.HTTPFound("/")
    return web.Response(text="not found", status=404)

# ---------------- API 接口（Token 验证） ----------------
async def set_handler(request):
    if not check_token(request):
        return web.json_response({"error": "unauthorized"}, status=401)

    data = await request.json()
    name, url = data.get("name"), data.get("url")
    if not name or not url:
        return web.json_response({"error": "name and url required"}, status=400)

    records = load_records(request.app['data_file'])
    records[name] = url
    save_records(request.app["data_file"], records)
    return web.json_response({"status": "ok", "data": {name: url}})

# ---------------- 动态跳转 ----------------
async def redirect_handler(request):
    name = request.match_info.get("name")
    records = load_records(request.app['data_file'])
    url = records.get(name)
    if url:
        raise web.HTTPFound(url)
    return web.Response(text="not found", status=404)

# ---------------- 主程序 ----------------
def main():
    parser = argparse.ArgumentParser()
    parser.add_argument("--host", default="0.0.0.0")
    parser.add_argument("--port", type=int, default=8080)
    parser.add_argument("--ssl-cert")
    parser.add_argument("--ssl-key")
    parser.add_argument("--token-file", default="token.txt")
    parser.add_argument("--data-file", default="records.json", help="URL 数据存储文件")
    args = parser.parse_args()

    token = load_or_create_token(args.token_file)
    print(f"[INFO] Token loaded from {args.token_file}: {token}")


    ssl_context = None
    if args.ssl_cert and args.ssl_key:
        ssl_context = ssl.create_default_context(ssl.Purpose.CLIENT_AUTH)
        ssl_context.load_cert_chain(args.ssl_cert, args.ssl_key)

    app = web.Application()
    app["token"] = token
    app["data_file"] = args.data_file

    app.router.add_get("/login", login_page)
    app.router.add_post("/login", login_handler)
    app.router.add_get("/", list_handler)
    app.router.add_get("/delete/{name}", delete_handler)
    app.router.add_post("/set", set_handler)
    app.router.add_get("/{name}", redirect_handler)  # 动态跳转，必须放最后

    web.run_app(app, host=args.host, port=args.port, ssl_context=ssl_context)

if __name__ == "__main__":
    main()
