# LangChat

> **Production-ready AI chatbots in minutes, not months**

A powerful, modular conversational AI library with vector search capabilities, designed to help developers build production-ready AI chatbots with minimal effort.

## 🎯 Motivation

LangChat was created to solve the complexity of building production-ready conversational AI systems. Instead of managing multiple libraries, API integrations, vector databases, and chat history tracking separately, LangChat provides a unified, modular architecture that handles all these concerns out of the box.

### Why LangChat?

- **🚀 Fast Development**: Get a production-ready chatbot running in minutes
- **🔧 Modular Architecture**: Mix and match components as needed
- **🔄 Automatic API Key Rotation**: Built-in fault tolerance for OpenAI API
- **📊 Production Ready**: Includes chat history, metrics, and feedback tracking
- **🎨 Highly Customizable**: Easy to extend with custom prompts and adapters
- **🐳 Docker Ready**: Auto-generated Dockerfile for easy deployment

## Features

### 🤖 LLM Integration
- **OpenAI**: Native OpenAI API support with automatic API key rotation
- **Fault Tolerant**: Automatic retry logic with multiple API keys

### 🔍 Vector Search
- **Pinecone Integration**: Seamless vector database integration
- **Reranking**: Flashrank reranker for improved search results
- **Auto-Configured Models**: Rerank models automatically stored in `src/langchat/models/rerank_models`

### 💾 Database Management
- **Supabase**: Built-in Supabase integration
- **ID Management**: Automatic ID generation with conflict resolution
- **Session Management**: User-specific chat history and memory

### 🎨 Customization
- **Custom Prompts**: Configure both system prompts and standalone question prompts
- **Flexible Configuration**: Environment variables or direct configuration
- **Port Configuration**: Configurable server port (auto-set in Dockerfile)

### 🚀 Developer Experience
- **Auto-Generated Interface**: Chat interface HTML auto-created on server startup
- **Auto-Generated Dockerfile**: Dockerfile auto-created with correct port configuration
- **Easy Setup**: Simple configuration and initialization

## Structure

The library is organized in the `src/langchat/` directory with the following structure:

```
src/langchat/
├── __init__.py              # Main exports
├── main.py                  # Developer entry point (no API)
├── config.py                # Configuration module
├── adapters/                # External service adapters
│   ├── supabase/            # Supabase adapters
│   │   ├── supabase_adapter.py
│   │   └── id_manager.py
│   ├── services/            # LLM service providers
│   │   └── openai_service.py
│   ├── vector_db/           # Vector database adapters
│   │   └── pinecone_adapter.py
│   └── reranker/            # Reranker adapters
│       └── flashrank_adapter.py
├── core/                    # Core functionality
│   ├── engine.py            # Main LangChat engine
│   ├── session.py           # User session management
│   └── prompts.py           # Prompt templates
├── api/                     # FastAPI routes
│   ├── app.py               # FastAPI app setup
│   ├── routes.py            # API routes
│   └── models.py            # Pydantic models
├── utils/                   # Utility functions
│   ├── interface_generator.py  # Auto-generate chat interface
│   └── docker_generator.py      # Auto-generate Dockerfile
└── models/                  # Model storage
    └── rerank_models/       # Reranker models (auto-configured)
```

## Installation

Install LangChat from PyPI:

```bash
pip install langchat
```

Or install in development mode:

```bash
pip install -e .
```

Or install dependencies directly:

```bash
pip install -r requirements.txt
```

## Quick Start

### Installation

Install LangChat from PyPI:

```bash
pip install langchat
```

### Basic Usage

#### As a Library

```python
import asyncio
from langchat import LangChat, LangChatConfig

async def main():
    # Create configuration
    config = LangChatConfig(
        openai_api_keys=["your-openai-api-key"],
        pinecone_api_key="your-pinecone-api-key",
        pinecone_index_name="your-index-name",
        supabase_url="your-supabase-url",
        supabase_key="your-supabase-key"
    )
    
    # Initialize LangChat
    langchat = LangChat(config=config)
    
    # Chat with the AI
    result = await langchat.chat(
        query="What are the best universities in Europe?",
        user_id="user123",
        domain="education"
    )
    
    print(result["response"])
    print(f"Status: {result['status']}")
    print(f"Response time: {result.get('response_time', 0):.2f}s")

# Run the example
asyncio.run(main())
```

#### With Custom Prompts

```python
config = LangChatConfig(
    openai_api_keys=["your-api-key"],
    pinecone_api_key="your-pinecone-key",
    pinecone_index_name="your-index",
    supabase_url="your-supabase-url",
    supabase_key="your-supabase-key",
    
    # Custom system prompt
    system_prompt_template="""You are a helpful assistant specializing in {domain}.
    
    Use the following context to answer questions:
    {context}
    
    Chat history: {chat_history}
    
    Question: {question}
    Answer:""",
    
    # Custom standalone question prompt
    standalone_question_prompt="""Convert this question to a standalone search query.
    
    Chat History: {chat_history}
    Question: {question}
    Standalone query:"""
)
```

#### As an API Server

Run the FastAPI server (auto-generates `chat_interface.html` and `Dockerfile`):

```python
# main.py
from langchat.api.app import create_app
from langchat.config import LangChatConfig
import uvicorn

# Create configuration
config = LangChatConfig(
    openai_api_keys=["your-openai-api-key"],
    pinecone_api_key="your-pinecone-api-key",
    pinecone_index_name="your-index-name",
    supabase_url="your-supabase-url",
    supabase_key="your-supabase-key",
    server_port=8007  # Your server port
)

# Create FastAPI app (auto-generates chat_interface.html and Dockerfile)
app = create_app(
    config=config,
    auto_generate_interface=True,
    auto_generate_docker=True
)

if __name__ == "__main__":
    # Start the server
    print(f"🚀 Starting LangChat API server on port {config.server_port}")
    print(f"📱 Chat interface available at: http://localhost:{config.server_port}/frontend")
    uvicorn.run(app, host="0.0.0.0", port=config.server_port)
```

After starting the server:
- **API Endpoint**: `http://localhost:8007/chat`
- **Frontend Interface**: `http://localhost:8007/frontend`
- **Health Check**: `http://localhost:8007/health`

Or use environment variables:

```bash
export OPENAI_API_KEYS="key1,key2"
export PINECONE_API_KEY="your-key"
export SUPABASE_URL="your-url"
export SUPABASE_KEY="your-key"

python main.py
```

## Configuration

All configuration can be done through `LangChatConfig`:

- **OpenAI**: API keys, model, temperature, embedding model
- **Pinecone**: API key, index name (must be configured)
- **Supabase**: URL, API key
- **Vector Search**: Retrieval parameters, reranker settings (auto-configured)
- **Session**: Chat history limits, memory window
- **Timezone**: For date/time formatting
- **Prompts**: Custom system prompts and standalone question prompts
- **Server**: Port configuration

### Configuration Examples

#### Basic Configuration

```python
config = LangChatConfig(
    openai_api_keys=["sk-...", "sk-..."],  # Multiple keys for rotation
    openai_model="gpt-4o-mini",
    openai_temperature=1.0,
    openai_embedding_model="text-embedding-3-large",
    pinecone_api_key="pcsk-...",
    pinecone_index_name="my-index",
    supabase_url="https://xxxxx.supabase.co",
    supabase_key="eyJhbGc...",
    server_port=8000
)
```

#### Using Environment Variables

```bash
export OPENAI_API_KEYS="key1,key2"
export OPENAI_MODEL="gpt-4o-mini"
export PINECONE_API_KEY="your-key"
export PINECONE_INDEX_NAME="your-index"
export SUPABASE_URL="your-url"
export SUPABASE_KEY="your-key"
export SERVER_PORT=8000
```

```python
# Load from environment variables
config = LangChatConfig.from_env()
```

#### Custom Prompts

```python
config = LangChatConfig(
    system_prompt_template="Your custom system prompt here...",
    standalone_question_prompt="Your custom standalone question prompt...",
    # ... other config ...
)
```

## Examples

See [examples/](examples/) for detailed usage examples:

- **[Travel AI Example](examples/travel_ai.py)**: Custom chatbot for travel assistance
- **[main.py](main.py)**: Full API server example with custom prompts

### Example: Education Chatbot

```python
config = LangChatConfig(
    openai_api_keys=["your-key"],
    pinecone_api_key="your-key",
    pinecone_index_name="education-index",
    supabase_url="your-url",
    supabase_key="your-key",
    system_prompt_template="""You are an expert education consultant.
    
    Help students find the best universities based on their profiles.
    
    Context: {context}
    History: {chat_history}
    Question: {question}
    Answer:""",
    retrieval_k=10,
    reranker_top_n=5
)
```

## Customization

The library is designed to be easily customizable:

1. **Custom LLM Providers**: Add new providers in `adapters/services/`
2. **Custom Vector DBs**: Add new adapters in `adapters/vector_db/`
3. **Custom Prompts**: Override `system_prompt_template` and `standalone_question_prompt` in config
4. **Custom Sessions**: Extend `UserSession` in `core/session.py`
5. **Auto-Generated Files**: Chat interface and Dockerfile auto-generated on startup

## API Endpoints

When running as an API server:

- `POST /chat` - Send a chat message
  - Body: `query`, `userId`, `domain`
- `GET /frontend` - Access the chat interface
- `GET /health` - Health check endpoint
- `POST /feedback` - Submit feedback

## Development

The library follows a modular architecture:

- **Adapters**: Handle external services (DBs, LLMs, Vector DBs)
- **Core**: Main business logic (Engine, Sessions, Prompts)
- **API**: FastAPI routes for HTTP API

Each module can be used independently or together.

## Contributing

We welcome contributions! Please see [CONTRIBUTING.md](CONTRIBUTING.md) for guidelines.

## License

This project is licensed under the MIT License - see the [LICENSE](LICENSE) file for details.

## Documentation

See [docs/](docs/) for detailed documentation:
- [Configuration Guide](docs/configuration.md)
- [API Reference](docs/api_reference.md) (coming soon)
