import os
import sys
import subprocess
import time
import logging
from typing import List, Dict, Optional, Union, BinaryIO
from dataclasses import dataclass
from enum import Enum, auto
import threading
import queue
from concurrent.futures import ThreadPoolExecutor
import hashlib

class BluetoothError(Exception):
    """蓝牙操作异常基类"""
    pass

class BluetoothState(Enum):
    """蓝牙状态枚举"""
    DISABLED = auto()
    ENABLED = auto()
    CONNECTING = auto()
    CONNECTED = auto()
    PAIRING = auto()
    PAIRED = auto()
    ERROR = auto()

class TransferPriority(Enum):
    """传输优先级"""
    LOW = auto()
    NORMAL = auto()
    HIGH = auto()
    URGENT = auto()

@dataclass
class BluetoothDevice:
    """蓝牙设备信息类"""
    name: str
    id: str
    address: Optional[str] = None
    device_class: Optional[str] = None
    connected: bool = False
    paired: bool = False
    signal_strength: Optional[int] = None
    last_seen: Optional[float] = None

@dataclass
class TransferTask:
    """传输任务类"""
    device_id: str
    file_path: str
    priority: TransferPriority = TransferPriority.NORMAL
    chunk_size: int = 8192  # 默认8KB块大小
    retry_count: int = 3
    checksum: Optional[str] = None

class BluetoothTransfer:
    """优化的蓝牙传输功能类"""
    
    def __init__(self, max_workers: int = 3, default_chunk_size: int = 65536):
        """
        初始化传输管理器
        
        Args:
            max_workers: 最大并行传输任务数
            default_chunk_size: 默认传输块大小（字节）
        """
        self.logger = logging.getLogger(__name__)
        self._setup_logging()
        self.max_workers = max_workers
        self.default_chunk_size = default_chunk_size
        self._transfer_queue = queue.PriorityQueue()
        self._executor = ThreadPoolExecutor(max_workers=max_workers)
        self._active_transfers: Dict[str, threading.Event] = {}
        self._transfer_stats: Dict[str, Dict] = {}

    def _setup_logging(self):
        """设置日志记录"""
        if not self.logger.handlers:
            handler = logging.StreamHandler()
            formatter = logging.Formatter(
                '%(asctime)s - %(name)s - %(levelname)s - %(message)s')
            handler.setFormatter(formatter)
            self.logger.addHandler(handler)
            self.logger.setLevel(logging.INFO)

    def _calculate_checksum(self, file_path: str) -> str:
        """计算文件校验和"""
        hash_md5 = hashlib.md5()
        with open(file_path, "rb") as f:
            for chunk in iter(lambda: f.read(4096), b""):
                hash_md5.update(chunk)
        return hash_md5.hexdigest()

    def _get_optimal_chunk_size(self, file_size: int) -> int:
        """
        根据文件大小获取最优传输块大小
        
        Args:
            file_size: 文件大小（字节）
            
        Returns:
            int: 最优块大小
        """
        if file_size < 1024 * 1024:  # 1MB以下
            return 4096
        elif file_size < 10 * 1024 * 1024:  # 10MB以下
            return 8192
        elif file_size < 100 * 1024 * 1024:  # 100MB以下
            return 16384
        else:  # 100MB以上
            return 32768

    def _transfer_chunk(self, device_id: str, chunk: bytes, chunk_num: int) -> bool:
        """
        传输单个数据块
        
        Args:
            device_id: 目标设备ID
            chunk: 数据块
            chunk_num: 块编号
            
        Returns:
            bool: 传输是否成功
        """
        try:
            cmd = (f'Send-BluetoothChunk -DeviceId "{device_id}" '
                  f'-ChunkData ([byte[]]@({",".join(str(b) for b in chunk)})) '
                  f'-ChunkNumber {chunk_num}')
            result = subprocess.run(['powershell', '-Command', cmd],
                                  capture_output=True, text=True, timeout=10)
            return result.returncode == 0
        except Exception as e:
            self.logger.error(f"Chunk transfer failed: {str(e)}")
            return False

    def _transfer_file_worker(self, task: TransferTask) -> bool:
        """
        文件传输工作线程
        
        Args:
            task: 传输任务
            
        Returns:
            bool: 传输是否成功
        """
        try:
            file_size = os.path.getsize(task.file_path)
            chunk_size = self._get_optimal_chunk_size(file_size)
            transferred = 0
            start_time = time.time()
            
            # 计算并验证校验和
            if not task.checksum:
                task.checksum = self._calculate_checksum(task.file_path)
            
            # 更新传输统计
            self._transfer_stats[task.device_id] = {
                'total_size': file_size,
                'transferred': 0,
                'start_time': start_time,
                'speed': 0
            }
            
            with open(task.file_path, 'rb') as f:
                chunk_num = 0
                while True:
                    chunk = f.read(chunk_size)
                    if not chunk:
                        break
                        
                    # 重试机制
                    retry_count = 0
                    while retry_count < task.retry_count:
                        if self._transfer_chunk(task.device_id, chunk, chunk_num):
                            break
                        retry_count += 1
                        time.sleep(0.1)  # 重试前等待
                    
                    if retry_count >= task.retry_count:
                        self.logger.error(f"Failed to transfer chunk {chunk_num}")
                        return False
                    
                    transferred += len(chunk)
                    chunk_num += 1
                    
                    # 更新传输统计
                    elapsed_time = time.time() - start_time
                    if elapsed_time > 0:
                        self._transfer_stats[task.device_id].update({
                            'transferred': transferred,
                            'speed': transferred / elapsed_time
                        })
            
            # 验证传输完整性
            if not self._verify_transfer(task.device_id, task.checksum):
                self.logger.error("File verification failed")
                return False
                
            return True
            
        except Exception as e:
            self.logger.error(f"File transfer failed: {str(e)}")
            return False
        finally:
            # 清理传输统计
            if task.device_id in self._transfer_stats:
                del self._transfer_stats[task.device_id]

    def _verify_transfer(self, device_id: str, expected_checksum: str) -> bool:
        """
        验证传输文件的完整性
        
        Args:
            device_id: 设备ID
            expected_checksum: 期望的校验和
            
        Returns:
            bool: 验证是否成功
        """
        try:
            cmd = f'Get-BluetoothFileChecksum -DeviceId "{device_id}"'
            result = subprocess.run(['powershell', '-Command', cmd],
                                  capture_output=True, text=True)
            if result.returncode == 0:
                received_checksum = result.stdout.strip()
                return received_checksum == expected_checksum
            return False
        except Exception as e:
            self.logger.error(f"Transfer verification failed: {str(e)}")
            return False

    def send_file(self, device_id: str, file_path: str, 
                  priority: TransferPriority = TransferPriority.NORMAL) -> bool:
        """
        发送文件到蓝牙设备（优化版）
        
        Args:
            device_id: 目标设备ID
            file_path: 要发送的文件路径
            priority: 传输优先级
            
        Returns:
            bool: 发送是否成功
        """
        if not os.path.exists(file_path):
            self.logger.error("File not found")
            return False
            
        task = TransferTask(
            device_id=device_id,
            file_path=file_path,
            priority=priority
        )
        
        # 提交传输任务
        future = self._executor.submit(self._transfer_file_worker, task)
        return future.result()

    def get_transfer_stats(self, device_id: str) -> Optional[Dict]:
        """
        获取传输统计信息
        
        Args:
            device_id: 设备ID
            
        Returns:
            Optional[Dict]: 传输统计信息
        """
        return self._transfer_stats.get(device_id)

    def cancel_transfer(self, device_id: str) -> bool:
        """
        取消传输任务
        
        Args:
            device_id: 设备ID
            
        Returns:
            bool: 取消是否成功
        """
        if device_id in self._active_transfers:
            self._active_transfers[device_id].set()
            return True
        return False

    def optimize_connection(self, device_id: str) -> bool:
        """
        优化蓝牙连接参数
        
        Args:
            device_id: 设备ID
            
        Returns:
            bool: 优化是否成功
        """
        try:
            cmd = (f'Optimize-BluetoothConnection -DeviceId "{device_id}" '
                   '-MaxPacketSize 1024 -MinInterval 7.5ms')
            result = subprocess.run(['powershell', '-Command', cmd],
                                  capture_output=True, text=True)
            return result.returncode == 0
        except Exception as e:
            self.logger.error(f"Connection optimization failed: {str(e)}")
            return False

class BluetoothManager:
    """Windows蓝牙设备管理类"""
    
    def __init__(self):
        self.logger = logging.getLogger(__name__)
        self._setup_logging()
        self._devices_cache: Dict[str, BluetoothDevice] = {}
        self._transfer = BluetoothTransfer()

    def _setup_logging(self):
        """设置日志记录"""
        if not self.logger.handlers:
            handler = logging.StreamHandler()
            formatter = logging.Formatter(
                '%(asctime)s - %(name)s - %(levelname)s - %(message)s')
            handler.setFormatter(formatter)
            self.logger.addHandler(handler)
            self.logger.setLevel(logging.INFO)

    def _execute_powershell_command(self, command: str) -> Optional[str]:
        """
        执行PowerShell命令并返回结果
        
        Args:
            command: 要执行的PowerShell命令
            
        Returns:
            Optional[str]: 命令输出，出错时返回None
        """
        try:
            result = subprocess.run(['powershell', '-Command', command],
                                  capture_output=True, text=True, timeout=30)
            if result.returncode != 0:
                self.logger.error(f"PowerShell command failed: {result.stderr}")
                return None
            return result.stdout
        except subprocess.TimeoutExpired:
            self.logger.error("PowerShell command timed out")
            return None
        except Exception as e:
            self.logger.error(f"Error executing PowerShell command: {str(e)}")
            return None

    def discover_devices(self, duration: int = 10, update_cache: bool = True) -> List[BluetoothDevice]:
        """
        发现附近的蓝牙设备（优化版）
        
        Args:
            duration: 扫描持续时间（秒）
            update_cache: 是否更新设备缓存
            
        Returns:
            List[BluetoothDevice]: 发现的设备列表
        """
        try:
            devices = []
            start_time = time.time()
            
            while time.time() - start_time < duration:
                cmd = ('Get-PnpDevice -Class Bluetooth | '
                      'Where-Object {$_.Status -eq "OK"} | '
                      'Select-Object FriendlyName, DeviceID, ClassGuid')
                
                output = self._execute_powershell_command(cmd)
                if not output:
                    continue
                    
                lines = output.strip().split('\n')[2:]  # 跳过标题行
                for line in lines:
                    if not line.strip():
                        continue
                        
                    parts = line.split()
                    if len(parts) >= 2:
                        device = BluetoothDevice(
                            name=' '.join(parts[:-1]),
                            id=parts[-1],
                            last_seen=time.time()
                        )
                        
                        # 更新缓存
                        if update_cache:
                            if device.id in self._devices_cache:
                                self._devices_cache[device.id].last_seen = device.last_seen
                            else:
                                self._devices_cache[device.id] = device
                                
                        devices.append(device)
                
                time.sleep(1)  # 避免过于频繁的扫描
                
            return devices
        except Exception as e:
            self.logger.error(f"Device discovery failed: {str(e)}")
            return []

    def get_device_state(self, device_id: str) -> BluetoothState:
        """
        获取设备状态
        
        Args:
            device_id: 设备ID
            
        Returns:
            BluetoothState: 设备状态
        """
        try:
            cmd = f'Get-PnpDevice -Id "{device_id}" | Select-Object Status'
            output = self._execute_powershell_command(cmd)
            
            if not output:
                return BluetoothState.ERROR
                
            if "OK" in output:
                return BluetoothState.CONNECTED
            elif "Error" in output:
                return BluetoothState.ERROR
            else:
                return BluetoothState.DISABLED
                
        except Exception as e:
            self.logger.error(f"Failed to get device state: {str(e)}")
            return BluetoothState.ERROR

    def send_file(self, device_id: str, file_path: str, 
                  priority: TransferPriority = TransferPriority.NORMAL) -> bool:
        """
        发送文件到蓝牙设备
        
        Args:
            device_id: 目标设备ID
            file_path: 要发送的文件路径
            priority: 传输优先级
            
        Returns:
            bool: 发送是否成功
        """
        return self._transfer.send_file(device_id, file_path, priority)

    def get_transfer_stats(self, device_id: str) -> Optional[Dict]:
        """
        获取传输统计信息
        
        Args:
            device_id: 设备ID
            
        Returns:
            Optional[Dict]: 传输统计信息
        """
        return self._transfer.get_transfer_stats(device_id)

    def cancel_transfer(self, device_id: str) -> bool:
        """
        取消传输任务
        
        Args:
            device_id: 设备ID
            
        Returns:
            bool: 取消是否成功
        """
        return self._transfer.cancel_transfer(device_id)

    def get_device_info(self, device_id: str) -> Optional[BluetoothDevice]:
        """
        获取设备详细信息
        
        Args:
            device_id: 设备ID
            
        Returns:
            Optional[BluetoothDevice]: 设备信息，不存在时返回None
        """
        try:
            cmd = (f'Get-PnpDevice -Id "{device_id}" | '
                   'Select-Object FriendlyName, DeviceID, ClassGuid, Status')
            output = self._execute_powershell_command(cmd)
            
            if not output:
                return None
                
            lines = output.strip().split('\n')[2:]
            for line in lines:
                if line.strip():
                    parts = line.split()
                    if len(parts) >= 2:
                        return BluetoothDevice(
                            name=' '.join(parts[:-2]),
                            id=parts[-2],
                            device_class=parts[-1],
                            connected="OK" in output
                        )
            return None
        except Exception as e:
            self.logger.error(f"Failed to get device info: {str(e)}")
            return None

    def cleanup_cache(self, max_age: float = 300) -> None:
        """
        清理过期的设备缓存
        
        Args:
            max_age: 最大缓存时间（秒）
        """
        current_time = time.time()
        expired_devices = [
            device_id for device_id, device in self._devices_cache.items()
            if current_time - device.last_seen > max_age
        ]
        
        for device_id in expired_devices:
            del self._devices_cache[device_id]
            
        self.logger.info(f"Cleaned up {len(expired_devices)} expired devices from cache")

    def is_bluetooth_enabled(self) -> bool:
        """检查蓝牙是否启用"""
        try:
            result = subprocess.run(['powershell', '-Command', 
                'Get-Service -Name bthserv | Select-Object Status'], 
                capture_output=True, text=True)
            return 'Running' in result.stdout
        except Exception:
            return False

    def enable_bluetooth(self) -> bool:
        """启用蓝牙服务"""
        try:
            subprocess.run(['powershell', '-Command', 'Start-Service bthserv'], 
                         check=True)
            return True
        except subprocess.CalledProcessError:
            return False

    def disable_bluetooth(self) -> bool:
        """禁用蓝牙服务"""
        try:
            subprocess.run(['powershell', '-Command', 'Stop-Service bthserv'], 
                         check=True)
            return True
        except subprocess.CalledProcessError:
            return False

    def pair_device(self, device_id: str) -> bool:
        """
        配对蓝牙设备
        
        Args:
            device_id: 设备ID
            
        Returns:
            bool: 配对是否成功
        """
        try:
            subprocess.run(['powershell', '-Command', 
                f'Add-BluetoothDevice -DeviceId "{device_id}"'], 
                check=True)
            return True
        except subprocess.CalledProcessError:
            return False

    def unpair_device(self, device_id: str) -> bool:
        """
        取消配对蓝牙设备
        
        Args:
            device_id: 设备ID
            
        Returns:
            bool: 取消配对是否成功
        """
        try:
            subprocess.run(['powershell', '-Command', 
                f'Remove-BluetoothDevice -DeviceId "{device_id}"'], 
                check=True)
            return True
        except subprocess.CalledProcessError:
            return False

    def connect_device(self, device_id: str) -> bool:
        """
        连接到已配对的蓝牙设备
        
        Args:
            device_id: 设备ID
            
        Returns:
            bool: 连接是否成功
        """
        try:
            subprocess.run(['powershell', '-Command', 
                f'Connect-BluetoothDevice -DeviceId "{device_id}"'], 
                check=True)
            return True
        except subprocess.CalledProcessError:
            return False

    def disconnect_device(self, device_id: str) -> bool:
        """
        断开蓝牙设备连接
        
        Args:
            device_id: 设备ID
            
        Returns:
            bool: 断开是否成功
        """
        try:
            subprocess.run(['powershell', '-Command', 
                f'Disconnect-BluetoothDevice -DeviceId "{device_id}"'], 
                check=True)
            return True
        except subprocess.CalledProcessError:
            return False

    def optimize_connection(self, device_id: str) -> bool:
        """
        优化蓝牙连接参数
        
        Args:
            device_id: 设备ID
            
        Returns:
            bool: 优化是否成功
        """
        return self._transfer.optimize_connection(device_id)
