import socket
import subprocess
import platform
import time
import win32api
import win32net
import win32netcon
import win32security
import win32file
import win32pipe
import win32event
import pywintypes
import struct
import threading
import queue
import psutil
import win32ras
import win32gui
import win32con
import win32com.client
import json
import re
from collections import defaultdict
from ctypes import *
from ctypes.wintypes import *

# Windows API 常量
AF_INET = 2
AF_INET6 = 23
SOCK_RAW = 3
IPPROTO_IP = 0
IP_HDRINCL = 2
SIO_RCVALL = 0x98000001
MIB_TCP_STATE_ESTAB = 5

class IPHeader(Structure):
    _fields_ = [
        ("ver_ihl", c_ubyte),
        ("tos", c_ubyte),
        ("total_len", c_ushort),
        ("id", c_ushort),
        ("frag_off", c_ushort),
        ("ttl", c_ubyte),
        ("protocol", c_ubyte),
        ("check", c_ushort),
        ("saddr", c_uint),
        ("daddr", c_uint)
    ]

class TCPRow(Structure):
    _fields_ = [
        ("state", c_ulong),
        ("local_addr", c_ulong),
        ("local_port", c_ulong),
        ("remote_addr", c_ulong),
        ("remote_port", c_ulong)
    ]

class WinNetHandler:
    def __init__(self, timeout=10):
        self.timeout = timeout
        self.raw_socket = None
        self.is_capturing = False
        self.packet_queue = queue.Queue()
        self.capture_thread = None
        
    def set_timeout(self, timeout):
        """设置超时时间"""
        self.timeout = timeout
        
    def get_network_interfaces(self):
        """获取所有网络接口"""
        try:
            interfaces = []
            for interface in win32net.NetServerEnum(None, 100, None):
                interfaces.append({
                    'name': interface['name'],
                    'comment': interface.get('comment', ''),
                    'type': interface['type']
                })
            return interfaces
        except:
            return []
            
    def get_active_connections(self):
        """获取所有活动连接"""
        try:
            connections = []
            tcp_table = (TCPRow * 1024)()
            size = c_ulong(sizeof(tcp_table))
            windll.iphlpapi.GetTcpTable(tcp_table, byref(size), False)
            
            for i in range(tcp_table[0].dwNumEntries):
                row = tcp_table[i]
                if row.dwState == MIB_TCP_STATE_ESTAB:
                    connections.append({
                        'local_addr': socket.inet_ntoa(struct.pack('!L', row.dwLocalAddr)),
                        'local_port': socket.htons(row.dwLocalPort),
                        'remote_addr': socket.inet_ntoa(struct.pack('!L', row.dwRemoteAddr)),
                        'remote_port': socket.htons(row.dwRemotePort),
                        'state': row.dwState
                    })
            return connections
        except:
            return []
            
    def get_local_ip(self):
        """获取本机IP地址"""
        try:
            hostname = win32api.GetComputerName()
            addr_info = socket.getaddrinfo(hostname, None)
            return [addr[4][0] for addr in addr_info if ':' not in addr[4][0]]
        except:
            return []
            
    def get_mac_address(self, ip):
        """获取指定IP的MAC地址"""
        try:
            result = subprocess.check_output(f'arp -a {ip}', shell=True)
            lines = result.decode().split('\n')
            for line in lines:
                if ip in line:
                    mac = line.split()[1]
                    return mac
            return None
        except:
            return None
            
    def create_raw_socket(self):
        """创建原始套接字"""
        try:
            self.raw_socket = socket.socket(AF_INET, SOCK_RAW, IPPROTO_IP)
            self.raw_socket.setsockopt(IPPROTO_IP, IP_HDRINCL, 1)
            self.raw_socket.bind((self.get_local_ip()[0], 0))
            self.raw_socket.ioctl(SIO_RCVALL, struct.pack("I", 1))
            return True
        except:
            return False
            
    def _capture_worker(self):
        """数据包捕获工作线程"""
        while self.is_capturing:
            try:
                packet = self.raw_socket.recvfrom(65535)[0]
                ip_header = IPHeader.from_buffer_copy(packet[:20])
                packet_info = {
                    'source_ip': socket.inet_ntoa(struct.pack("!I", ip_header.saddr)),
                    'dest_ip': socket.inet_ntoa(struct.pack("!I", ip_header.daddr)),
                    'protocol': ip_header.protocol,
                    'ttl': ip_header.ttl,
                    'data': packet[20:],
                    'timestamp': time.time()
                }
                self.packet_queue.put(packet_info)
            except:
                pass
                
    def start_capture(self):
        """开始捕获数据包"""
        if not self.raw_socket:
            if not self.create_raw_socket():
                return False
                
        self.is_capturing = True
        self.capture_thread = threading.Thread(target=self._capture_worker)
        self.capture_thread.daemon = True
        self.capture_thread.start()
        return True
        
    def get_captured_packets(self):
        """获取捕获的数据包"""
        packets = []
        while not self.packet_queue.empty():
            packets.append(self.packet_queue.get())
        return packets
        
    def stop_capture(self):
        """停止捕获数据包"""
        self.is_capturing = False
        if self.capture_thread:
            self.capture_thread.join(timeout=1)
        if self.raw_socket:
            self.raw_socket.close()
            self.raw_socket = None
            
    def modify_packet(self, packet, src_ip=None, dst_ip=None, ttl=None):
        """修改数据包"""
        try:
            ip_header = IPHeader.from_buffer_copy(packet[:20])
            
            if src_ip:
                ip_header.saddr = struct.unpack("!I", socket.inet_aton(src_ip))[0]
            if dst_ip:
                ip_header.daddr = struct.unpack("!I", socket.inet_aton(dst_ip))[0]
            if ttl is not None:
                ip_header.ttl = ttl
                
            ip_header.check = 0
            packet = bytes(ip_header) + packet[20:]
            return packet
        except:
            return None
            
    def send_packet(self, packet):
        """发送数据包"""
        try:
            if not self.raw_socket:
                self.create_raw_socket()
            self.raw_socket.send(packet)
            return True
        except:
            return False
            
    def check_port(self, host, port):
        """检查端口状态"""
        try:
            sock = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
            sock.settimeout(self.timeout)
            result = sock.connect_ex((host, port))
            sock.close()
            return result == 0
        except:
            return False
            
    def ping(self, host, count=1):
        """测试网络延迟"""
        try:
            cmd = ['ping', '-n', str(count), host]
            output = subprocess.check_output(cmd, universal_newlines=True)
            times = []
            for line in output.split('\n'):
                if '时间=' in line:
                    time_str = line.split('时间=')[1].split('ms')[0]
                    times.append(float(time_str))
            return {
                'times': times,
                'avg': sum(times) / len(times) if times else None,
                'min': min(times) if times else None,
                'max': max(times) if times else None
            }
        except:
            return None
            
    def get_route_table(self):
        """获取路由表"""
        try:
            result = subprocess.check_output('route print', shell=True)
            routes = []
            for line in result.decode().split('\n'):
                if '0.0.0.0' in line or '::' in line:
                    parts = line.split()
                    if len(parts) >= 5:
                        routes.append({
                            'destination': parts[0],
                            'netmask': parts[1],
                            'gateway': parts[2],
                            'interface': parts[3],
                            'metric': parts[4]
                        })
            return routes
        except:
            return []
            
    def add_route(self, destination, netmask, gateway, metric=1):
        """添加路由"""
        try:
            cmd = f'route add {destination} mask {netmask} {gateway} metric {metric}'
            subprocess.check_call(cmd, shell=True)
            return True
        except:
            return False
            
    def delete_route(self, destination):
        """删除路由"""
        try:
            cmd = f'route delete {destination}'
            subprocess.check_call(cmd, shell=True)
            return True
        except:
            return False

class VPNManager:
    def __init__(self):
        self.connections = {}
        self.traffic_stats = defaultdict(lambda: {'sent': 0, 'recv': 0})
        self.monitoring = False
        self.monitor_thread = None
        
    def create_vpn_connection(self, name, server, username, password, vpn_type="L2TP"):
        """创建VPN连接"""
        try:
            ras_entry = {
                'dwSize': win32ras.RASENTRYNAME_SIZE,
                'dwCountryCode': 0,
                'dwCountryID': 0,
                'szAreaCode': '',
                'szLocalPhoneNumber': server,
                'szDeviceName': 'WAN Miniport (L2TP)' if vpn_type == "L2TP" else 'WAN Miniport (PPTP)',
                'szDeviceType': 'vpn',
                'dwFramingProtocol': win32ras.RASFP_Ppp,
                'dwfNetProtocols': win32ras.RASNP_Ip,
                'dwEncryptionType': win32ras.RASET_Vpn,
                'dwOptions': win32ras.RASEO_RemoteDefaultGateway | win32ras.RASEO_ModemLights,
                'dwType': win32ras.RASET_Vpn,
                'szPhoneNumber': server
            }
            
            win32ras.RasSetEntryDialParams(None, name, server, '', '', '', 1)
            win32ras.RasSetEntryProperties(None, name, ras_entry)
            win32ras.RasSetEntryDialParams(None, name, server, '', username, password, 1)
            return True
        except:
            return False
            
    def connect_vpn(self, name):
        """连接VPN"""
        try:
            handle, err = win32ras.RasDial(None, None, name, '', '')
            if err == 0:
                self.connections[name] = handle
                return True
            return False
        except:
            return False
            
    def disconnect_vpn(self, name):
        """断开VPN连接"""
        try:
            if name in self.connections:
                win32ras.RasHangUp(self.connections[name])
                del self.connections[name]
                return True
            return False
        except:
            return False
            
    def get_vpn_status(self, name):
        """获取VPN连接状态"""
        try:
            if name in self.connections:
                status = win32ras.RasGetConnectStatus(self.connections[name])
                return {
                    'connected': True,
                    'duration': status['rasconnstate'],
                    'error': status['dwError']
                }
            return {'connected': False}
        except:
            return {'connected': False}
            
    def start_traffic_monitoring(self):
        """开始流量监控"""
        if self.monitoring:
            return
            
        self.monitoring = True
        self.monitor_thread = threading.Thread(target=self._monitor_traffic)
        self.monitor_thread.daemon = True
        self.monitor_thread.start()
        
    def _monitor_traffic(self):
        """流量监控工作线程"""
        while self.monitoring:
            net_io = psutil.net_io_counters(pernic=True)
            
            for interface, stats in net_io.items():
                if interface in self.connections:
                    self.traffic_stats[interface]['sent'] = stats.bytes_sent
                    self.traffic_stats[interface]['recv'] = stats.bytes_recv
                    
            time.sleep(1)
            
    def stop_traffic_monitoring(self):
        """停止流量监控"""
        self.monitoring = False
        if self.monitor_thread:
            self.monitor_thread.join(timeout=1)
            
    def get_traffic_stats(self):
        """获取流量统计"""
        return dict(self.traffic_stats)
        
    def get_process_network_usage(self):
        """获取进程网络使用情况"""
        try:
            process_network = {}
            for proc in psutil.process_iter(['pid', 'name']):
                try:
                    io_counters = proc.io_counters()
                    if io_counters:
                        process_network[proc.info['name']] = {
                            'pid': proc.info['pid'],
                            'bytes_sent': io_counters.write_bytes,
                            'bytes_recv': io_counters.read_bytes
                        }
                except (psutil.NoSuchProcess, psutil.AccessDenied):
                    continue
            return process_network
        except:
            return {}
            
    def get_bandwidth_usage(self):
        """获取带宽使用情况"""
        try:
            net_io = psutil.net_io_counters()
            return {
                'total_sent': net_io.bytes_sent,
                'total_recv': net_io.bytes_recv,
                'packets_sent': net_io.packets_sent,
                'packets_recv': net_io.packets_recv
            }
        except:
            return {}

class NetworkSecurityManager:
    def __init__(self):
        self.firewall_manager = None
        self.proxy_settings = {}
        self.suspicious_connections = []
        self.monitoring = False
        self.monitor_thread = None
        self.connection_patterns = {
            'high_frequency': defaultdict(int),
            'unusual_ports': defaultdict(list),
            'large_transfers': defaultdict(list)
        }
        self._init_firewall()
        
    def _init_firewall(self):
        """初始化防火墙管理器"""
        try:
            self.firewall_manager = win32com.client.Dispatch("HNetCfg.FwPolicy2")
        except:
            print("无法初始化防火墙管理器")
            
    def add_firewall_rule(self, name, direction="in", protocol="TCP", 
                          local_port=None, remote_port=None, 
                          remote_addr=None, action="allow"):
        """添加防火墙规则"""
        try:
            rule = win32com.client.Dispatch("HNetCfg.FWRule")
            rule.Name = name
            rule.Direction = getattr(win32com.client.constants, 
                                   f'NET_FW_{direction.upper()}_DIR')
            rule.Protocol = getattr(win32com.client.constants,
                                  f'NET_FW_{protocol.upper()}_PROTOCOL')
            
            if local_port:
                rule.LocalPorts = str(local_port)
            if remote_port:
                rule.RemotePorts = str(remote_port)
            if remote_addr:
                rule.RemoteAddresses = remote_addr
                
            rule.Action = getattr(win32com.client.constants,
                                f'NET_FW_{action.upper()}_ACTION')
            rule.Enabled = True
            
            self.firewall_manager.Rules.Add(rule)
            return True
        except:
            return False
            
    def remove_firewall_rule(self, name):
        """删除防火墙规则"""
        try:
            self.firewall_manager.Rules.Remove(name)
            return True
        except:
            return False
            
    def get_firewall_rules(self):
        """获取所有防火墙规则"""
        try:
            rules = []
            for rule in self.firewall_manager.Rules:
                rules.append({
                    'name': rule.Name,
                    'direction': rule.Direction,
                    'protocol': rule.Protocol,
                    'local_ports': rule.LocalPorts,
                    'remote_ports': rule.RemotePorts,
                    'remote_addresses': rule.RemoteAddresses,
                    'action': rule.Action,
                    'enabled': rule.Enabled
                })
            return rules
        except:
            return []
            
    def set_system_proxy(self, server=None, port=None, bypass=None):
        """设置系统代理"""
        try:
            key = win32api.RegOpenKeyEx(
                win32con.HKEY_CURRENT_USER,
                r"Software\Microsoft\Windows\CurrentVersion\Internet Settings",
                0,
                win32con.KEY_SET_VALUE
            )
            
            if server and port:
                proxy = f"{server}:{port}"
                win32api.RegSetValueEx(key, "ProxyServer", 0, win32con.REG_SZ, proxy)
                win32api.RegSetValueEx(key, "ProxyEnable", 0, win32con.REG_DWORD, 1)
            else:
                win32api.RegSetValueEx(key, "ProxyEnable", 0, win32con.REG_DWORD, 0)
                
            if bypass:
                win32api.RegSetValueEx(key, "ProxyOverride", 0, win32con.REG_SZ, bypass)
                
            win32api.RegCloseKey(key)
            
            win32api.SendMessageTimeout(
                win32con.HWND_BROADCAST,
                win32con.WM_SETTINGCHANGE,
                0,
                "Environment",
                win32con.SMTO_ABORTIFHUNG,
                5000
            )
            return True
        except:
            return False
            
    def get_system_proxy(self):
        """获取系统代理设置"""
        try:
            key = win32api.RegOpenKeyEx(
                win32con.HKEY_CURRENT_USER,
                r"Software\Microsoft\Windows\CurrentVersion\Internet Settings",
                0,
                win32con.KEY_READ
            )
            
            enabled, _ = win32api.RegQueryValueEx(key, "ProxyEnable")
            if enabled:
                proxy, _ = win32api.RegQueryValueEx(key, "ProxyServer")
                bypass, _ = win32api.RegQueryValueEx(key, "ProxyOverride")
                return {
                    'enabled': True,
                    'server': proxy.split(':')[0],
                    'port': proxy.split(':')[1],
                    'bypass': bypass
                }
            return {'enabled': False}
        except:
            return {'enabled': False}
            
    def start_connection_monitoring(self):
        """开始连接监控"""
        if self.monitoring:
            return
            
        self.monitoring = True
        self.monitor_thread = threading.Thread(target=self._monitor_connections)
        self.monitor_thread.daemon = True
        self.monitor_thread.start()
        
    def _monitor_connections(self):
        """连接监控工作线程"""
        while self.monitoring:
            try:
                connections = psutil.net_connections()
                current_time = time.time()
                
                for conn in connections:
                    if conn.status == 'ESTABLISHED':
                        if conn.raddr:
                            key = f"{conn.raddr.ip}:{conn.raddr.port}"
                            self.connection_patterns['high_frequency'][key] += 1
                            
                            if conn.raddr.port not in [80, 443, 22, 21, 25, 53]:
                                self.connection_patterns['unusual_ports'][key].append(current_time)
                                
                        try:
                            proc = psutil.Process(conn.pid)
                            io_counters = proc.io_counters()
                            if io_counters:
                                transfer_size = io_counters.read_bytes + io_counters.write_bytes
                                if transfer_size > 100 * 1024 * 1024:
                                    self.connection_patterns['large_transfers'][conn.pid].append({
                                        'time': current_time,
                                        'size': transfer_size
                                    })
                        except (psutil.NoSuchProcess, psutil.AccessDenied):
                            pass
                            
                self._detect_suspicious_activity()
                time.sleep(1)
                
            except:
                pass
                
    def _detect_suspicious_activity(self):
        """检测可疑活动"""
        current_time = time.time()
        
        for conn, count in self.connection_patterns['high_frequency'].items():
            if count > 1000:
                self.suspicious_connections.append({
                    'type': 'high_frequency',
                    'connection': conn,
                    'count': count,
                    'time': current_time
                })
                
        for conn, times in self.connection_patterns['unusual_ports'].items():
            recent_times = [t for t in times if current_time - t < 60]
            if len(recent_times) > 10:
                self.suspicious_connections.append({
                    'type': 'unusual_port',
                    'connection': conn,
                    'count': len(recent_times),
                    'time': current_time
                })
                
        for pid, transfers in self.connection_patterns['large_transfers'].items():
            recent_transfers = [t for t in transfers if current_time - t['time'] < 60]
            if len(recent_transfers) > 5:
                self.suspicious_connections.append({
                    'type': 'large_transfer',
                    'pid': pid,
                    'count': len(recent_transfers),
                    'time': current_time
                })
                
    def stop_connection_monitoring(self):
        """停止连接监控"""
        self.monitoring = False
        if self.monitor_thread:
            self.monitor_thread.join(timeout=1)
            
    def get_suspicious_connections(self):
        """获取可疑连接"""
        return self.suspicious_connections
        
    def clear_suspicious_connections(self):
        """清除可疑连接记录"""
        self.suspicious_connections = []
        
    def block_suspicious_ip(self, ip):
        """阻止可疑IP"""
        return self.add_firewall_rule(
            name=f"Block_{ip}",
            direction="in",
            remote_addr=ip,
            action="block"
        )
