import os
import sys
from pathlib import Path

from dotenv import load_dotenv

from howler.plugins import get_plugins

load_dotenv()

# We append the plugin directory for howler to the python part
PLUGIN_PATH = Path(os.environ.get("HWL_PLUGIN_DIRECTORY", "/etc/howler/plugins"))
sys.path.insert(0, str(PLUGIN_PATH))

from howler.odm.models.config import config

if config.ui.debug and PLUGIN_PATH.exists():
    for _plugin in PLUGIN_PATH.iterdir():
        sys.path.append(
            str(Path(os.path.realpath(_plugin)) / f"../.venv/lib/python3.{sys.version_info.minor}/site-packages")
        )

import logging
from typing import Any, cast

from authlib.integrations.flask_client import OAuth
from elasticapm.contrib.flask import ElasticAPM
from flasgger import Swagger
from flask import Flask
from flask.blueprints import Blueprint
from flask.logging import default_handler
from prometheus_client import make_wsgi_app
from werkzeug.middleware.dispatcher import DispatcherMiddleware

from howler.api.base import api
from howler.api.socket import socket_api
from howler.api.v1 import apiv1
from howler.api.v1.action import action_api
from howler.api.v1.analytic import analytic_api
from howler.api.v1.auth import auth_api
from howler.api.v1.configs import config_api
from howler.api.v1.dossier import dossier_api
from howler.api.v1.help import help_api
from howler.api.v1.hit import hit_api
from howler.api.v1.overview import overview_api
from howler.api.v1.search import search_api
from howler.api.v1.template import template_api
from howler.api.v1.tool import tool_api
from howler.api.v1.user import user_api
from howler.api.v1.view import view_api
from howler.common.logging import get_logger
from howler.config import (
    DEBUG,
    HWL_UNSECURED_UI,
    HWL_USE_JOB_SYSTEM,
    HWL_USE_REST_API,
    HWL_USE_WEBSOCKET_API,
    SECRET_KEY,
    cache,
    config,
)
from howler.cronjobs import setup_jobs
from howler.error import errors
from howler.healthz import healthz

logger = get_logger(__file__)

app = Flask(
    "howler-api",
    static_url_path="/api/static",
    static_folder=config.ui.static_folder,
)
# Disable strict check on trailing slashes for endpoints
app.url_map.strict_slashes = False
app.config["JSON_SORT_KEYS"] = False

app.wsgi_app = DispatcherMiddleware(app.wsgi_app, {"/metrics": make_wsgi_app()})  # type: ignore[method-assign]

swagger_template = {
    "info": {
        "title": "Howler API",
        "description": (
            "Howler is an application that allows analysts to triage hits and alerts. It provides a way for "
            "analysts to efficiently review and analyze alerts generated by different analytics and detections."
        ),
    }
}
swagger = Swagger(
    app,
    template=swagger_template,
    config={
        "headers": [],
        "static_url_path": "/api/swagger_static",
        "specs": [
            {
                "endpoint": "apispec_v1",
                "route": "/api/apispec_v1.json",
                "rule_filter": lambda rule: True,  # all in
                "model_filter": lambda tag: True,  # all in
            }
        ],
        "specs_route": "/api/docs",
    },
)

cache.init_app(app)

app.logger.setLevel(60)  # This completely turns off the flask logger
if HWL_UNSECURED_UI:
    app.config.update(SESSION_COOKIE_SECURE=False, SECRET_KEY=SECRET_KEY, PREFERRED_URL_SCHEME="http")
else:
    app.config.update(SESSION_COOKIE_SECURE=True, SECRET_KEY=SECRET_KEY, PREFERRED_URL_SCHEME="https")

app.register_blueprint(errors)
app.register_blueprint(healthz)

if HWL_USE_REST_API or DEBUG:
    logger.debug("Enabled REST API")
    app.register_blueprint(action_api)
    app.register_blueprint(analytic_api)
    app.register_blueprint(api)
    app.register_blueprint(apiv1)
    app.register_blueprint(auth_api)
    app.register_blueprint(config_api)
    app.register_blueprint(help_api)
    app.register_blueprint(hit_api)
    app.register_blueprint(search_api)
    app.register_blueprint(template_api)
    app.register_blueprint(overview_api)
    app.register_blueprint(tool_api)
    app.register_blueprint(user_api)
    app.register_blueprint(view_api)
    app.register_blueprint(dossier_api)

    if config.core.notebook.enabled:
        from howler.api.v1.notebook import notebook_api

        logger.debug("Enabled Notebook Integration")
        app.register_blueprint(notebook_api)

    if config.core.borealis.enabled:
        from howler.api.v1.borealis import borealis_api

        logger.debug("Enabled Borealis Integration")
        app.register_blueprint(borealis_api)

    logger.info("Checking plugins for additional routes")
    for plugin in get_plugins():
        if not plugin.modules.routes:
            continue

        for route in cast(list[Blueprint], plugin.modules.routes):
            logger.info("Enabling additional endpoint: %s", route.url_prefix)
            app.register_blueprint(route)


else:
    logger.info("Disabled REST API")

if HWL_USE_WEBSOCKET_API or DEBUG:
    logger.debug("Enabled Websocket API")
    app.register_blueprint(socket_api)
else:
    logger.info("Disabled Websocket API")

if HWL_USE_JOB_SYSTEM or DEBUG:
    setup_jobs()


# Setup OAuth providers
if config.auth.oauth.enabled:
    providers = []
    for name, provider in config.auth.oauth.providers.items():
        p: dict[str, Any] = provider.model_dump()

        # Set provider name
        p["name"] = name

        # Remove howler specific fields from oAuth config
        p.pop("auto_create", None)
        p.pop("auto_sync", None)
        p.pop("user_get", None)
        p.pop("auto_properties", None)
        p.pop("uid_regex", None)
        p.pop("uid_format", None)
        p.pop("user_groups", None)
        p.pop("user_groups_data_field", None)
        p.pop("user_groups_name_field", None)
        p.pop("app_provider", None)

        # Add the provider to the list of providers
        providers.append(p)

    if providers:
        oauth = OAuth()
        for p in providers:
            oauth.register(**p)
        oauth.init_app(app)

# Setup logging
app.logger.setLevel(logger.getEffectiveLevel())
app.logger.removeHandler(default_handler)
if logger.parent:
    for ph in logger.parent.handlers:
        app.logger.addHandler(ph)

# Setup APMs
if config.core.metrics.apm_server.server_url is not None:
    logger.info(f"Exporting application metrics to: {config.core.metrics.apm_server.server_url}")
    ElasticAPM(
        app,
        server_url=config.core.metrics.apm_server.server_url,
        service_name="howler_api",
    )

wlog = logging.getLogger("werkzeug")
wlog.setLevel(logging.WARNING)
if logger.parent:  # pragma: no cover
    for h in logger.parent.handlers:
        wlog.addHandler(h)


def main():
    """Main application function"""
    app.jinja_env.cache = {}
    app.run(
        host="0.0.0.0",  # noqa: S104
        debug=DEBUG,
        port=int(os.getenv("FLASK_RUN_PORT", os.getenv("PORT", 5000))),
        extra_files=os.environ.get("FLASK_RUN_EXTRA_FILES", "").split(":"),
    )


if __name__ == "__main__":
    main()
