"""Repository mixin for automatic JSON passthrough support."""

import json
import logging
from typing import Any

from fraiseql.core.raw_json_executor import RawJSONResult

logger = logging.getLogger(__name__)


class PassthroughMixin:
    """Mixin that adds automatic JSON passthrough to repository methods.

    When the repository detects passthrough mode from the context,
    it returns RawJSONResult instead of Python objects, bypassing
    GraphQL type validation entirely.
    """

    def _should_use_passthrough(self) -> bool:
        """Check if passthrough mode is enabled."""
        if not hasattr(self, "context"):
            logger.debug("PassthroughMixin: No context attribute")
            return False

        context = self.context
        result = (
            context.get("mode") in ("production", "staging")
            or context.get("json_passthrough", False)
            or context.get("execution_mode") == "passthrough"
            or context.get("_passthrough_enabled", False)
        )
        logger.info(
            f"PassthroughMixin: _should_use_passthrough = {result}, mode={context.get('mode')}, json_passthrough={context.get('json_passthrough')}"
        )
        return result

    def _get_field_name(self) -> str:
        """Get the current GraphQL field name from context."""
        if hasattr(self, "context"):
            return (
                self.context.get("_passthrough_field")
                or self.context.get("graphql_field_name")
                or "data"
            )
        return "data"

    def _wrap_as_raw_json(self, result: Any) -> Any:
        """Wrap result as RawJSONResult if in passthrough mode."""
        if not self._should_use_passthrough():
            return result

        # Don't double-wrap
        if isinstance(result, RawJSONResult):
            return result

        # Get field name
        field_name = self._get_field_name()

        # For None, lists, and dicts, wrap as raw JSON
        if result is None or isinstance(result, (list, dict)):
            # Create the GraphQL response structure
            graphql_response = {"data": {field_name: result}}
            return RawJSONResult(json.dumps(graphql_response))

        # For other types, return as-is
        return result

    # Override common repository methods to add passthrough support

    async def find(self, *args, **kwargs):
        """Find with automatic passthrough support."""
        result = await super().find(*args, **kwargs)
        return self._wrap_as_raw_json(result)

    async def find_one(self, *args, **kwargs):
        """Find one with automatic passthrough support."""
        logger.info("PassthroughMixin.find_one called, passthrough check...")
        result = await super().find_one(*args, **kwargs)
        wrapped = self._wrap_as_raw_json(result)
        logger.info(
            f"PassthroughMixin.find_one: result type={type(result).__name__}, wrapped type={type(wrapped).__name__}"
        )
        return wrapped

    async def query(self, *args, **kwargs):
        """Query with automatic passthrough support."""
        result = await super().query(*args, **kwargs)
        return self._wrap_as_raw_json(result)

    async def get(self, *args, **kwargs):
        """Get with automatic passthrough support."""
        result = await super().get(*args, **kwargs)
        return self._wrap_as_raw_json(result)

    # For repositories that have these common methods
    async def query_with_tenant(self, *args, **kwargs):
        """Query with tenant and automatic passthrough support."""
        result = await super().query_with_tenant(*args, **kwargs)
        return self._wrap_as_raw_json(result)

    async def get_by_id_with_tenant(self, *args, **kwargs):
        """Get by ID with tenant and automatic passthrough support."""
        result = await super().get_by_id_with_tenant(*args, **kwargs)
        return self._wrap_as_raw_json(result)
