# Spire

Spire is a thin wrapper around [doit](http://pydoit.org/). It eases the declaration of tasks through:
- Class-based task declarations
- Built-in factories for repetitive tasks
- Optional pruning of the task graph when some dependencies are missing

Moreover, tasks will be rerun whenever their actions are modified.

## Task declaration

Spire tasks can be classes: this syntax facilitates the reusability of dependencies and targets in the list of actions. 

```python
import spire

class BuildHouse(spire.Task):
    file_dep = ["brick", "mortar"]
    targets = ["house", "dog_house"]
    actions = [["build"]+file_dep+targets]
```

This task file can then be run with the usual `doit` command:

```console
$ doit run -f build_house.py -d /home/somebody/vacant_lot
. BuildHouse
```

For simple tasks (single target or single action), it is not mandatory to use lists. In this case, the singular form of the member name must be used (i.e. _targets_ becomes _target_ and _actions_ becomes _action_).

```python
import spire

class BuildShed(spire.Task):
    file_dep = "wood"
    target = "shed"
    action = ["build", file_dep, target]
```

Spire tasks are cleanable by default: using the previous examples, calling `doit clean -f ... -d ...` will remove the targets.

## Repetitive tasks

For repetitive tasks, Spire provides the `TaskFactory` class. Classes derived from `TaskFactory` need to set the following members for each object:
- The task name, through the constructor of `TaskFactory`
- `file_dep`, `targets` and `actions`

```python
import spire

class BuildHouse(spire.TaskFactory):
    def __init__(self, material):
        spire.TaskFactory.__init__(self, "Build{}House".format(material))
        self.file_dep = [material]
        self.targets = ["{}_house".format(material)]
        self.actions = [["build", material]]

houses = [BuildHouse(material) for material in ["Straw", "Sticks", "Bricks"]]
```

## Pruning the task graph

Tasks with missing dependencies may be skipped instead of being executed and failing. For this, missing dependencies must be specified as `None` entries in `file_dep`, and the function `spire.prune()` must be called. The task graph will be pruned starting at the current task, ensuring that no error will occur on account of these missing targets.

In the following example, if either `brick` or `mortar` is missing, neither `BuildHouse` nor `BuildDogHouse` will be executed:
- `BuildHouse` will be skipped since `file_dep` contains entries which are `None` and `spire.prune()` was called
- `BuildDogHouse` will be skipped since one of its parent has been skipped. 

On the other hand, if `brick` and `mortar` are present but `doggie_basket` is missing, `BuildHouse` will be successfully executed but `BuildDogHouse` will fail as none of its `file_dep` equal `None`.

```python
import os
import spire

class BuildHouse(spire.Task):
    file_dep = [x if os.path.isfile(x) else None for x in ["brick", "mortar"]]
    target = "house"
    action = ["build"] + file_dep  + [target]

class BuildDogHouse(spire.Task):
    file_dep = [BuildHouse.target, "doggie_basket"]
    target = "dog_house"
    action = ["build"] + file_dep  + [target]

spire.prune()
```

## Graphical representation of the task graph

A graphical representation of the task graph, in the [Graphviz](http://graphviz.org/) format, can be generated by calling the `spire` module:

```
$ python3 -m spire graph tasks.py tasks.dot
```

A simplified representation, omitting the targets and dependencies nodes, can be generated py passing the option `--tasks-only`. Any other option will be passed directly to _doit_, e.g. to specify command-line variables.
