"""Support for Nexia Automations."""

from typing import Any

from nexia.automation import NexiaAutomation

from homeassistant.components.scene import Scene
from homeassistant.core import HomeAssistant
from homeassistant.helpers.entity_platform import AddConfigEntryEntitiesCallback
from homeassistant.helpers.event import async_call_later

from .const import ATTR_DESCRIPTION
from .coordinator import NexiaDataUpdateCoordinator
from .entity import NexiaEntity
from .types import NexiaConfigEntry

SCENE_ACTIVATION_TIME = 5


async def async_setup_entry(
    hass: HomeAssistant,
    config_entry: NexiaConfigEntry,
    async_add_entities: AddConfigEntryEntitiesCallback,
) -> None:
    """Set up automations for a Nexia device."""
    coordinator = config_entry.runtime_data
    nexia_home = coordinator.nexia_home
    async_add_entities(
        NexiaAutomationScene(
            coordinator, nexia_home.get_automation_by_id(automation_id)
        )
        for automation_id in nexia_home.get_automation_ids()
    )


class NexiaAutomationScene(NexiaEntity, Scene):
    """Provides Nexia automation support."""

    _attr_translation_key = "automation"

    def __init__(
        self, coordinator: NexiaDataUpdateCoordinator, automation: NexiaAutomation
    ) -> None:
        """Initialize the automation scene."""
        super().__init__(coordinator, automation.automation_id)
        self._attr_name = automation.name
        self._automation = automation
        self._attr_extra_state_attributes = {ATTR_DESCRIPTION: automation.description}

    async def async_activate(self, **kwargs: Any) -> None:
        """Activate an automation scene."""
        await self._automation.activate()

        async def refresh_callback(_):
            await self.coordinator.async_refresh()

        async_call_later(self.hass, SCENE_ACTIVATION_TIME, refresh_callback)
