import json
from typing import Any, Type, Optional

from pydantic import BaseModel, Field

from codemie_tools.base.codemie_tool import CodeMieTool
from codemie_tools.data_management.elastic.elastic_wrapper import SearchElasticIndexResults
from codemie_tools.data_management.elastic.models import ElasticConfig
from codemie_tools.data_management.elastic.tools_vars import SEARCH_ES_INDEX_TOOL


class SearchElasticIndexInput(BaseModel):
    query: str = Field(
        description="""
            String text. It's a query to Elastic API in a form of a Query DSL generated by LLM which 
            will be used to find necessary information in a specified index to fulfill user request. 
            Important: string formatting and escaping SHOULD NOT be used when passing query to the tool.
            """.strip(),
    )
    index: str = Field(
        description="""
                String text. It's name of the Elastic index which should be used to apply generated query to 
                fulfill user request.
                """.strip(),
    )


class SearchElasticIndex(CodeMieTool):
    elastic_config: Optional[ElasticConfig] = Field(exclude=True, default=None)
    name: str = SEARCH_ES_INDEX_TOOL.name
    description: str = SEARCH_ES_INDEX_TOOL.description
    args_schema: Type[BaseModel] = SearchElasticIndexInput

    def execute(self, index: str, query: str, **kwargs: Any) -> Any:
        if not self.elastic_config:
            raise ValueError("Elastic configuration is not provided")
        mapping = json.loads(query)
        response = SearchElasticIndexResults.search(index=index, query=mapping, elastic_config=self.elastic_config)
        return response
