#!/usr/bin/env python3
"""
Ariadne Routing Demo: Intelligent Quantum Circuit Routing

Demonstrates Ariadne's intelligent routing system that automatically selects
the optimal backend for quantum circuits using Bell Labs-style information theory.
"""

import time
from typing import Any

from qiskit import QuantumCircuit
from qiskit.circuit.library import EfficientSU2
from rich.console import Console
from rich.panel import Panel
from rich.table import Table
from rich.text import Text

# Import Ariadne components
from ariadne import QuantumRouter
from ariadne.route.analyze import analyze_circuit
from ariadne.router import BackendType

console = Console()


def create_clifford_circuit(n_qubits: int = 10) -> QuantumCircuit:
    """Create a Clifford circuit (perfect for Stim)."""
    circuit = QuantumCircuit(n_qubits)

    # Add random Clifford gates
    for i in range(n_qubits * 2):
        # Random single-qubit Clifford gates
        gate_type = i % 4
        qubit = i % n_qubits

        if gate_type == 0:
            circuit.h(qubit)
        elif gate_type == 1:
            circuit.s(qubit)
        elif gate_type == 2:
            circuit.x(qubit)
        else:
            circuit.z(qubit)

        # Add some CNOT gates
        if i % 3 == 0 and qubit < n_qubits - 1:
            circuit.cx(qubit, qubit + 1)

    return circuit


def create_mixed_circuit(n_qubits: int = 5) -> QuantumCircuit:
    """Create a mixed circuit with Clifford + T gates (good for Qiskit)."""
    circuit = QuantumCircuit(n_qubits)

    # Start with some Clifford gates
    for i in range(n_qubits):
        circuit.h(i)
        if i < n_qubits - 1:
            circuit.cx(i, i + 1)

    # Add some T gates (non-Clifford)
    for i in range(n_qubits):
        circuit.t(i)
        if i % 2 == 0:
            circuit.tdg(i)

    # More Clifford gates
    for i in range(n_qubits - 1):
        circuit.cx(i, i + 1)

    return circuit


def create_large_circuit(n_qubits: int = 15) -> QuantumCircuit:
    """Create a large circuit (good for tensor networks)."""
    circuit = QuantumCircuit(n_qubits)

    # Create a complex variational circuit
    ansatz = EfficientSU2(n_qubits, reps=2).decompose()
    if ansatz.parameters:
        zero_params = dict.fromkeys(ansatz.parameters, 0.1)
        ansatz = ansatz.assign_parameters(zero_params)
    circuit.compose(ansatz, inplace=True)

    # Add measurements
    circuit.measure_all()

    return circuit


def demonstrate_routing():
    """Demonstrate Ariadne's intelligent routing capabilities."""
    console.print(
        Panel.fit(
            "[bold blue]Ariadne: Intelligent Quantum Router[/bold blue]\n\n"
            "Watch as Ariadne automatically routes different circuit types to optimal backends.",
            title="Demo Start",
            border_style="blue",
        )
    )

    # Create different types of circuits
    circuits = {
        "Clifford Circuit (10 qubits)": create_clifford_circuit(10),
        "Mixed Circuit (5 qubits)": create_mixed_circuit(5),
        "Large Variational Circuit (15 qubits)": create_large_circuit(15),
    }

    # Initialize router
    router = QuantumRouter()

    # Create results table
    table = Table(title="Intelligent Routing Results")
    table.add_column("Circuit Type", style="cyan", no_wrap=True)
    table.add_column("Entropy H(Q)", style="magenta")
    table.add_column("Optimal Backend", style="green")
    table.add_column("Confidence", style="yellow")
    table.add_column("Expected Speedup", style="red")
    table.add_column("Execution Time", style="blue")

    results = []

    for name, circuit in circuits.items():
        console.print(f"\n[bold]Analyzing {name}...[/bold]")

        # Analyze circuit using the shared analyzer helper (the dedicated
        # ``QuantumRouter.circuit_entropy`` helper was removed in 0.4.0).
        analysis = analyze_circuit(circuit)
        entropy = analysis.get("gate_entropy")
        console.print(f"  Clifford ratio: {analysis['clifford_ratio']:.2f}, depth {analysis['depth']}")

        # Get routing decision
        decision = router.select_optimal_backend(circuit)

        # Simulate with intelligent routing
        start_time = time.time()
        result = router.simulate(circuit, shots=1000)
        execution_time = time.time() - start_time

        # Add to table
        entropy_display = f"{entropy:.2f} bits" if entropy is not None else "—"

        table.add_row(
            name,
            entropy_display,
            decision.recommended_backend.value,
            f"{decision.confidence_score:.1%}",
            f"{decision.expected_speedup:.1f}x",
            f"{execution_time:.3f}s",
        )

        results.append(
            {
                "name": name,
                "entropy": entropy,
                "backend": decision.recommended_backend,
                "confidence": decision.confidence_score,
                "speedup": decision.expected_speedup,
                "execution_time": execution_time,
                "result": result,
            }
        )

    console.print(table)

    # Show detailed analysis
    console.print("\n[bold blue]Detailed Analysis[/bold blue]")

    for result in results:
        show_circuit_analysis(result)


def show_circuit_analysis(result: dict[str, Any]):
    """Show detailed analysis of a circuit routing decision."""
    name = result["name"]
    entropy = result["entropy"]
    backend = result["backend"]
    confidence = result["confidence"]
    speedup = result["speedup"]

    # Create analysis panel
    analysis_text = Text()
    analysis_text.append(f"Circuit: {name}\n", style="bold cyan")
    entropy_line = f"{entropy:.2f} bits" if entropy is not None else "N/A"
    analysis_text.append(f"Information Content: {entropy_line}\n", style="magenta")
    analysis_text.append(f"Optimal Backend: {backend.value}\n", style="green")
    analysis_text.append(f"Routing Confidence: {confidence:.1%}", style="yellow")
    analysis_text.append(f"Expected Speedup: {speedup:.1f}x vs Qiskit", style="red")

    # Backend-specific insights
    insights = get_backend_insights(backend, entropy)
    analysis_text.append(f"\n{insights}", style="blue")

    console.print(Panel(analysis_text, title=f"{name} Analysis", border_style="blue"))


def get_backend_insights(backend: BackendType, entropy: float | None) -> str:
    """Get insights about why a particular backend was chosen."""
    insights = {
        BackendType.STIM: "Perfect for Clifford circuits – often 50-100× faster than general simulators",
        BackendType.QISKIT: "Reliable general-purpose simulator for mixed gate types",
        BackendType.TENSOR_NETWORK: "Memory efficient for large, sparse circuits",
        BackendType.JAX_METAL: "GPU-accelerated simulation with Apple Silicon optimization",
        BackendType.DDSIM: "Decision diagram-based simulation for structured circuits",
    }

    return insights.get(backend, "General-purpose simulation")


def demonstrate_information_theory():
    """Demonstrate the information theory behind Ariadne's routing."""
    console.print(
        Panel.fit(
            "[bold blue]Bell Labs-Style Information Theory[/bold blue]\n\n"
            "Ariadne applies Claude Shannon's principles to quantum simulation routing.",
            title="Theory Demo",
            border_style="blue",
        )
    )

    # Create a simple circuit for analysis
    circuit = QuantumCircuit(3)
    circuit.h(0)
    circuit.cx(0, 1)
    circuit.cx(1, 2)
    circuit.t(0)  # Add a non-Clifford gate

    router = QuantumRouter()

    # Show information theory calculations using the analyzer helper.
    analysis = analyze_circuit(circuit)
    entropy = analysis.get("gate_entropy") or 0.0
    decision = router.select_optimal_backend(circuit)

    console.print("[bold]Circuit Analysis:[/bold]")
    console.print(f"  • Circuit entropy H(Q): {entropy:.3f} bits")
    console.print("  • Gate distribution: H, CX, CX, T")
    console.print(f"  • Information content: {entropy:.3f} bits of quantum information")

    console.print("\n[bold]Backend Capacities:[/bold]")
    console.print(f"  • Selected backend capacity match: {decision.channel_capacity_match:.1%}")

    console.print("\n[bold]Routing Decision:[/bold]")
    console.print(f"  • Optimal backend: {decision.recommended_backend.value}")
    console.print(f"  • Confidence: {decision.confidence_score:.1%}")
    console.print(f"  • Expected speedup: {decision.expected_speedup:.1f}x")


def demonstrate_speed_comparison():
    """Demonstrate speed improvements with intelligent routing."""
    console.print(
        Panel.fit(
            "[bold red]Performance Comparison[/bold red]\n\n"
            "See how Ariadne's intelligent routing provides speed improvements.",
            title="Speed Demo",
            border_style="red",
        )
    )

    # Create a Clifford circuit that should be very fast with Stim
    clifford_circuit = create_clifford_circuit(8)

    # Method 1: Naive Qiskit (always)
    start = time.time()
    from ariadne import simulate

    result1 = simulate(clifford_circuit, 1000, backend="qiskit")
    qiskit_time = time.time() - start

    # Method 2: Intelligent routing
    start = time.time()
    result2 = simulate(clifford_circuit, 1000)
    intelligent_time = time.time() - start

    speedup = qiskit_time / intelligent_time if intelligent_time > 0 else float("inf")

    console.print("[bold]Results for 8-qubit Clifford circuit:[/bold]")
    console.print(f"  • Naive Qiskit time: {qiskit_time:.3f}s")
    console.print(f"  • Intelligent routing time: {intelligent_time:.3f}s")
    console.print(f"  • Speedup: {speedup:.1f}x")
    console.print(f"  • Sample naive counts: {dict(list(result1.counts.items())[:3])}")

    if result2.routing_decision.recommended_backend == BackendType.STIM:
        console.print(f"  • Backend chosen: {result2.backend_used.value} (optimal for Clifford)")
    else:
        console.print(f"  • Backend chosen: {result2.backend_used.value}")


def main():
    """Run the complete Ariadne routing demonstration."""
    console.print(
        Panel.fit(
            "[bold green]Welcome to Ariadne: The Intelligent Quantum Router[/bold green]\n\n"
            "This demo showcases Bell Labs-style information theory applied to quantum simulation.\n"
            "Watch as Ariadne automatically routes circuits to optimal backends!",
            title="Ariadne Demo",
            border_style="green",
        )
    )

    try:
        # Run demonstrations
        demonstrate_routing()
        console.print("\n" + "=" * 60 + "\n")

        demonstrate_information_theory()
        console.print("\n" + "=" * 60 + "\n")

        demonstrate_speed_comparison()

        # Final summary
        console.print(
            Panel.fit(
                "[bold green]Demo Complete[/bold green]\n\n"
                "Ariadne demonstrated intelligent routing:\n"
                "• Clifford circuits → Stim (50-100× speedup)\n"
                "• Mixed circuits → Qiskit (reliable)\n"
                "• Large circuits → Tensor networks (memory efficient)\n"
                "• Apple Silicon → JAX/Metal (GPU acceleration)\n\n"
                "[italic]Bell Labs-style information theory applied to quantum simulation.[/italic]",
                title="Success",
                border_style="green",
            )
        )

    except ImportError as e:
        console.print(f"[red]Import Error: {e}[/red]")
        console.print("[yellow]Install required packages:[/yellow]")
        console.print("  pip install qiskit stim jax jax-metal")
    except Exception as e:
        console.print(f"[red]Error: {e}[/red]")
        console.print("[yellow]Make sure quantum computing libraries are installed[/yellow]")


if __name__ == "__main__":
    main()
