"""Shell completion commands for pltr CLI."""

import os
from pathlib import Path
from typing import Optional

import typer
from rich.console import Console
from rich.panel import Panel
from rich.syntax import Syntax

console = Console()
app = typer.Typer(help="Manage shell completions for pltr CLI")


def get_shell_from_env() -> Optional[str]:
    """Detect the current shell from environment."""
    shell_path = os.environ.get("SHELL", "")
    if "bash" in shell_path:
        return "bash"
    elif "zsh" in shell_path:
        return "zsh"
    elif "fish" in shell_path:
        return "fish"
    return None


@app.command()
def install(
    shell: Optional[str] = typer.Option(
        None,
        "--shell",
        "-s",
        help="Shell type (bash, zsh, fish). Auto-detected if not specified.",
    ),
    path: Optional[Path] = typer.Option(
        None,
        "--path",
        "-p",
        help="Custom path to install completion file.",
    ),
):
    """Install shell completions for pltr CLI.

    This command will install shell completions for the specified shell
    (or auto-detect from your environment). After installation, restart
    your shell or source the appropriate file to enable completions.

    Examples:
        pltr completion install
        pltr completion install --shell bash
        pltr completion install --shell zsh --path ~/.config/custom/completions
    """
    # Auto-detect shell if not specified
    if shell is None:
        shell = get_shell_from_env()
        if shell is None:
            console.print(
                "[red]Could not detect shell type. Please specify with --shell option.[/red]"
            )
            console.print("Supported shells: bash, zsh, fish")
            raise typer.Exit(1)
        console.print(f"[cyan]Auto-detected shell: {shell}[/cyan]")

    shell = shell.lower()
    if shell not in ["bash", "zsh", "fish"]:
        console.print(f"[red]Unsupported shell: {shell}[/red]")
        console.print("Supported shells: bash, zsh, fish")
        raise typer.Exit(1)

    # Generate completion script
    completion_script = generate_completion_script(shell)

    # Determine installation path
    if path is None:
        path = get_default_completion_path(shell)

    # Ensure directory exists
    path.parent.mkdir(parents=True, exist_ok=True)

    # Write completion file
    try:
        path.write_text(completion_script)
        console.print(f"[green]✓[/green] Completion file written to: {path}")
    except Exception as e:
        console.print(f"[red]Failed to write completion file: {e}[/red]")
        raise typer.Exit(1)

    # Show instructions for activation
    show_activation_instructions(shell, path)


@app.command()
def show(
    shell: Optional[str] = typer.Option(
        None,
        "--shell",
        "-s",
        help="Shell type (bash, zsh, fish). Auto-detected if not specified.",
    ),
):
    """Show the completion script for the specified shell.

    This displays the completion script without installing it,
    useful for manual installation or debugging.
    """
    # Auto-detect shell if not specified
    if shell is None:
        shell = get_shell_from_env()
        if shell is None:
            console.print(
                "[red]Could not detect shell type. Please specify with --shell option.[/red]"
            )
            console.print("Supported shells: bash, zsh, fish")
            raise typer.Exit(1)

    shell = shell.lower()
    if shell not in ["bash", "zsh", "fish"]:
        console.print(f"[red]Unsupported shell: {shell}[/red]")
        console.print("Supported shells: bash, zsh, fish")
        raise typer.Exit(1)

    # Generate and display completion script
    completion_script = generate_completion_script(shell)
    syntax = Syntax(completion_script, "bash", theme="monokai", line_numbers=True)

    panel = Panel(
        syntax,
        title=f"{shell.capitalize()} Completion Script",
        border_style="cyan",
    )
    console.print(panel)


def generate_completion_script(shell: str) -> str:
    """Generate completion script for the specified shell."""
    if shell == "bash":
        return generate_bash_completion()
    elif shell == "zsh":
        return generate_zsh_completion()
    elif shell == "fish":
        return generate_fish_completion()
    else:
        raise ValueError(f"Unsupported shell: {shell}")


def generate_bash_completion() -> str:
    """Generate Bash completion script."""
    return """# pltr bash completion script
# Generated by pltr completion install

_pltr_completion() {
    local IFS=$'\\n'
    local response

    response=$(env COMP_WORDS="${COMP_WORDS[*]}" COMP_CWORD=$COMP_CWORD _PLTR_COMPLETE=bash_complete pltr)

    for completion in $response; do
        IFS=',' read type value <<< "$completion"

        if [[ $type == 'dir' ]]; then
            COMPREPLY+=("$value/")
            compopt -o dirnames
        elif [[ $type == 'file' ]]; then
            COMPREPLY+=("$value")
            compopt -o filenames
        elif [[ $type == 'plain' ]]; then
            COMPREPLY+=("$value")
        fi
    done

    return 0
}

_pltr_completion_setup() {
    complete -o nosort -F _pltr_completion pltr
}

_pltr_completion_setup
"""


def generate_zsh_completion() -> str:
    """Generate Zsh completion script."""
    return """#compdef pltr
# pltr zsh completion script
# Generated by pltr completion install

_pltr() {
    local -a completions
    local -a completions_with_descriptions
    local -a response
    (( ! $+commands[pltr] )) && return 1

    response=("${(@f)$(env COMP_WORDS="${words[*]}" COMP_CWORD=$((CURRENT-1)) _PLTR_COMPLETE=zsh_complete pltr)}")

    for type key descr in ${response}; do
        if [[ "$type" == "plain" ]]; then
            if [[ "$descr" == "_" ]]; then
                completions+=("$key")
            else
                completions_with_descriptions+=("$key":"$descr")
            fi
        elif [[ "$type" == "dir" ]]; then
            _path_files -/
        elif [[ "$type" == "file" ]]; then
            _path_files -f
        fi
    done

    if [ -n "$completions_with_descriptions" ]; then
        _describe -V unsorted completions_with_descriptions -U
    fi

    if [ -n "$completions" ]; then
        compadd -U -V unsorted -a completions
    fi
}

if [[ $zsh_eval_context[-1] == loadautofunc ]]; then
    _pltr "$@"
else
    compdef _pltr pltr
fi
"""


def generate_fish_completion() -> str:
    """Generate Fish completion script."""
    return """# pltr fish completion script
# Generated by pltr completion install

function _pltr_completion
    set -l response (env COMP_WORDS=(commandline -cp) COMP_CWORD=(commandline -t) _PLTR_COMPLETE=fish_complete pltr)

    for completion in $response
        set -l metadata (string split "," $completion)

        if test $metadata[1] = "dir"
            __fish_complete_directories $metadata[2]
        else if test $metadata[1] = "file"
            __fish_complete_path $metadata[2]
        else if test $metadata[1] = "plain"
            echo $metadata[2]
        end
    end
end

complete -c pltr -f -a "(_pltr_completion)"
"""


def get_default_completion_path(shell: str) -> Path:
    """Get the default path for completion file installation."""
    home = Path.home()

    if shell == "bash":
        # Try bash-completion paths
        paths = [
            home / ".local" / "share" / "bash-completion" / "completions" / "pltr",
            home / ".bash_completion.d" / "pltr",
            Path("/etc/bash_completion.d/pltr"),
        ]
        # Use first writable path
        for p in paths[:-1]:  # Skip system path for user installs
            if p.parent.exists() or not p.parent.parent.exists():
                return p
        return paths[0]

    elif shell == "zsh":
        # Zsh completion paths
        return home / ".zfunc" / "_pltr"

    elif shell == "fish":
        # Fish completion path
        return home / ".config" / "fish" / "completions" / "pltr.fish"

    else:
        raise ValueError(f"Unknown shell: {shell}")


def show_activation_instructions(shell: str, path: Path):
    """Show instructions for activating completions."""
    console.print("\n[yellow]Activation Instructions:[/yellow]")

    if shell == "bash":
        console.print(
            Panel(
                f"""To activate completions, add this to your ~/.bashrc:

[cyan]# Load pltr completions
if [ -f {path} ]; then
    source {path}
fi[/cyan]

Then reload your shell:
[cyan]source ~/.bashrc[/cyan]

Or start a new terminal session.""",
                title="Bash Activation",
                border_style="green",
            )
        )

    elif shell == "zsh":
        console.print(
            Panel(
                f"""To activate completions, add this to your ~/.zshrc:

[cyan]# Add custom completion directory
fpath=({path.parent} $fpath)

# Initialize completions (if not already done)
autoload -Uz compinit && compinit[/cyan]

Then reload your shell:
[cyan]source ~/.zshrc[/cyan]

Or start a new terminal session.""",
                title="Zsh Activation",
                border_style="green",
            )
        )

    elif shell == "fish":
        console.print(
            Panel(
                f"""Completions should be automatically loaded from:
{path}

If not working, reload your shell:
[cyan]exec fish[/cyan]

Or start a new terminal session.""",
                title="Fish Activation",
                border_style="green",
            )
        )


@app.command()
def uninstall(
    shell: Optional[str] = typer.Option(
        None,
        "--shell",
        "-s",
        help="Shell type (bash, zsh, fish). Auto-detected if not specified.",
    ),
):
    """Uninstall shell completions for pltr CLI."""
    # Auto-detect shell if not specified
    if shell is None:
        shell = get_shell_from_env()
        if shell is None:
            console.print(
                "[red]Could not detect shell type. Please specify with --shell option.[/red]"
            )
            raise typer.Exit(1)

    shell = shell.lower()
    if shell not in ["bash", "zsh", "fish"]:
        console.print(f"[red]Unsupported shell: {shell}[/red]")
        raise typer.Exit(1)

    # Get default completion path
    path = get_default_completion_path(shell)

    if path.exists():
        try:
            path.unlink()
            console.print(f"[green]✓[/green] Removed completion file: {path}")
            console.print(
                "\n[yellow]Remember to remove any sourcing commands from your shell config.[/yellow]"
            )
        except Exception as e:
            console.print(f"[red]Failed to remove completion file: {e}[/red]")
            raise typer.Exit(1)
    else:
        console.print(f"[yellow]No completion file found at: {path}[/yellow]")


if __name__ == "__main__":
    app()
