"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ActionsIdentityPoolUtils = exports.ActionsIdentityPoolBase = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_cognito_1 = require("aws-cdk-lib/aws-cognito");
const iam = require("aws-cdk-lib/aws-iam");
const constructs_1 = require("constructs");
const util_1 = require("./util");
class ActionsIdentityPoolBase extends constructs_1.Construct {
    constructor(scope, id, baseProps, allowClassicFlow) {
        super(scope, id);
        this.baseProps = baseProps;
        this._mappedClaims = baseProps.claimMapping.mappedClaims;
        const tagNames = this._mappedClaims.map(x => x.tagName);
        if (this._mappedClaims.length !== new Set(tagNames).size) {
            aws_cdk_lib_1.Annotations.of(this).addError('More than one claim provided maps to the same tag name. Use unique tag names for each claim to map.');
        }
        if (this._mappedClaims.length === 0) {
            aws_cdk_lib_1.Annotations.of(this).addError('No claims have been mapped');
        }
        const enterpriseSlug = scope.node.tryGetContext('@catnekaise/actions-identity-pool:enterpriseSlug');
        let idpSuffix = '';
        if (enterpriseSlug && typeof enterpriseSlug === 'string' && enterpriseSlug.match(/^[a-zA-Z0-9-]+$/)) {
            idpSuffix = `/${enterpriseSlug}`;
        }
        this.openIdConnectProvider = baseProps.openIdConnectProvider ??
            iam.OpenIdConnectProvider.fromOpenIdConnectProviderArn(this, 'IdentityProvider', `arn:aws:iam::${aws_cdk_lib_1.Stack.of(this).account}:oidc-provider/token.actions.githubusercontent.com${idpSuffix}`);
        this.identityPool = new aws_cognito_1.CfnIdentityPool(this, 'Pool', {
            identityPoolName: baseProps.identityPoolName,
            allowUnauthenticatedIdentities: false,
            allowClassicFlow,
            openIdConnectProviderArns: [
                this.openIdConnectProvider.openIdConnectProviderArn,
            ],
        });
        const principalTags = this.mappedClaims.reduce((tags, mappedClaim) => {
            tags[mappedClaim.tagName] = mappedClaim.claim;
            return tags;
        }, {});
        new aws_cognito_1.CfnIdentityPoolPrincipalTag(this, 'PrincipalTags', {
            identityPoolId: this.identityPoolId,
            identityProviderName: this.openIdConnectProvider.openIdConnectProviderArn,
            principalTags,
        });
        new aws_cdk_lib_1.CfnOutput(this, 'PoolId', {
            value: this.identityPool.ref,
            exportName: baseProps.poolIdExportName,
        });
    }
    get identityPoolId() {
        return this.identityPool.ref;
    }
    get mappedClaims() {
        return this._mappedClaims.map(x => ({ ...x }));
    }
    get util() {
        return new ActionsIdentityPoolUtils(this.baseProps.claimMapping);
    }
    /**
     * Create Principal with default Trust Policy for this Identity Pool
     */
    createPrincipalForPool(requirements, amr) {
        // eslint-disable-next-line max-len
        return util_1.ActionsIdentityPrincipalBuilder.create(this.baseProps.claimMapping, this.identityPoolId, this.baseProps.authenticatedMethodReference, this.openIdConnectProvider.openIdConnectProviderArn)
            .createPrincipal(requirements ?? this.baseProps.principalClaimRequirements, amr);
    }
}
exports.ActionsIdentityPoolBase = ActionsIdentityPoolBase;
_a = JSII_RTTI_SYMBOL_1;
ActionsIdentityPoolBase[_a] = { fqn: "@catnekaise/actions-constructs.ActionsIdentityPoolBase", version: "0.2.17" };
class ActionsIdentityPoolUtils {
    constructor(claimMapping) {
        this.claimMapping = claimMapping;
    }
    get chainedPrincipal() {
        return util_1.ActionsIdentityChainedPrincipalBuilder.fromClaimMapping(this.claimMapping);
    }
    get iamResourcePath() {
        return util_1.ActionsIdentityIamResourcePathBuilder.fromClaimMapping(this.claimMapping);
    }
}
exports.ActionsIdentityPoolUtils = ActionsIdentityPoolUtils;
_b = JSII_RTTI_SYMBOL_1;
ActionsIdentityPoolUtils[_b] = { fqn: "@catnekaise/actions-constructs.ActionsIdentityPoolUtils", version: "0.2.17" };
//# sourceMappingURL=data:application/json;base64,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