"""
Agent runtime core implementation.

Provides Agent base class with planning, execution, tool calling,
delegation, memory, caching, and auto-registration on message bus.
"""

from __future__ import annotations

import asyncio
from dataclasses import dataclass
import hashlib
import json
import time
from typing import Any
import json
import uuid

from .config import AgentConfig, BackendFactory, EvenAgeConfig
from .tooling import ToolRegistry, discover_tools


@dataclass
class AgentMemory:
    """Memory interface for agents."""

    agent_name: str
    database: Any  # DatabaseService
    job_id: str | None = None

    def put(self, key: str, value: Any) -> None:
        """Store a memory entry."""
        self.database.memory_put(self.agent_name, key, value, self.job_id)

    def get(self, key: str) -> Any:
        """Retrieve a memory entry."""
        return self.database.memory_get(self.agent_name, key, self.job_id)

    def list(self) -> dict[str, Any]:
        """List all memory entries."""
        return self.database.memory_list(self.agent_name, self.job_id)


class Agent:
    """
    Base Agent class with full runtime capabilities.
    
    Provides:
    - Planning and execution
    - Tool calling with validation and caching
    - Delegation to other agents via message bus
    - Memory storage
    - Auto-registration on message bus
    - Trace logging to database
    
    Subclasses should override plan() method for custom planning logic.
    """

    # Class-level metadata (set by @actor decorator or subclass)
    ROLE: str = "agent"
    GOAL: str = "Complete assigned tasks"

    def __init__(
        self,
        config: AgentConfig,
        env_config: EvenAgeConfig | None = None,
        tools: ToolRegistry | None = None
    ):
        """
        Initialize agent.
        
        Args:
            config: Agent-specific configuration
            env_config: Environment configuration (loaded if None)
            tools: Tool registry (auto-discovered if None)
        """
        self.config = config
        self.env_config = env_config or EvenAgeConfig()

        # Create backends
        self.factory = BackendFactory(self.env_config)
        self.bus = self.factory.create_queue_backend()
        self.database = self.factory.create_database_backend()
        self.llm = self.factory.create_llm_backend()
        self.cache_backend = self.factory.create_cache_backend()
        self.storage = self.factory.create_storage_backend()

        # Tool registry - start with user tools, then add system tools
        self.tools = tools or discover_tools(config.name)

        # Add system tools for delegation and artifact storage
        self._register_system_tools()

        # Memory interface
        self.memory: AgentMemory | None = None

        # Current job context
        self.current_job_id: str | None = None

        # Heartbeat task
        self._heartbeat_task: asyncio.Task | None = None

        # Coordinator support
        self.is_coordinator = getattr(self.env_config, "has_coordinator", False) and (
            self.config.name == getattr(self.env_config, "coordinator_agent", "coordinator")
        )
        self.coordinator_agent = (
            getattr(self.env_config, "coordinator_agent", "coordinator")
            if getattr(self.env_config, "has_coordinator", False)
            else None
        )

    def _register_system_tools(self):
        """Register built-in system tools for delegation and artifacts."""
        from .system_tools import create_system_tools

        system_tools = create_system_tools(self.bus, self.storage)

        for name, func in system_tools.items():
            # Register each system tool
            if not self.tools.has(name):
                self.tools.register(func)

    async def connect_bus(self) -> None:
        """
        Connect to message bus and register agent.
        
        Starts heartbeat to maintain registration.
        """
        metadata = {
            "role": self.config.role or self.ROLE,
            "goal": self.config.goal or self.GOAL,
            "tools": self.tools.list_names(),
            "status": "active",
            "max_iterations": self.config.max_iterations,
            "allow_delegation": self.config.allow_delegation
        }

        await self.bus.register_agent(self.config.name, metadata)
        self.database.register_agent(self.config.name, metadata)

        # Start heartbeat
        if self._heartbeat_task is None:
            self._heartbeat_task = asyncio.create_task(self._heartbeat_loop())

    async def _heartbeat_loop(self) -> None:
        """Periodic heartbeat to keep agent registered."""
        while True:
            try:
                await asyncio.sleep(30)  # Every 30 seconds
                metadata = {
                    "role": self.config.role,
                    "goal": self.config.goal,
                    "tools": self.tools.list_names(),
                    "status": "active"
                }
                await self.bus.register_agent(self.config.name, metadata)
            except asyncio.CancelledError:
                break
            except Exception:
                continue

    async def shutdown(self) -> None:
        """Shutdown agent and cleanup."""
        if self._heartbeat_task:
            self._heartbeat_task.cancel()
            try:
                await self._heartbeat_task
            except asyncio.CancelledError:
                pass

        # Update status to inactive
        metadata = {
            "role": self.config.role,
            "goal": self.config.goal,
            "tools": self.tools.list_names(),
            "status": "inactive"
        }
        await self.bus.register_agent(self.config.name, metadata)

    def plan(self, task: dict) -> list[dict]:
        """
        Create execution plan for a task.
        
        Default implementation: simple single-step plan.
        Subclasses should override for custom planning logic.
        
        Args:
            task: Task inputs
        
        Returns:
            List of steps, where each step is a dict with:
            - action: "tool" | "delegate" | "llm" | "custom"
            - Additional fields depending on action type
        """
        # Default: single LLM call step
        return [
            {
                "action": "llm",
                "prompt": f"Task: {json.dumps(task)}",
                "system": f"You are {self.config.role}. Your goal: {self.config.goal}"
            }
        ]

    async def handle(self, task: dict) -> dict:
        """
        Handle a task: plan, execute, return result.
        
        This is the main entry point for task execution.
        
        Args:
            task: Task inputs
        
        Returns:
            Structured result dict with status and output
        """
        # Set up job context (support both job_id and prompt_id for compatibility)
        execution_id = task.get("prompt_id") or task.get("job_id") or str(uuid.uuid4())
        self.current_job_id = execution_id  # Keep internal name for now

        self.memory = AgentMemory(
            agent_name=self.config.name,
            database=self.database,
            job_id=execution_id
        )

        # Trace start (log as prompt_id for new terminology)
        self.database.append_trace(
            execution_id,
            self.config.name,
            "task_start",
            {"inputs": task}
        )

        try:
            # Plan
            plan = self.plan(task)

            self.database.append_trace(
                execution_id,
                self.config.name,
                "plan_created",
                {"plan": plan}
            )

            # Execute
            result = await self.execute_plan(plan, task)

            # Trace completion
            self.database.append_trace(
                execution_id,
                self.config.name,
                "task_complete",
                {"result": result}
            )

            # Auto-route result to coordinator if applicable
            if not self.is_coordinator and self.coordinator_agent:
                # Send result back to coordinator for synthesis
                try:
                    await self.bus.publish_task(
                        self.coordinator_agent,
                        {
                            "type": "specialist_result",
                            "source_agent": self.config.name,
                            "job_id": execution_id,
                            "result": result
                        }
                    )
                except Exception:
                    # Non-fatal: continue even if coordinator routing fails
                    pass

            return {
                "status": "success",
                "result": result,
                "job_id": execution_id,  # Keep for backward compatibility
                "prompt_id": execution_id  # Add new terminology
            }

        except Exception as e:
            # Trace error
            self.database.append_trace(
                execution_id,
                self.config.name,
                "task_error",
                {"error": str(e)}
            )

            return {
                "status": "error",
                "error": str(e),
                "job_id": execution_id,
                "prompt_id": execution_id
            }

    async def execute_plan(self, plan: list[dict], task: dict) -> Any:
        """
        Execute a plan.
        
        Iterates through steps and executes each based on action type.
        
        Args:
            plan: List of steps
            task: Original task inputs
        
        Returns:
            Final result (from last step)
        """
        context = {"task": task}
        result = None

        for i, step in enumerate(plan):
            action = step.get("action")

            if action == "tool":
                result = await self._execute_tool_step(step, context)
            elif action == "delegate":
                result = await self._execute_delegate_step(step, context)
            elif action == "llm":
                result = await self._execute_llm_step(step, context)
            elif action == "custom":
                result = await self._execute_custom_step(step, context)
            else:
                result = {"error": f"Unknown action: {action}"}

            context[f"step_{i}"] = result
            context["last"] = result

        return result

    async def _execute_tool_step(self, step: dict, context: dict) -> Any:
        """Execute a tool call step."""
        tool_name = step.get("name")
        params = step.get("params", {})
        use_cache = step.get("use_cache", True)

        return await self.call_tool(tool_name, params, use_cache=use_cache)

    async def _execute_delegate_step(self, step: dict, context: dict) -> Any:
        """Execute a delegation step."""
        agent_name = step.get("agent")
        payload = step.get("payload", {})

        task_id = await self.delegate(agent_name, payload)

        # Optionally wait for response (blocking)
        wait = step.get("wait", False)
        if wait:
            timeout = step.get("timeout", 30)
            response = await self.bus.wait_for_response(task_id, timeout)
            return response or {"status": "pending", "task_id": task_id}

        return {"status": "delegated", "task_id": task_id}

    async def _execute_llm_step(self, step: dict, context: dict) -> Any:
        """Execute an LLM call step."""
        prompt = step.get("prompt", "")
        system = step.get("system")

        # Simple string replacement for context variables
        for key, value in context.items():
            try:
                rep = value if isinstance(value, str) else json.dumps(value, ensure_ascii=False)
            except Exception:
                rep = str(value)
            prompt = prompt.replace(f"{{{key}}}", rep)

        response = await self.llm.generate(prompt, system=system)
        return {"llm_response": response}

    async def _execute_custom_step(self, step: dict, context: dict) -> Any:
        """Execute a custom step (override in subclass)."""
        return {"error": "Custom step not implemented"}

    async def call_tool(
        self,
        name: str,
        params: dict[str, Any],
        use_cache: bool = True
    ) -> Any:
        """
        Call a tool with validation, guardrails, and caching.
        
        Args:
            name: Tool name
            params: Tool parameters
            use_cache: Whether to use cache
        
        Returns:
            Tool execution result
        """
        tool = self.tools.get(name)
        if not tool:
            raise ValueError(f"Tool not found: {name}")

        # Cache key
        cache_key = self._tool_cache_key(name, params)

        # Check cache
        if use_cache:
            cached = await self._get_from_cache(cache_key)
            if cached is not None:
                self.database.append_trace(
                    self.current_job_id or "unknown",
                    self.config.name,
                    "cache_hit",
                    {"tool": name, "params": params}
                )
                return cached

        # Validate inputs
        try:
            validated_params = tool.validate_inputs(params)
        except Exception as e:
            raise ValueError(f"Tool input validation failed: {e}")

        # Trace tool call
        self.database.append_trace(
            self.current_job_id or "unknown",
            self.config.name,
            "tool_call",
            {"tool": name, "params": validated_params}
        )

        # Execute
        start = time.time()
        try:
            result = tool.invoke(**validated_params)
            duration_ms = int((time.time() - start) * 1000)

            self.database.append_trace(
                self.current_job_id or "unknown",
                self.config.name,
                "tool_result",
                {
                    "tool": name,
                    "duration_ms": duration_ms,
                    "status": "success"
                }
            )

            # Cache result
            if use_cache:
                await self._put_in_cache(cache_key, result, ttl=300)

            return result

        except Exception as e:
            self.database.append_trace(
                self.current_job_id or "unknown",
                self.config.name,
                "tool_error",
                {"tool": name, "error": str(e)}
            )
            raise

    async def delegate(self, agent_name: str, payload: dict) -> str:
        """
        Delegate a task to another agent.
        
        Args:
            agent_name: Target agent name
            payload: Task payload
        
        Returns:
            Task ID for tracking
        """
        if not self.config.allow_delegation:
            raise ValueError("Delegation not allowed for this agent")

        task_id = await self.bus.publish_task(agent_name, payload)

        self.database.append_trace(
            self.current_job_id or "unknown",
            self.config.name,
            "delegation",
            {
                "target_agent": agent_name,
                "task_id": task_id,
                "payload": payload
            }
        )

        return task_id

    def _tool_cache_key(self, name: str, params: dict) -> str:
        """Generate cache key for tool call."""
        param_str = json.dumps(params, sort_keys=True)
        key_data = f"{name}:{param_str}"
        return f"tool:{hashlib.md5(key_data.encode()).hexdigest()}"

    async def _get_from_cache(self, key: str) -> Any:
        """Get value from cache backend."""
        try:
            if hasattr(self.cache_backend, 'get') and asyncio.iscoroutinefunction(self.cache_backend.get):
                return await self.cache_backend.get(key)
            return self.cache_backend.get(key)
        except Exception:
            return None

    async def _put_in_cache(self, key: str, value: Any, ttl: int | None = None) -> None:
        """Put value in cache backend."""
        try:
            if hasattr(self.cache_backend, 'set') and asyncio.iscoroutinefunction(self.cache_backend.set):
                await self.cache_backend.set(key, value, ttl)
            else:
                self.cache_backend.set(key, value, ttl)
        except Exception:
            pass
