"""
Vector store backend implementations.

Provides pluggable vector stores: LocalVectorStore (filesystem), PostgresVectorStore (pgvector).
"""

from __future__ import annotations

from contextlib import contextmanager
import json
from pathlib import Path

from sqlalchemy import create_engine, text
from sqlalchemy.engine import Engine


class LocalVectorStore:
    """
    Local filesystem-based vector store for dev.
    
    Stores vectors as JSON files. Not suitable for production scale.
    """

    def __init__(self, storage_path: str = "./.evenage/vectors"):
        self.storage_path = Path(storage_path)
        self.storage_path.mkdir(parents=True, exist_ok=True)

    def store(self, key: str, vector: list[float], metadata: dict) -> None:
        """Store a vector with metadata."""
        data = {
            "key": key,
            "vector": vector,
            "metadata": metadata
        }

        file_path = self.storage_path / f"{self._sanitize_key(key)}.json"
        with open(file_path, "w") as f:
            json.dump(data, f)

    def search(self, vector: list[float], k: int = 5) -> list[dict]:
        """
        Search for k nearest vectors.
        
        Uses simple cosine similarity (inefficient for large scale).
        """
        results = []

        for file_path in self.storage_path.glob("*.json"):
            try:
                with open(file_path) as f:
                    data = json.load(f)

                stored_vector = data["vector"]
                similarity = self._cosine_similarity(vector, stored_vector)

                results.append({
                    "key": data["key"],
                    "metadata": data["metadata"],
                    "similarity": similarity
                })
            except Exception:
                continue

        # Sort by similarity and return top k
        results.sort(key=lambda x: x["similarity"], reverse=True)
        return results[:k]

    def delete(self, key: str) -> None:
        """Delete a vector by key."""
        file_path = self.storage_path / f"{self._sanitize_key(key)}.json"
        if file_path.exists():
            file_path.unlink()

    def _sanitize_key(self, key: str) -> str:
        """Sanitize key for use as filename."""
        return key.replace("/", "_").replace("\\", "_").replace(":", "_")

    def _cosine_similarity(self, a: list[float], b: list[float]) -> float:
        """Compute cosine similarity between two vectors."""
        if len(a) != len(b):
            return 0.0

        dot_product = sum(x * y for x, y in zip(a, b))
        magnitude_a = sum(x * x for x in a) ** 0.5
        magnitude_b = sum(y * y for y in b) ** 0.5

        if magnitude_a == 0 or magnitude_b == 0:
            return 0.0

        return dot_product / (magnitude_a * magnitude_b)


class PostgresVectorStore:
    """
    Postgres vector store using pgvector extension.

    Requires the database to support the 'vector' extension (our docker image does).
    """

    def __init__(self, database_url: str, dimension: int = 1536, table: str = "vectors"):
        self.database_url = database_url
        self.dimension = dimension
        self.table = table
        self._engine: Engine | None = None
        self._ensure_schema()

    def _get_engine(self) -> Engine:
        if self._engine is None:
            self._engine = create_engine(self.database_url, echo=False)
        return self._engine

    @contextmanager
    def _conn(self):  # type: () -> Generator
        engine = self._get_engine()
        with engine.begin() as conn:
            yield conn

    def _ensure_schema(self):
        try:
            with self._conn() as conn:
                # Enable extension and create table if not exists
                conn.execute(text("CREATE EXTENSION IF NOT EXISTS vector"))
                conn.execute(text(
                    f"""
                    CREATE TABLE IF NOT EXISTS {self.table} (
                        key TEXT PRIMARY KEY,
                        embedding VECTOR({self.dimension}),
                        metadata JSONB
                    )
                    """
                ))
                # Basic index for cosine distance
                conn.execute(text(
                    f"CREATE INDEX IF NOT EXISTS idx_{self.table}_embedding ON {self.table} USING ivfflat (embedding vector_cosine_ops) WITH (lists = 100)"
                ))
        except Exception:
            # If ensuring schema fails, we'll raise on first use instead
            pass

    def store(self, key: str, vector: list[float], metadata: dict) -> None:
        """Upsert a vector with metadata."""
        if not isinstance(vector, list) or not vector:
            raise ValueError("vector must be a non-empty list[float]")

        # pgvector expects a vector literal; cast the string to vector
        vec_literal = "[" + ",".join(str(float(x)) for x in vector) + "]"
        with self._conn() as conn:
            conn.execute(
                text(
                    f"""
                    INSERT INTO {self.table} (key, embedding, metadata)
                    VALUES (:key, (:embedding)::vector, CAST(:metadata AS JSONB))
                    ON CONFLICT (key) DO UPDATE SET
                        embedding = EXCLUDED.embedding,
                        metadata = EXCLUDED.metadata
                    """
                ),
                {
                    "key": key,
                    "embedding": vec_literal,
                    "metadata": json.dumps(metadata or {}),
                },
            )

    def search(self, vector: list[float], k: int = 5) -> list[dict]:
        """Find top-k nearest neighbors by cosine distance."""
        if not isinstance(vector, list) or not vector:
            raise ValueError("vector must be a non-empty list[float]")

        vec_literal = "[" + ",".join(str(float(x)) for x in vector) + "]"
        with self._conn() as conn:
            rows = conn.execute(
                text(
                    f"""
                    SELECT key, metadata,
                           1 - (embedding <=> (:qvec)::vector) AS similarity
                    FROM {self.table}
                    ORDER BY embedding <=> (:qvec)::vector
                    LIMIT :k
                    """
                ),
                {"qvec": vec_literal, "k": k},
            ).mappings().all()

        results = []
        for r in rows:
            try:
                meta = r["metadata"] if isinstance(r["metadata"], dict) else json.loads(r["metadata"]) if r["metadata"] else {}
            except Exception:
                meta = {}
            results.append({"key": r["key"], "metadata": meta, "similarity": float(r["similarity"])})
        return results

    def delete(self, key: str) -> None:
        with self._conn() as conn:
            conn.execute(text(f"DELETE FROM {self.table} WHERE key = :key"), {"key": key})
