"""
LLM connector helpers.

Provides connect_llm() to construct an LLM backend explicitly.
"""
from __future__ import annotations

from typing import Any, Literal


def connect_llm(
    backend: Literal["noop", "openai", "anthropic", "gemini", "http", "vllm"],
    *,
    model: str | None = None,
    openai_api_key: str | None = None,
    openai_base_url: str | None = None,
    anthropic_api_key: str | None = None,
    gemini_api_key: str | None = None,
    http_endpoint: str | None = None,
) -> Any:
    from evenage.core.llm import HTTPLLM, AnthropicLLM, GeminiLLM, NoOpLLM, OpenAILLM

    if backend == "noop":
        return NoOpLLM()
    if backend == "openai":
        return OpenAILLM(openai_api_key, model, base_url=openai_base_url)
    if backend == "anthropic":
        return AnthropicLLM(anthropic_api_key, model)
    if backend == "gemini":
        return GeminiLLM(gemini_api_key, model)
    if backend == "http":
        if not http_endpoint:
            raise ValueError("http_endpoint is required for http LLM backend")
        return HTTPLLM(http_endpoint)
    if backend == "vllm":
        # vLLM typically exposes an OpenAI-compatible API; use OpenAI client with base_url
        if not openai_base_url:
            raise ValueError("openai_base_url is required for vllm backend")
        return OpenAILLM(openai_api_key, model, base_url=openai_base_url)
    raise ValueError(f"Unsupported llm backend: {backend}")
