<img align="right" src="https://raw.githubusercontent.com/mcbeet/beet/main/logo.png?sanitize=true" alt="logo" width="76">

# Beet

[![GitHub Actions](https://github.com/mcbeet/beet/workflows/CI/badge.svg)](https://github.com/mcbeet/beet/actions)
[![PyPI](https://img.shields.io/pypi/v/beet.svg)](https://pypi.org/project/beet/)
[![PyPI - Python Version](https://img.shields.io/pypi/pyversions/beet.svg)](https://pypi.org/project/beet/)
[![Discord](https://img.shields.io/discord/900530660677156924?color=7289DA&label=discord&logo=discord&logoColor=fff)](https://discord.gg/98MdSGMm8j)

> The Minecraft pack development kit.

## Introduction

Minecraft [resource packs](https://minecraft.wiki/w/Resource_pack) and [data packs](https://minecraft.wiki/w/Data_pack) work well as _distribution_ formats but can be pretty limiting as _authoring_ formats. You can quickly end up having to manage hundreds of files, some of which might be buried within the bundled output of various generators.

The `beet` project is a development kit that tries to unify data pack and resource pack tooling into a single pipeline. The community is always coming up with pre-processors, frameworks, and generators of all kinds to make the developer experience more ergonomic. With `beet` you can seamlessly integrate all these tools in your project.

### Screencasts

- **Quick start** [https://youtu.be/JGrJTOhG3pY](https://youtu.be/JGrJTOhG3pY)
- **Command-line** [https://youtu.be/fQ9up0ELPNE](https://youtu.be/fQ9up0ELPNE)
- **Library overview** [https://youtu.be/LDvV4_l-PSc](https://youtu.be/LDvV4_l-PSc)
- **Plugins basics** [https://youtu.be/XTzKmvHqd1g](https://youtu.be/XTzKmvHqd1g)
- **Pipeline configuration** [https://youtu.be/QsnQncGxAAs](https://youtu.be/QsnQncGxAAs)

### Library

```python
from beet import ResourcePack, Texture

# Open a zipped resource pack and add a custom stone texture
with ResourcePack(path="stone.zip") as assets:
    assets["minecraft:block/stone"] = Texture(source_path="custom.png")
```

The `beet` library provides carefully crafted primitives for working with Minecraft resource packs and data packs.

- Create, read, edit and merge resource packs and data packs
- Handle zipped and unzipped packs
- Fast and lazy by default, files are transparently loaded when needed
- Statically typed API enabling rich intellisense and autocompletion
- First-class [`pytest`](https://github.com/pytest-dev/pytest/) integration with detailed assertion explanations

### Toolchain

```python
from beet import Context, Function

def greet(ctx: Context):
    """Plugin that adds a function for greeting the player."""
    ctx.data["greet:hello"] = Function(["say hello"], tags=["minecraft:load"])
```

The `beet` toolchain is designed to support a wide range of use-cases. The most basic pipeline will let you create configurable resource packs and data packs, but plugins make it easy to implement arbitrarily advanced workflows and tools like linters, asset generators and function pre-processors.

- Compose plugins that can inspect and edit the generated resource pack and data pack
- Configure powerful build systems for development and creating releases
- First-class template integration approachable without prior Python knowledge
- Link the generated resource pack and data pack to Minecraft
- Automatically rebuild the project on file changes with watch mode
- Batteries-included package that comes with a few handy plugins out of the box
- Rich ecosystem, extensible CLI, and powerful generator and worker API

## Installation

We recommend `uv` (https://github.com/astral-sh/uv#installation). With `uv` installed, you can try `beet` by running `uvx beet`. You can also install `beet` as a global tool on your machine.

```console
$ uv tool install beet
```

> If you see the message `warning: ... is not on your PATH`, you'll need to add the specified directory to your global path to invoke `beet` directly instead of using `uvx beet`.

You can make sure that `beet` was successfully installed by trying to use the toolchain from the command-line.

```console
$ beet --help
Usage: beet [OPTIONS] COMMAND [ARGS]...

  The beet toolchain.

Options:
  -p, --project PATH  Select project.
  -s, --set OPTION    Set config option.
  -l, --log LEVEL     Configure output verbosity.
  -v, --version       Show the version and exit.
  -h, --help          Show this message and exit.

Commands:
  build  Build the current project.
  cache  Inspect or clear the cache.
  link   Link the generated resource pack and data pack to Minecraft.
  watch  Watch the project directory and build on file changes.
```

## Contributing

Contributions are welcome. Make sure to first open an issue discussing the problem or the new feature before creating a pull request. The project uses [`uv`](https://github.com/astral-sh/uv).

```console
$ uv sync
```

You can run the tests with `uv run pytest`. We use [`pytest-minecraft`](https://github.com/vberlier/pytest-minecraft) to run tests against actual Minecraft releases.

```console
$ uv run pytest
$ uv run pytest --minecraft-latest
```

We also use [`pytest-insta`](https://github.com/vberlier/pytest-insta) for snapshot testing. Data pack and resource pack snapshots make it easy to monitor and review changes.

```console
$ uv run pytest --insta review
```

The code is formatted and checked with [`ruff`](https://github.com/astral-sh/ruff).

```console
$ uv run ruff format
$ uv run ruff check
```

---

License - [MIT](https://github.com/mcbeet/beet/blob/main/LICENSE)
