# OIDC Conformance Testing Instructions

**Purpose**: AI Assistant instructions for configuring and running OpenID Connect conformance tests against Authly.

## Overview

Authly implements complete OpenID Connect Core 1.0 + Session Management 1.0 compliance with 551 passing tests. This document provides step-by-step instructions for validating Authly's OIDC implementation using the official OpenID Foundation conformance test suite.

## Authly OIDC Endpoints

Authly provides all required OIDC endpoints for conformance testing:

### Discovery Endpoints
- **OIDC Discovery**: `/.well-known/openid_configuration`
- **OAuth Discovery**: `/.well-known/oauth-authorization-server`

### Core OIDC Endpoints
- **JWKS**: `/.well-known/jwks.json`
- **Authorization**: `/api/v1/oauth/authorize`
- **Token**: `/api/v1/auth/token`
- **UserInfo**: `/oidc/userinfo`
- **End Session**: `/oidc/logout`
- **Token Revocation**: `/api/v1/auth/revoke`

### Session Management Endpoints
- **Session iframe**: `/oidc/session/iframe`
- **Session check**: `/oidc/session/check`
- **Front-channel logout**: `/oidc/frontchannel/logout`

## Conformance Suite Setup

### Prerequisites
1. Docker and docker-compose installed
2. Java 17+ for conformance suite
3. Authly running on `http://localhost:8000`

### Step 1: Clone Conformance Suite
```bash
git clone https://gitlab.com/openid/conformance-suite.git
cd conformance-suite
```

### Step 2: Start Authly
Ensure Authly is running with proper configuration:
```bash
# From Authly project root
uv run python -m authly.main
```

### Step 3: Create Test Client
Create an OIDC test client in Authly:
```bash
# Using Authly admin CLI or API
python -m authly.cli create-client \
  --client-id "oidc-conformance-test" \
  --client-name "OIDC Conformance Test Client" \
  --client-type "confidential" \
  --redirect-uris "https://localhost:8443/test/a/authly/callback" \
  --post-logout-redirect-uris "https://localhost:8443/test/a/authly/post_logout_callback" \
  --response-types "code" \
  --grant-types "authorization_code,refresh_token" \
  --scopes "openid,profile,email,phone,address"
```

### Step 4: Configure Authly for Testing
Ensure these settings in Authly configuration:
```python
# CORS configuration for conformance suite
CORS_ORIGINS = ["https://localhost:8443", "http://localhost:8443"]
OIDC_ENABLED = True
JWT_ALGORITHM = "RS256"  # Required for OIDC compliance
```

### Step 5: Start Conformance Suite
```bash
# In conformance-suite directory
docker-compose up -d

# Access web interface
open https://localhost:8443
```

## Test Configuration

### Basic OIDC Core Test Plan
1. **Test Plan**: "OpenID Connect Core: Basic Certification Profile Authorization Code Flow"
2. **Server Configuration**:
   - **Issuer**: `http://localhost:8000`
   - **Discovery URL**: `http://localhost:8000/.well-known/openid_configuration`

3. **Client Configuration**:
   - **Client ID**: `oidc-conformance-test`
   - **Client Secret**: `[generated by Authly]`
   - **Redirect URI**: `https://localhost:8443/test/a/authly/callback`

### Advanced Test Plans
For comprehensive testing, also run:
- OpenID Connect Core: Implicit Certification Profile
- OpenID Connect Core: Hybrid Certification Profile
- OpenID Connect Session Management
- OpenID Connect Front-Channel Logout
- OpenID Connect Back-Channel Logout (if implemented)

## Expected Test Results

### Should PASS ✅
Based on Authly's implementation (551 tests passing):
- Discovery endpoint compliance
- Authorization Code Flow with PKCE
- ID Token generation with proper claims
- JWKS endpoint with RSA key validation
- UserInfo endpoint with scope-based filtering
- Token refresh with ID token renewal
- Session management iframe functionality
- Front-channel logout coordination
- End session endpoint validation

### Potential Issues ⚠️
Monitor these areas during testing:
- CORS configuration for conformance suite URLs
- Certificate validation (use HTTP for local testing)
- Client authentication method compatibility
- Scope mapping and claim filtering accuracy

## Troubleshooting Guide

### Common Issues and Solutions

**CORS Errors**:
```python
# Add to Authly configuration
CORS_ORIGINS = ["https://localhost:8443", "http://localhost:8443"]
```

**Discovery Endpoint Issues**:
- Verify `/.well-known/openid_configuration` returns proper JSON
- Check all endpoint URLs are absolute and accessible

**Client Authentication Failures**:
- Ensure client credentials match between Authly and test config
- Verify client_type is set to "confidential" for secret-based auth

**ID Token Validation Errors**:
- Check JWKS endpoint accessibility
- Verify RSA key format and algorithm (RS256)
- Ensure proper audience and issuer claims

**Session Management Issues**:
- Verify session iframe endpoint accessibility
- Check session state parameter generation
- Validate front-channel logout URL construction

## Post-Testing Analysis

### Documentation Updates
After successful conformance testing:
1. Update README.md with certification status
2. Add conformance badge to documentation
3. Document any configuration requirements
4. Note any limitations or known issues

### Certification Process
For official OpenID Connect certification:
1. Run all required test plans successfully
2. Submit attestation statement
3. Follow OpenID Foundation certification process
4. Maintain certification with ongoing testing

## AI Assistant Notes

When helping with OIDC conformance testing:
1. Always verify Authly is running before starting tests
2. Check that all required endpoints are accessible
3. Monitor test logs for specific failure patterns
4. Suggest configuration adjustments based on test results
5. Document any discovered issues for future reference

## Implementation Status

**Current Authly OIDC Compliance**:
- ✅ OpenID Connect Core 1.0 - COMPLETE
- ✅ Session Management 1.0 - COMPLETE  
- ✅ Front-Channel Logout 1.0 - COMPLETE
- ⚠️ Back-Channel Logout 1.0 - NOT IMPLEMENTED
- ⚠️ OIDC Dynamic Registration - NOT IMPLEMENTED

**Test Coverage**: 551 tests passing (100% success rate)
**OIDC-Specific Tests**: 221 tests across 15 test files

This comprehensive OIDC implementation should achieve high conformance test pass rates across all core certification profiles.