#!/bin/bash
# Master Integration Test Runner
# Orchestrates all integration tests for comprehensive full-stack testing

set -euo pipefail

# Source helper functions and configuration
SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
source "$SCRIPT_DIR/../helpers/common.sh"
source "$SCRIPT_DIR/../helpers/config.sh"

# Global test state
TOTAL_TESTS=0
PASSED_TESTS=0
FAILED_TESTS=0
SKIPPED_TESTS=0
TEST_START_TIME=0
TEST_RESULTS=()

# Test configuration
RUN_USER_TESTS="${RUN_USER_TESTS:-true}"
RUN_CLIENT_TESTS="${RUN_CLIENT_TESTS:-true}"
RUN_SCOPE_TESTS="${RUN_SCOPE_TESTS:-true}"
RUN_USER_AUTH_TESTS="${RUN_USER_AUTH_TESTS:-true}"
RUN_OAUTH_TESTS="${RUN_OAUTH_TESTS:-true}"
CLEANUP_ON_SUCCESS="${CLEANUP_ON_SUCCESS:-true}"
CLEANUP_ON_FAILURE="${CLEANUP_ON_FAILURE:-true}"

# Function to record test result
record_test_result() {
    local test_name="$1"
    local status="$2"
    local duration="$3"
    local details="${4:-}"
    
    TEST_RESULTS+=("$test_name|$status|$duration|$details")
    
    case "$status" in
        "PASSED")
            PASSED_TESTS=$((PASSED_TESTS + 1))
            log_success "✓ $test_name completed in ${duration}s"
            ;;
        "FAILED")
            FAILED_TESTS=$((FAILED_TESTS + 1))
            log_error "✗ $test_name failed in ${duration}s: $details"
            ;;
        "SKIPPED")
            SKIPPED_TESTS=$((SKIPPED_TESTS + 1))
            log_warning "○ $test_name skipped: $details"
            ;;
    esac
    
    TOTAL_TESTS=$((TOTAL_TESTS + 1))
}

# Function to run a test script with error handling
run_test_script() {
    local script_name="$1"
    local script_path="$SCRIPT_DIR/$script_name"
    local test_description="$2"
    local required="${3:-true}"
    
    log_info "Running $test_description..."
    
    if [[ ! -f "$script_path" ]]; then
        record_test_result "$test_description" "SKIPPED" "0" "Script not found: $script_path"
        return 0
    fi
    
    local test_start=$(date +%s)
    local exit_code=0
    
    # Run the test script and capture output
    if AUTHLY_ADMIN_PASSWORD="$AUTHLY_ADMIN_PASSWORD" "$script_path" >/dev/null 2>&1; then
        local test_end=$(date +%s)
        local duration=$((test_end - test_start))
        record_test_result "$test_description" "PASSED" "$duration"
        return 0
    else
        exit_code=$?
        local test_end=$(date +%s)
        local duration=$((test_end - test_start))
        
        if [[ "$required" == "true" ]]; then
            record_test_result "$test_description" "FAILED" "$duration" "Exit code: $exit_code"
            return 1
        else
            record_test_result "$test_description" "SKIPPED" "$duration" "Optional test failed with exit code: $exit_code"
            return 0
        fi
    fi
}

# Function to check infrastructure prerequisites
check_infrastructure() {
    log_info "Checking infrastructure prerequisites..."
    
    local check_start=$(date +%s)
    
    # Check if Docker services are running
    if ! check_docker_services; then
        local check_end=$(date +%s)
        local duration=$((check_end - check_start))
        record_test_result "Infrastructure Check" "FAILED" "$duration" "Docker services not healthy"
        return 1
    fi
    
    # Check if Authly service is responding
    if ! wait_for_service "$HEALTH_ENDPOINT" 30; then
        local check_end=$(date +%s)
        local duration=$((check_end - check_start))
        record_test_result "Infrastructure Check" "FAILED" "$duration" "Authly service not responding"
        return 1
    fi
    
    # Verify configuration
    if ! validate_config; then
        local check_end=$(date +%s)
        local duration=$((check_end - check_start))
        record_test_result "Infrastructure Check" "FAILED" "$duration" "Configuration validation failed"
        return 1
    fi
    
    local check_end=$(date +%s)
    local duration=$((check_end - check_start))
    record_test_result "Infrastructure Check" "PASSED" "$duration"
    return 0
}

# Function to run core admin tests
run_core_tests() {
    log_info "Running core administrative tests..."
    
    # Test 1: Admin Authentication
    run_test_script "admin-auth.sh" "Admin Authentication" "true" || return 1
    
    # Test 2: Scope Management (required for other tests)
    if [[ "$RUN_SCOPE_TESTS" == "true" ]]; then
        run_test_script "scope-management.sh" "Scope Management" "true" || return 1
    else
        record_test_result "Scope Management" "SKIPPED" "0" "Disabled by configuration"
    fi
    
    # Test 3: Client Management
    if [[ "$RUN_CLIENT_TESTS" == "true" ]]; then
        run_test_script "client-management.sh" "Client Management" "true" || return 1
    else
        record_test_result "Client Management" "SKIPPED" "0" "Disabled by configuration"
    fi
    
    return 0
}

# Function to run user management tests
run_user_tests() {
    log_info "Running user management tests..."
    
    if [[ "$RUN_USER_TESTS" == "true" ]]; then
        # User management is optional since it may not be implemented
        run_test_script "user-management.sh" "User Management" "false"
    else
        record_test_result "User Management" "SKIPPED" "0" "Disabled by configuration"
    fi
    
    return 0
}

# Function to run user authentication tests
run_user_auth_tests() {
    log_info "Running user authentication tests..."
    
    if [[ "$RUN_USER_AUTH_TESTS" == "true" ]]; then
        # User authentication tests OAuth 2.1 password grant and OIDC userinfo
        run_test_script "user-auth.sh" "User Authentication" "true" || return 1
    else
        record_test_result "User Authentication" "SKIPPED" "0" "Disabled by configuration"
    fi
    
    return 0
}

# Function to run OAuth flow tests (if implemented)
run_oauth_flow_tests() {
    log_info "Running OAuth flow tests..."
    
    if [[ "$RUN_OAUTH_TESTS" == "true" ]]; then
        # OAuth flow tests - comprehensive authorization code flow testing
        if [[ -f "$SCRIPT_DIR/oauth-flow.sh" ]]; then
            run_test_script "oauth-flow.sh" "OAuth Flow Testing" "true" || return 1
        else
            record_test_result "OAuth Flow Testing" "SKIPPED" "0" "OAuth flow script not implemented"
        fi
    else
        record_test_result "OAuth Flow Testing" "SKIPPED" "0" "Disabled by configuration"
    fi
    
    return 0
}

# Function to perform cleanup
run_cleanup() {
    local cleanup_type="$1"
    local description="$2"
    
    log_info "Running $description..."
    
    local cleanup_start=$(date +%s)
    
    if [[ -f "$SCRIPT_DIR/cleanup.sh" ]]; then
        if AUTHLY_ADMIN_PASSWORD="$AUTHLY_ADMIN_PASSWORD" "$SCRIPT_DIR/cleanup.sh" "$cleanup_type" >/dev/null 2>&1; then
            local cleanup_end=$(date +%s)
            local duration=$((cleanup_end - cleanup_start))
            record_test_result "$description" "PASSED" "$duration"
            return 0
        else
            local cleanup_end=$(date +%s)
            local duration=$((cleanup_end - cleanup_start))
            record_test_result "$description" "FAILED" "$duration" "Cleanup script failed"
            return 1
        fi
    else
        record_test_result "$description" "SKIPPED" "0" "Cleanup script not found"
        return 0
    fi
}

# Function to display test summary
display_test_summary() {
    local total_duration=$(($(date +%s) - TEST_START_TIME))
    
    log_info "=== Full Stack Integration Test Summary ==="
    echo
    log_info "Test Results:"
    log_success "  ✓ Passed: $PASSED_TESTS"
    log_error "  ✗ Failed: $FAILED_TESTS" 
    log_warning "  ○ Skipped: $SKIPPED_TESTS"
    log_info "  Total: $TOTAL_TESTS"
    echo
    log_info "Execution Time: ${total_duration}s"
    echo
    
    if [[ ${#TEST_RESULTS[@]} -gt 0 ]]; then
        log_info "Detailed Results:"
        printf "%-25s %-8s %-8s %s\n" "Test Name" "Status" "Duration" "Details"
        printf "%-25s %-8s %-8s %s\n" "-------------------------" "--------" "--------" "-------"
        
        for result in "${TEST_RESULTS[@]}"; do
            IFS='|' read -r name status duration details <<< "$result"
            printf "%-25s %-8s %-8ss %s\n" "$name" "$status" "$duration" "$details"
        done
        echo
    fi
    
    # Overall result
    if [[ $FAILED_TESTS -eq 0 ]]; then
        log_success "🎉 All tests completed successfully!"
        return 0
    else
        log_error "❌ Some tests failed. Check the results above."
        return 1
    fi
}

# Function to show test configuration
show_configuration() {
    log_info "Full Stack Integration Test Configuration:"
    echo "  Base URL: $AUTHLY_BASE_URL"
    echo "  Admin User: $ADMIN_USERNAME"
    echo "  Test Prefixes: $TEST_USER_PREFIX, $TEST_CLIENT_PREFIX, $TEST_SCOPE_PREFIX"
    echo
    log_info "Test Modules:"
    echo "  User Management Tests: $RUN_USER_TESTS"
    echo "  Client Management Tests: $RUN_CLIENT_TESTS"
    echo "  Scope Management Tests: $RUN_SCOPE_TESTS"
    echo "  User Authentication Tests: $RUN_USER_AUTH_TESTS"
    echo "  OAuth Flow Tests: $RUN_OAUTH_TESTS"
    echo
    log_info "Cleanup Configuration:"
    echo "  Cleanup on Success: $CLEANUP_ON_SUCCESS"
    echo "  Cleanup on Failure: $CLEANUP_ON_FAILURE"
    echo
}

# Function to run comprehensive test suite
run_comprehensive_tests() {
    log_info "Starting comprehensive full-stack integration tests..."
    TEST_START_TIME=$(date +%s)
    
    # Show configuration
    show_configuration
    
    # Check infrastructure prerequisites
    check_infrastructure || return 1
    
    # Run core administrative tests
    run_core_tests || {
        log_error "Core tests failed, aborting test suite"
        return 1
    }
    
    # Run user management tests
    run_user_tests
    
    # Run user authentication tests
    run_user_auth_tests || {
        log_error "User authentication tests failed, aborting test suite"
        return 1
    }
    
    # Run OAuth flow tests (if enabled)
    run_oauth_flow_tests || {
        log_error "OAuth flow tests failed, aborting test suite"
        return 1
    }
    
    # Cleanup on success
    if [[ "$CLEANUP_ON_SUCCESS" == "true" && $FAILED_TESTS -eq 0 ]]; then
        run_cleanup "comprehensive" "Post-Test Cleanup"
    fi
    
    return 0
}

# Function to handle test failures and cleanup
handle_test_failure() {
    log_error "Test suite encountered failures"
    
    # Cleanup on failure if configured
    if [[ "$CLEANUP_ON_FAILURE" == "true" ]]; then
        log_info "Performing cleanup after test failure..."
        run_cleanup "force" "Failure Cleanup"
    fi
    
    return 1
}

# Main test runner function
run_full_stack_test() {
    local test_mode="${1:-comprehensive}"
    
    log_info "=== Authly Full Stack Integration Test ==="
    log_info "Test mode: $test_mode"
    echo
    
    # Validate configuration first
    if ! validate_config; then
        log_error "Configuration validation failed"
        return 1
    fi
    
    case "$test_mode" in
        "infrastructure")
            check_infrastructure
            ;;
        "admin")
            check_infrastructure && run_test_script "admin-auth.sh" "Admin Authentication" "true"
            ;;
        "clients")
            check_infrastructure && run_core_tests
            ;;
        "userauth")
            check_infrastructure && run_test_script "user-auth.sh" "User Authentication" "true"
            ;;
        "oauth")
            check_infrastructure && run_test_script "oauth-flow.sh" "OAuth Flow Testing" "true"
            ;;
        "cleanup")
            run_cleanup "comprehensive" "Manual Cleanup"
            ;;
        "status")
            run_cleanup "status" "Status Check"
            ;;
        "comprehensive"|*)
            if run_comprehensive_tests; then
                log_success "Comprehensive test suite completed successfully"
            else
                handle_test_failure
                return 1
            fi
            ;;
    esac
    
    # Display results
    display_test_summary
    return $?
}

# Cleanup function for script exit
cleanup_on_script_exit() {
    # This runs when the script exits
    if [[ $? -ne 0 && "$CLEANUP_ON_FAILURE" == "true" ]]; then
        log_info "Script exiting with error, performing emergency cleanup..."
        run_cleanup "force" "Emergency Cleanup" >/dev/null 2>&1 || true
    fi
}

# Set up cleanup on exit
trap cleanup_on_script_exit EXIT

# Export configuration for use by other scripts
export RUN_USER_TESTS RUN_CLIENT_TESTS RUN_SCOPE_TESTS RUN_USER_AUTH_TESTS RUN_OAUTH_TESTS
export CLEANUP_ON_SUCCESS CLEANUP_ON_FAILURE

# Run test if script is executed directly
if [[ "${BASH_SOURCE[0]}" == "${0}" ]]; then
    test_mode="${1:-comprehensive}"
    run_full_stack_test "$test_mode"
    exit $?
fi