from __future__ import annotations

from typing import Any, Optional, Union

import pytest

from wexample_helpers.testing.abstract_test_helpers import AbstractTestHelpers


class TestHelperType(AbstractTestHelpers):
    def test_callable_annotations_behavior(self) -> None:
        from collections.abc import Callable

        from wexample_helpers.exception.not_allowed_variable_type_exception import (
            NotAllowedVariableTypeException,
        )
        from wexample_helpers.helpers.type import type_validate_or_fail

        def annotated_ok() -> bool:
            return True

        def annotated_bad() -> str:
            return "nope"

        def no_annotations() -> int:
            return 1

        # Accept when return type matches
        type_validate_or_fail(annotated_ok, Callable[..., bool])
        # Reject when return type mismatches
        with pytest.raises(NotAllowedVariableTypeException):
            type_validate_or_fail(annotated_bad, Callable[..., bool])
        # No annotations: accept for generic Callable and for Callable[..., Any]
        type_validate_or_fail(no_annotations, Callable)
        type_validate_or_fail(no_annotations, Callable[..., Any])

    def test_empty_generics_are_accepted(self) -> None:
        from wexample_helpers.helpers.type import type_validate_or_fail

        # Empty containers should validate for any inner type
        type_validate_or_fail([], list[int])
        type_validate_or_fail(set(), set[int])
        type_validate_or_fail({}, dict[str, int])

    def test_pep604_union_equivalents(self) -> None:
        from wexample_helpers.exception.not_allowed_variable_type_exception import (
            NotAllowedVariableTypeException,
        )
        from wexample_helpers.helpers.type import (
            type_is_compatible,
            type_validate_or_fail,
        )

        # Compatibility API currently targets typing.Union, ensure baseline
        assert type_is_compatible(str, Union[str, int])
        assert type_is_compatible(int, Union[str, int])
        assert not type_is_compatible(float, Union[str, int])

        # Validation path accepts PEP 604 `|` unions
        type_validate_or_fail("x", str | int)
        type_validate_or_fail(1, str | int)
        with pytest.raises(NotAllowedVariableTypeException):
            type_validate_or_fail(1.0, str | int)

    def test_tuple_len_mismatch(self) -> None:
        from wexample_helpers.exception.not_allowed_variable_type_exception import (
            NotAllowedVariableTypeException,
        )
        from wexample_helpers.helpers.type import type_validate_or_fail

        # Exact length required
        with pytest.raises(NotAllowedVariableTypeException):
            type_validate_or_fail((1, 2), tuple[int, int, int])

    def test_type_is_compatibility(self) -> None:
        from collections.abc import Callable

        from wexample_helpers.helpers.type import type_is_compatible

        success_cases = [
            (str, Any),
            (bool, Any),
            (str, str),
            (int, int),
            (float, float),
            (type(None), type(None)),  # NoneType
            (list, list),
            (dict, dict),
            (dict[str, str], dict[Any, Any]),
            (dict[str, int], dict[str, int]),
            (str, Union[str, dict[str, Any]]),
            (Callable[..., bool], Callable),
            (Callable[..., Any], Callable[..., Any]),
            (Callable[..., bool], Callable[..., bool]),
            (Callable[..., Callable[..., str]], Callable[..., Callable]),
        ]

        failure_cases = [
            (int, str),
            (str, int),
            (type(None), int),  # NoneType incompatible with int
            (dict, list),
            (list, dict),
            (dict[str, int], dict[str, str]),
            (int, Union[str, dict[str, Any]]),
            (Callable[..., bool], Callable[..., str]),
        ]

        for actual_type, expected_type in success_cases:
            assert type_is_compatible(
                actual_type, expected_type
            ), f"Expected {actual_type} to be compatible with {expected_type}"

        for actual_type, expected_type in failure_cases:
            assert not type_is_compatible(
                actual_type, expected_type
            ), f"Expected {actual_type} to be incompatible with {expected_type}"

    def test_type_is_generic(self) -> None:
        from wexample_helpers.helpers.type import type_is_generic

        # Types that should be detected as generic
        should_be_true = [
            list,
            dict,
            tuple,
            Union,
            list[str],
            dict[str, Any],
            Union[list[int], dict[int, int]],
        ]

        # Types that should NOT be detected as generic
        should_not_be_true = [
            int,
            str,
            float,
            bool,
            complex,
            Any,  # Any is not considered a generic type here
            None,  # NoneType is also non-generic
            object,
            type,  # Built-in 'type' itself is not a generic
        ]

        for type_ in should_be_true:
            assert type_is_generic(
                type_
            ), f"{type_} should be detected as a generic type"

        for type_ in should_not_be_true:
            assert not type_is_generic(
                type_
            ), f"{type_} should NOT be detected as a generic type"

    def test_type_to_name(self) -> None:
        from wexample_helpers.helpers.type import type_to_name

        # Builtins
        assert type_to_name(int) == "int"
        # UnionType formatting may vary; ensure it doesn't crash and contains members
        name = type_to_name(int | str)
        assert "int" in name and "str" in name

    def test_validation(self) -> bool:
        from collections.abc import Callable
        from types import NoneType

        def _test_callable() -> bool:
            return True

        class TestClassA:
            pass

        class TestClassB(TestClassA):
            pass

        self._test_type_validate_or_fail(
            success_cases=[
                # Basic types
                ("str", Any),
                (True, Any),
                ("str", str),
                (123, int),
                (123.123, float),
                (None, NoneType),
                # Basic containers
                ([], list),
                ({}, dict),
                # Typed containers
                (["item1", "item2"], list[str]),
                ([1, 2, 3], list[int]),
                ({"lorem": "ipsum"}, dict[str, str]),
                ({"lorem": 123}, dict[str, int]),
                ([TestClassA(), TestClassB()], list[TestClassA]),
                # Union and Optional
                (None, Optional[str]),
                ("hello", Union[str, int]),
                (123, Union[str, int]),
                ({"key": "value"}, Union[str, dict[str, str]]),
                (None, Union[str, NoneType]),
                ("optional", Optional[str]),
                # Callable
                (_test_callable, Callable),
                (_test_callable, Callable[..., Any]),
                (_test_callable, Callable[..., bool]),
                # Tuples and Sets
                ((1, "str"), tuple[int, str]),
                ({"apple", "banana"}, set[str]),
                ({1, 2, 3}, set[int]),
                # Nested types
                ({"nested": {"key": "value"}}, dict[str, dict[str, str]]),
                ([{"key": 1}, {"key": 2}], list[dict[str, int]]),
                (({"key": "value"}, 123), tuple[dict[str, str], int]),
                # Classes and types
                (TestClassA(), TestClassA),
                (TestClassB(), TestClassA),  # Compatibility with superclass
                (TestClassB(), TestClassB),
                # Class types
                (TestClassA, type[TestClassA]),
                (TestClassB, type[TestClassB]),
                (TestClassA, type[Any]),
                (TestClassB, type[Any]),
                # Nested Unions and Optionals
                ({"key": "value"}, Union[dict[str, str], list[str]]),
                ([1, 2, 3], Union[list[int], set[int]]),
                ({"key": "value"}, Union[dict[str, str], NoneType]),
                (None, Union[dict[str, str], NoneType]),
                (123, Union[Optional[int], NoneType]),
                (TestClassB(), Union[TestClassA, TestClassB]),
                (TestClassB(), Union[TestClassA, str]),
                (TestClassB, Union[type[TestClassA], type[TestClassB]]),
            ],
            failure_cases=[
                # Type mismatches
                (123, str),
                ("123", int),
                (None, int),
                # Container type mismatches
                ({}, list),
                ([], dict),
                ({"lorem": 123}, dict[str, str]),
                # Union mismatches
                (123, Union[str, dict[str, Any]]),
                ("not_callable", Callable[..., str]),
                # Callable with incompatible signature
                (_test_callable, Callable[..., str]),
                # Class type mismatches
                (
                    TestClassA(),
                    TestClassB,
                ),  # TestClassA instance is not compatible with TestClassB type
                (123, TestClassA),
                ("str", TestClassB),
                # Type mismatches for class types
                (
                    TestClassA(),
                    type[TestClassA],
                ),  # TestClassA() is an instance, not a type
                (TestClassB(), type[Any]),
                # Non-type values for Type
                (123, type[Any]),
                ("str", type[Any]),
                # Incorrect nested types
                ([1, "str"], list[int]),
                ({"key": 123}, dict[str, str]),
                ({"key": "value"}, list[dict[str, str]]),  # Expecting List, got Dict
                (["str"], set[str]),  # List vs. Set
                # Incorrect Optional and Union usage
                (123, Optional[str]),  # Expecting str or None
                (None, int),  # NoneType mismatch
                # Tuple and Set mismatches
                ((1, "str"), tuple[str, int]),  # Type order mismatch in Tuple
                ({"apple", 1}, set[str]),  # Mixed types in Set
                (TestClassA, Union[int, str, float]),
                (TestClassA, Union[TestClassB]),
                (TestClassA, TestClassB),
                (TestClassA, TestClassA),
            ],
        )
