import logging
import os
import sys
import time

from sandroid.core.toolbox import Toolbox

from .datagather import DataGather

try:
    from dexray_intercept import AppProfiler, setup_frida_device
except ImportError:
    logger = logging.getLogger(__name__)
    logger.warning(
        "dexray-intercept package not installed. Malware monitoring will be disabled."
    )
    AppProfiler = None
    setup_frida_device = None

logger = logging.getLogger(__name__)


# Set up dedicated dexray log file
def _setup_dexray_logging():
    """Set up dedicated file logging for dexray-intercept in the Sandroid results folder."""
    dexray_logger = logging.getLogger("dexray_intercept")

    # Check if we already have a file handler to avoid duplicates
    has_file_handler = any(
        isinstance(handler, logging.FileHandler) for handler in dexray_logger.handlers
    )

    if not has_file_handler and os.getenv("RESULTS_PATH"):
        # Create dexray_intercept folder at results root level (sibling to raw/)
        dexray_dir = os.path.join(os.getenv('RESULTS_PATH'), 'dexray_intercept')
        os.makedirs(dexray_dir, exist_ok=True)

        dexray_log_path = os.path.join(dexray_dir, 'dexray.log')
        file_handler = logging.FileHandler(dexray_log_path)
        file_handler.setLevel(logging.DEBUG)
        file_formatter = logging.Formatter(
            "%(asctime)s~%(levelname)s~%(message)s~module:%(module)s~function:%(funcName)s"
        )
        file_handler.setFormatter(file_formatter)
        dexray_logger.addHandler(file_handler)
        dexray_logger.setLevel(logging.DEBUG)

        logger.info(
            f"Dexray logs will be saved to {dexray_log_path}"
        )


def _strip_color_codes(text):
    """Strip ANSI color codes from text to get visual length.

    Args:
        text: Text possibly containing ANSI color codes

    Returns:
        Text with color codes removed
    """
    import re

    ansi_escape = re.compile(r"\x1B(?:[@-Z\\-_]|\[[0-?]*[ -/]*[@-~])")
    return ansi_escape.sub("", text)


def _box_line(content, width=76, align="center", border_color=None):
    """Create a properly aligned box line with color support.

    Args:
        content: Text content (may contain color codes)
        width: Total width of the box content area (default 76)
        align: Alignment ('center', 'left', or 'right')
        border_color: Color for the padding area (default Fore.CYAN)

    Returns:
        Properly formatted box line with borders
    """
    from colorama import Fore

    if border_color is None:
        border_color = Fore.CYAN

    # Calculate visual length (without color codes)
    visual_length = len(_strip_color_codes(content))

    # Calculate padding
    if align == "center":
        total_padding = width - visual_length
        left_pad = total_padding // 2
        right_pad = total_padding - left_pad
    elif align == "left":
        left_pad = 2
        right_pad = width - visual_length - left_pad
    else:  # right
        right_pad = 2
        left_pad = width - visual_length - right_pad

    return f"║{' ' * left_pad}{content}{border_color}{' ' * right_pad}║"


def _interactive_hook_selection(current_hooks):
    """Interactive hook selection for dexray-intercept with visual toggle interface.

    Args:
        current_hooks: Dictionary of currently enabled hooks

    Returns:
        Updated hook configuration dictionary
    """
    import click
    from colorama import Fore, Style

    # Define hook groups with their descriptions
    hook_groups = {
        "crypto": {
            "name": "Crypto Hooks",
            "description": "AES, encodings, keystore operations",
            "hooks": ["aes_hooks", "encodings_hooks", "keystore_hooks"],
        },
        "network": {
            "name": "Network Hooks",
            "description": "Web requests, sockets, HTTP/HTTPS traffic",
            "hooks": ["web_hooks", "socket_hooks"],
        },
        "filesystem": {
            "name": "Filesystem Hooks",
            "description": "File operations, database access",
            "hooks": ["file_system_hooks", "database_hooks"],
        },
        "ipc": {
            "name": "IPC Hooks",
            "description": "Binder, intents, broadcasts, shared preferences",
            "hooks": [
                "shared_prefs_hooks",
                "binder_hooks",
                "intent_hooks",
                "broadcast_hooks",
            ],
        },
        "process": {
            "name": "Process Hooks",
            "description": "Native libraries, runtime, DEX unpacking",
            "hooks": [
                "dex_unpacking_hooks",
                "java_dex_unpacking_hooks",
                "native_library_hooks",
                "process_hooks",
                "runtime_hooks",
            ],
        },
        "services": {
            "name": "Service Hooks",
            "description": "Bluetooth, camera, clipboard, location, telephony",
            "hooks": [
                "bluetooth_hooks",
                "camera_hooks",
                "clipboard_hooks",
                "location_hooks",
                "telephony_hooks",
            ],
        },
        "bypass": {
            "name": "Anti-Analysis Bypass Hooks",
            "description": "Root, frida, debugger, emulator detection",
            "hooks": ["bypass_hooks"],
        },
    }

    def display_hooks(hooks, groups):
        """Display hooks with visual status indicators."""
        print(f"\n{Fore.CYAN}╔{'═' * 76}╗")
        print(
            _box_line(
                f"{Style.BRIGHT}Dexray-Intercept Hook Configuration{Style.RESET_ALL}"
            )
        )
        print(f"╚{'═' * 76}╝{Style.RESET_ALL}\n")

        # Count enabled hooks
        enabled_count = sum(
            1
            for hook_list in groups.values()
            for hook in hook_list["hooks"]
            if hooks.get(hook, False)
        )
        total_individual_hooks = sum(len(group["hooks"]) for group in groups.values())

        print(
            f"{Fore.WHITE}Status: {Fore.GREEN}{enabled_count}{Fore.WHITE}/{total_individual_hooks} hook categories enabled{Style.RESET_ALL}\n"
        )

        # Display each group
        for idx, (group_key, group_info) in enumerate(groups.items(), 1):
            # Check if any hook in this group is enabled
            group_enabled = any(hooks.get(hook, False) for hook in group_info["hooks"])

            # Visual status indicator
            if group_enabled:
                status_icon = f"{Fore.GREEN}●{Style.RESET_ALL}"
                status_text = f"{Fore.GREEN}ON{Style.RESET_ALL} "
            else:
                status_icon = f"{Fore.RED}○{Style.RESET_ALL}"
                status_text = f"{Fore.RED}OFF{Style.RESET_ALL}"

            # Display hook group
            print(
                f"{Fore.YELLOW}[{idx}]{Style.RESET_ALL} {status_icon} {Style.BRIGHT}{group_info['name']:<30}{Style.RESET_ALL} [{status_text}]"
            )
            print(f"    {Fore.CYAN}{group_info['description']}{Style.RESET_ALL}")

            # Show individual hooks in this group
            hooks_in_group = [
                f"{Fore.GREEN if hooks.get(h, False) else Fore.RED}{h.replace('_', ' ').title()}{Style.RESET_ALL}"
                for h in group_info["hooks"]
            ]
            print(
                f"    {Fore.WHITE}Hooks: {', '.join(hooks_in_group)}{Style.RESET_ALL}"
            )
            print()

    updated_hooks = current_hooks.copy()

    # Ask if user wants to customize
    display_hooks(updated_hooks, hook_groups)

    print(f"{Fore.CYAN}╔{'═' * 76}╗")
    print(
        _box_line(f"{Style.BRIGHT}Would you like to customize hooks?{Style.RESET_ALL}")
    )
    print(f"╠{'═' * 76}╣")
    print(
        _box_line(
            f"{Fore.YELLOW}[y]{Fore.CYAN} Customize hook selection (toggle categories by number)",
            align="left",
        )
    )
    print(
        _box_line(
            f"{Fore.YELLOW}[n]{Fore.CYAN} Use current configuration (press ENTER to accept default)",
            align="left",
        )
    )
    print(f"╚{'═' * 76}╝{Style.RESET_ALL}\n")

    try:
        choice = click.getchar().lower()
        if choice not in ["y", "yes"]:
            enabled_count = sum(1 for v in updated_hooks.values() if v)
            print(
                f"\n{Fore.GREEN}✓ Using current configuration ({enabled_count} hook categories enabled){Style.RESET_ALL}\n"
            )
            return updated_hooks
    except (KeyboardInterrupt, EOFError):
        print(f"\n{Fore.YELLOW}Using default configuration{Style.RESET_ALL}\n")
        return updated_hooks

    # Interactive toggle mode
    while True:
        display_hooks(updated_hooks, hook_groups)

        print(f"{Fore.CYAN}╔{'═' * 76}╗")
        print(_box_line(f"{Style.BRIGHT}Toggle Hook Categories{Style.RESET_ALL}"))
        print(f"╠{'═' * 76}╣")
        print(
            _box_line(
                f"{Fore.YELLOW}[1-{len(hook_groups)}]{Fore.CYAN} Toggle a hook category on/off",
                align="left",
            )
        )
        print(
            _box_line(
                f"{Fore.YELLOW}[a]{Fore.CYAN}   Enable all hook categories",
                align="left",
            )
        )
        print(
            _box_line(
                f"{Fore.YELLOW}[d]{Fore.CYAN}   Disable all hook categories",
                align="left",
            )
        )
        print(
            _box_line(
                f"{Fore.YELLOW}[q]{Fore.CYAN}   Finish and save configuration",
                align="left",
            )
        )
        print(f"╚{'═' * 76}╝{Style.RESET_ALL}\n")

        print(f"{Fore.GREEN}► Enter your choice:{Style.RESET_ALL} ", end="", flush=True)

        try:
            choice = click.getchar().lower()
            print(f"{Fore.YELLOW}{choice}{Style.RESET_ALL}")  # Echo choice

            if choice == "q":
                break
            if choice == "a":
                # Enable all hooks
                for group_info in hook_groups.values():
                    for hook in group_info["hooks"]:
                        updated_hooks[hook] = True
                print(f"{Fore.GREEN}✓ All hooks enabled{Style.RESET_ALL}\n")
            elif choice == "d":
                # Disable all hooks
                for group_info in hook_groups.values():
                    for hook in group_info["hooks"]:
                        updated_hooks[hook] = False
                print(f"{Fore.YELLOW}✓ All hooks disabled{Style.RESET_ALL}\n")
            elif choice.isdigit() and 1 <= int(choice) <= len(hook_groups):
                # Toggle specific group
                idx = int(choice) - 1
                group_key = list(hook_groups.keys())[idx]
                group_info = hook_groups[group_key]

                # Check current state
                group_enabled = any(
                    updated_hooks.get(hook, False) for hook in group_info["hooks"]
                )

                # Toggle all hooks in this group
                new_state = not group_enabled
                for hook in group_info["hooks"]:
                    updated_hooks[hook] = new_state

                state_text = (
                    f"{Fore.GREEN}enabled{Style.RESET_ALL}"
                    if new_state
                    else f"{Fore.RED}disabled{Style.RESET_ALL}"
                )
                print(
                    f"{Fore.CYAN}✓ {group_info['name']} {state_text}{Style.RESET_ALL}\n"
                )
            else:
                print(f"{Fore.RED}Invalid choice. Please try again.{Style.RESET_ALL}\n")

        except (KeyboardInterrupt, EOFError):
            print(
                f"\n{Fore.YELLOW}Configuration cancelled. Using previous settings.{Style.RESET_ALL}\n"
            )
            return current_hooks

    # Show final summary
    enabled_hooks = [hook for hook, enabled in updated_hooks.items() if enabled]
    print(f"\n{Fore.GREEN}╔{'═' * 76}╗")
    print(
        _box_line(
            f"{Style.BRIGHT}Configuration Saved{Style.RESET_ALL}",
            border_color=Fore.GREEN,
        )
    )
    print(f"╚{'═' * 76}╝{Style.RESET_ALL}\n")

    if enabled_hooks:
        print(
            f"{Fore.GREEN}✓ {len(enabled_hooks)} hook categories enabled:{Style.RESET_ALL}"
        )
        for hook in enabled_hooks:
            print(f"  {Fore.GREEN}●{Style.RESET_ALL} {hook.replace('_', ' ').title()}")
    else:
        print(
            f"{Fore.YELLOW}⚠ Warning: No hooks enabled. Dexray-intercept will run but won't capture events.{Style.RESET_ALL}"
        )

    print()
    return updated_hooks


def _interactive_approfiler_configuration(current_settings=None):
    """Interactive AppProfiler configuration with visual interface.

    Args:
        current_settings: Dict with default settings (optional)

    Returns:
        Dict with configured settings, or None if cancelled
    """
    from colorama import Fore, Style
    import click

    # Default settings
    settings = {
        'enable_stacktrace': current_settings.get('enable_stacktrace', False) if current_settings else False,
        'deactivate_unlink': current_settings.get('deactivate_unlink', False) if current_settings else False,
        'enable_fritap': current_settings.get('enable_fritap', False) if current_settings else False,
        'fritap_output_dir': current_settings.get('fritap_output_dir', 'fritap_output') if current_settings else 'fritap_output',
        'custom_scripts': current_settings.get('custom_scripts', []) if current_settings else []
    }

    # Setting descriptions
    descriptions = {
        'enable_stacktrace': 'Show full call stacks for hook invocations (helps identify caller)',
        'deactivate_unlink': 'Prevent file unlink operations (keeps files from being deleted)',
        'enable_fritap': 'Enable TLS key extraction and traffic capture',
        'fritap_output_dir': 'Directory for fritap output files (relative to dexray_intercept/)',
        'custom_scripts': 'Additional Frida scripts to load alongside dexray-intercept'
    }

    def draw_box():
        """Draw the configuration interface."""
        width = 76

        print(f"\n{Fore.CYAN}╔{'═' * width}╗")
        print(_box_line(
            f"{Style.BRIGHT}AppProfiler Configuration{Style.RESET_ALL}",
            width,
            border_color=Fore.CYAN
        ))
        print(
            _box_line(
                f"{Fore.YELLOW}Press Enter to use defaults, or configure options below{Style.RESET_ALL}",
                width,
                border_color=Fore.CYAN
            )
        )
        print(f"╠{'═' * width}╣")

        # Stack traces
        status = f"{Fore.GREEN}✓{Style.RESET_ALL}" if settings['enable_stacktrace'] else " "
        print(_box_line(
            f"[{status}] Stack Traces:      {Fore.YELLOW}{'Enabled' if settings['enable_stacktrace'] else 'Disabled'}{Style.RESET_ALL}",
            width,
            align="left",
            border_color=Fore.CYAN
        ))
        print(_box_line(
            f"    {Fore.LIGHTBLACK_EX}{descriptions['enable_stacktrace']}{Style.RESET_ALL}",
            width,
            align="left",
            border_color=Fore.CYAN
        ))
        print(_box_line("", width, border_color=Fore.CYAN))

        # File unlinking
        status = f"{Fore.GREEN}✓{Style.RESET_ALL}" if settings['deactivate_unlink'] else " "
        print(_box_line(
            f"[{status}] Deactivate Unlink: {Fore.YELLOW}{'Yes (keep files)' if settings['deactivate_unlink'] else 'No (normal behavior)'}{Style.RESET_ALL}",
            width,
            align="left",
            border_color=Fore.CYAN
        ))
        print(_box_line(
            f"    {Fore.LIGHTBLACK_EX}{descriptions['deactivate_unlink']}{Style.RESET_ALL}",
            width,
            align="left",
            border_color=Fore.CYAN
        ))
        print(_box_line("", width, border_color=Fore.CYAN))

        # FriTap
        status = f"{Fore.GREEN}✓{Style.RESET_ALL}" if settings['enable_fritap'] else " "
        print(_box_line(
            f"[{status}] FriTap:            {Fore.YELLOW}{'Enabled' if settings['enable_fritap'] else 'Disabled'}{Style.RESET_ALL}",
            width,
            align="left",
            border_color=Fore.CYAN
        ))
        print(_box_line(
            f"    {Fore.LIGHTBLACK_EX}{descriptions['enable_fritap']}{Style.RESET_ALL}",
            width,
            align="left",
            border_color=Fore.CYAN
        ))
        print(_box_line(
            f"    Output Dir: {Fore.WHITE}{settings['fritap_output_dir']}{Style.RESET_ALL}",
            width,
            align="left",
            border_color=Fore.CYAN
        ))
        print(_box_line("", width, border_color=Fore.CYAN))

        # Custom scripts
        script_count = len(settings['custom_scripts'])
        script_display = f"{script_count} script(s)" if script_count > 0 else "None"
        print(_box_line(
            f"Custom Scripts:    {Fore.YELLOW}{script_display}{Style.RESET_ALL}",
            width,
            align="left",
            border_color=Fore.CYAN
        ))
        if script_count > 0:
            for i, script in enumerate(settings['custom_scripts'][:3], 1):  # Show first 3
                script_name = script.split('/')[-1]  # Show just filename
                print(_box_line(
                    f"    {i}. {Fore.WHITE}{script_name}{Style.RESET_ALL}",
                    width,
                    align="left",
                    border_color=Fore.CYAN
                ))
            if script_count > 3:
                print(_box_line(
                    f"    ... and {script_count - 3} more",
                    width,
                    align="left",
                    border_color=Fore.CYAN
                ))

        # Controls
        print(f"╠{'═' * width}╣")
        print(_box_line(
            f"{Fore.WHITE}[S]{Style.RESET_ALL} Toggle Stack Traces  "
            f"{Fore.WHITE}[U]{Style.RESET_ALL} Toggle Unlink  "
            f"{Fore.WHITE}[F]{Style.RESET_ALL} Toggle FriTap",
            width,
            align="left",
            border_color=Fore.CYAN
        ))
        print(_box_line(
            f"{Fore.WHITE}[D]{Style.RESET_ALL} Set FriTap Dir  "
            f"{Fore.WHITE}[C]{Style.RESET_ALL} Add Custom Script  "
            f"{Fore.WHITE}[R]{Style.RESET_ALL} Remove Script",
            width,
            align="left",
            border_color=Fore.CYAN
        ))
        print(_box_line(
            f"{Fore.GREEN}[Enter]{Style.RESET_ALL} Continue  "
            f"{Fore.RED}[Esc]{Style.RESET_ALL} Cancel",
            width,
            align="left",
            border_color=Fore.CYAN
        ))
        print(f"╚{'═' * width}╝{Style.RESET_ALL}")

    # Main interaction loop
    while True:
        click.clear()
        draw_box()

        print(f"\n{Fore.CYAN}Your choice:{Style.RESET_ALL} ", end="", flush=True)
        choice = click.getchar().lower()

        if choice == '\r' or choice == '\n':  # Enter
            break
        elif choice == '\x1b':  # ESC
            print(
                f"\n{Fore.YELLOW}Configuration cancelled. Using previous settings.{Style.RESET_ALL}\n"
            )
            return current_settings
        elif choice == 's':  # Toggle stack traces
            settings['enable_stacktrace'] = not settings['enable_stacktrace']
        elif choice == 'u':  # Toggle unlink
            settings['deactivate_unlink'] = not settings['deactivate_unlink']
        elif choice == 'f':  # Toggle fritap
            settings['enable_fritap'] = not settings['enable_fritap']
        elif choice == 'd':  # Set fritap dir
            print(f"\n{Fore.CYAN}Enter FriTap output directory (relative to dexray_intercept/):{Style.RESET_ALL}")
            print(f"{Fore.LIGHTBLACK_EX}Current: {settings['fritap_output_dir']}{Style.RESET_ALL}")
            print(f"{Fore.LIGHTBLACK_EX}Press Enter to keep current, or type new path:{Style.RESET_ALL}")
            new_dir = input(f"{Fore.YELLOW}» {Style.RESET_ALL}").strip()
            if new_dir:
                settings['fritap_output_dir'] = new_dir
        elif choice == 'c':  # Add custom script
            print(f"\n{Fore.CYAN}Enter path to custom Frida script:{Style.RESET_ALL}")
            script_path = input(f"{Fore.YELLOW}» {Style.RESET_ALL}").strip()
            if script_path:
                import os
                if os.path.exists(script_path):
                    settings['custom_scripts'].append(script_path)
                    print(f"{Fore.GREEN}✓ Script added: {script_path}{Style.RESET_ALL}")
                else:
                    print(f"{Fore.RED}✗ File not found: {script_path}{Style.RESET_ALL}")
                input(f"\n{Fore.CYAN}Press Enter to continue...{Style.RESET_ALL}")
        elif choice == 'r':  # Remove custom script
            if settings['custom_scripts']:
                print(f"\n{Fore.CYAN}Custom Scripts:{Style.RESET_ALL}")
                for i, script in enumerate(settings['custom_scripts'], 1):
                    print(f"{Fore.YELLOW}{i}.{Style.RESET_ALL} {script}")
                print(f"\n{Fore.CYAN}Enter number to remove (or press Enter to cancel):{Style.RESET_ALL}")
                choice_str = input(f"{Fore.YELLOW}» {Style.RESET_ALL}").strip()
                if choice_str.isdigit():
                    idx = int(choice_str) - 1
                    if 0 <= idx < len(settings['custom_scripts']):
                        removed = settings['custom_scripts'].pop(idx)
                        print(f"{Fore.GREEN}✓ Removed: {removed}{Style.RESET_ALL}")
                        input(f"\n{Fore.CYAN}Press Enter to continue...{Style.RESET_ALL}")

    # Show final summary
    click.clear()
    print(f"\n{Fore.GREEN}╔{'═' * 76}╗")
    print(
        _box_line(
            f"{Style.BRIGHT}Configuration Saved{Style.RESET_ALL}",
            border_color=Fore.GREEN,
        )
    )
    print(f"╚{'═' * 76}╝{Style.RESET_ALL}\n")

    # Show enabled features
    enabled_features = []
    if settings['enable_stacktrace']:
        enabled_features.append("Stack Traces")
    if settings['deactivate_unlink']:
        enabled_features.append("Deactivate Unlink")
    if settings['enable_fritap']:
        enabled_features.append(f"FriTap (output: {settings['fritap_output_dir']})")
    if settings['custom_scripts']:
        enabled_features.append(f"{len(settings['custom_scripts'])} Custom Script(s)")

    if enabled_features:
        print(f"{Fore.GREEN}✓ Enabled features:{Style.RESET_ALL}")
        for feature in enabled_features:
            print(f"  {Fore.GREEN}●{Style.RESET_ALL} {feature}")
    else:
        print(f"{Fore.CYAN}Using default configuration (no optional features enabled){Style.RESET_ALL}")

    print()
    return settings


class MalwareMonitor(DataGather):
    """This class monitors provides a wrapper for the Dexray Intercept package to implement it into Sandroid.

    .. note::
        Inherits from :class:`DataGather`.

    **Attributes:**
        - **logger** (*Logger*): Logger instance for the class.
        - **last_results** (*dict*): Stores the last results of the monitoring.
        - **job_manager** (*JobManager*): Manages Frida jobs.
        - **process_session** (*Session*): Current Frida process session.
        - **app_package** (*str*): Package name of the application being monitored.
        - **profiler** (*AppProfiler*): Instance of the AppProfiler class.
        - **frida_script_path** (*str*): Path to the Frida script.
        - **running** (*bool*): Indicates if the monitoring is currently running.
        - **last_output** (*str*): Stores the last output of the profiling.
        - **_new_results_available** (*bool*): Indicates if there are new results available.
    """

    def __init__(self, path_filters=None, debug_mode=False,
                 enable_full_stacktrace=False, enable_fritap=False,
                 fritap_output_dir="fritap_output", custom_scripts=None,
                 deactivate_unlink=False):
        """Initializes the MalwareMonitor with dexray-intercept API.

        :param path_filters: List of file paths to filter for monitoring.
        :type path_filters: list or None
        :param debug_mode: Enable verbose/debug output (shows devlog messages from dexray-intercept)
        :type debug_mode: bool
        :param enable_full_stacktrace: Enable full stack traces for hook invocations
        :type enable_full_stacktrace: bool
        :param enable_fritap: Enable fritap for TLS key extraction
        :type enable_fritap: bool
        :param fritap_output_dir: Directory for fritap output files (relative to dexray_intercept/)
        :type fritap_output_dir: str
        :param custom_scripts: List of paths to custom Frida scripts to load
        :type custom_scripts: list or None
        :param deactivate_unlink: Prevent file unlink operations
        :type deactivate_unlink: bool
        """
        super().__init__()

        if AppProfiler is None:
            logger.error(
                "dexray-intercept not available. MalwareMonitor will not function."
            )
            self.available = False
            return

        self.available = True
        if path_filters == []:
            self.path_filters = None
        else:
            self.path_filters = path_filters

        self.debug_mode = debug_mode  # Store debug mode for passing to dexray-intercept

        # Store AppProfiler settings (can be overridden by interactive configuration)
        self.profiler_settings = {
            'enable_stacktrace': enable_full_stacktrace,
            'deactivate_unlink': deactivate_unlink,
            'enable_fritap': enable_fritap,
            'fritap_output_dir': fritap_output_dir,
            'custom_scripts': custom_scripts or []
        }

        # Keep individual attributes for backward compatibility and easy access
        self.enable_full_stacktrace = enable_full_stacktrace
        self.enable_fritap = enable_fritap
        self.fritap_output_dir = fritap_output_dir
        self.deactivate_unlink = deactivate_unlink
        self.custom_script = custom_scripts or []

        self.last_results = {}
        self.job_manager = Toolbox.get_frida_job_manager()
        self.process_session = None  # Will be set up with new API
        self.app_package = None
        self.frida_device = None
        self.profiler = None

        self.running = False
        self.last_output = None
        self._new_results_available = False

        # Snapshot management (for spawn mode)
        self.snapshot_name = None  # Will be set if user creates snapshot before analysis
        self.should_revert_snapshot = False  # Flag to track if we should revert on stop

        # Hook configuration for dexray-intercept
        # Enable critical hooks by default for malware analysis
        # Matches: dexray-intercept -s --enable-dex-unpacking --enable-aes --hooks-filesystem --hooks-network
        self.hook_config = {
            # Crypto hooks
            "aes_hooks": True,
            # Network hooks (--hooks-network)
            "web_hooks": True,
            "socket_hooks": True,
            # Filesystem hooks (--hooks-filesystem)
            "file_system_hooks": True,
            "database_hooks": False,
            # DEX unpacking hooks (--enable-dex-unpacking)
            "dex_unpacking_hooks": True,
            "java_dex_unpacking_hooks": True,
        }

        # Set up dedicated dexray logging
        _setup_dexray_logging()

        logger.info(
            "MalwareMonitor initialized with dexray-intercept API and path filters: "
            + str(self.path_filters)
        )

    def gather(self):
        """Gather data from the monitored application using dexray-intercept.

        .. warning::
            Context dependent behavior: Calling this method acts as a toggle, it starts or stops the monitoring process based on the current state.
        """
        if not self.available:
            logger.error(
                "MalwareMonitor not available due to missing dexray-intercept package"
            )
            return

        if self.running:
            # Stop monitoring and collect results
            try:
                if self.profiler:
                    self.profiler.stop_profiling()
                    self.last_output = self.profiler.get_profile_data()

                    # Export JSON profile to dexray_intercept folder
                    # Use RESULTS_PATH to put dexray_intercept at root level (sibling to raw/)
                    results_path = os.getenv("RESULTS_PATH", "./results/")
                    dexray_dir = os.path.join(results_path, "dexray_intercept")

                    # Ensure dexray_intercept directory exists
                    os.makedirs(dexray_dir, exist_ok=True)

                    # Use timestamp format matching standalone dexray-intercept
                    timestamp = time.strftime("%Y-%m-%d_%H-%M-%S")
                    json_filename = f"profile_{self.app_package}_{timestamp}.json"
                    json_filepath = os.path.join(dexray_dir, json_filename)

                    try:
                        # Write profile data as JSON
                        written_path = (
                            self.profiler.profile_collector.write_profile_to_file(
                                json_filepath
                            )
                        )
                        success_msg = f"Dexray profile exported to: {written_path}"
                        logger.info(success_msg)
                        print(f"\n[+] {success_msg}")
                    except Exception as export_error:
                        error_msg = f"Failed to export JSON profile to {json_filepath}: {export_error}"
                        logger.error(error_msg)
                        print(f"\n[-] {error_msg}")

                    logger.info("MalwareMonitor stopped and data collected")

                # Snapshot revert prompt (if snapshot was created)
                if self.snapshot_name and self.should_revert_snapshot:
                    if sys.stdin.isatty() and sys.stdout.isatty():
                        from colorama import Fore, Style
                        import click

                        print(f"\n{Fore.CYAN}╔{'═' * 76}╗")
                        print(_box_line(
                            f"{Style.BRIGHT}Snapshot Revert{Style.RESET_ALL}",
                            border_color=Fore.CYAN
                        ))
                        print(_box_line(
                            f"{Fore.YELLOW}Revert to snapshot: {self.snapshot_name.decode()}?{Style.RESET_ALL}",
                            border_color=Fore.CYAN
                        ))
                        print(f"╠{'═' * 76}╣")
                        print(_box_line(
                            f"{Fore.WHITE}This will restore the AVD to the state before analysis.{Style.RESET_ALL}",
                            align="left",
                            border_color=Fore.CYAN
                        ))
                        print(_box_line(
                            f"{Fore.WHITE}All changes made during analysis will be lost.{Style.RESET_ALL}",
                            align="left",
                            border_color=Fore.CYAN
                        ))
                        print(f"╠{'═' * 76}╣")
                        print(_box_line(
                            f"{Fore.GREEN}[Y]{Style.RESET_ALL} Yes (recommended)  "
                            f"{Fore.RED}[N]{Style.RESET_ALL} No (keep changes)",
                            align="left",
                            border_color=Fore.CYAN
                        ))
                        print(f"╚{'═' * 76}╝{Style.RESET_ALL}")

                        print(f"\n{Fore.CYAN}Revert to snapshot? [{Fore.GREEN}Y{Fore.CYAN}/{Fore.RED}n{Fore.CYAN}]:{Style.RESET_ALL} ", end="", flush=True)
                        choice = click.getchar().lower()
                        print(f"{Fore.YELLOW}{choice}{Style.RESET_ALL}")

                        if choice != 'n':  # Default to yes
                            print(f"\n{Fore.CYAN}Reverting to snapshot: {Fore.YELLOW}{self.snapshot_name.decode()}{Style.RESET_ALL}")
                            Toolbox.load_snapshot(self.snapshot_name)
                            print(f"{Fore.GREEN}✓ Snapshot restored successfully{Style.RESET_ALL}\n")
                            logger.info(f"Reverted to snapshot: {self.snapshot_name.decode()}")
                        else:
                            print(f"\n{Fore.YELLOW}Keeping current AVD state{Style.RESET_ALL}\n")
                            logger.info("User declined snapshot revert")

                self.running = False
                Toolbox.malware_monitor_running = False
                self._new_results_available = True
            except Exception as e:
                logger.error(f"Error stopping MalwareMonitor: {e}")

        elif not self.running:
            # Start monitoring
            try:
                # Interactive hook selection FIRST (before spawning/attaching)
                if sys.stdin.isatty() and sys.stdout.isatty():
                    try:
                        self.hook_config = _interactive_hook_selection(self.hook_config)
                    except (KeyboardInterrupt, EOFError):
                        print(
                            "\n\nHook selection cancelled. Using default configuration."
                        )
                        logger.info(
                            "Interactive hook selection cancelled by user, using defaults"
                        )

                    # AppProfiler configuration SECOND (after hooks, before session)
                    try:
                        configured_settings = _interactive_approfiler_configuration(self.profiler_settings)
                        if configured_settings:
                            # Update settings with user configuration
                            self.profiler_settings = configured_settings
                            # Sync individual attributes
                            self.enable_full_stacktrace = configured_settings['enable_stacktrace']
                            self.deactivate_unlink = configured_settings['deactivate_unlink']
                            self.enable_fritap = configured_settings['enable_fritap']
                            self.fritap_output_dir = configured_settings['fritap_output_dir']
                            self.custom_script = configured_settings['custom_scripts']
                    except (KeyboardInterrupt, EOFError):
                        print(
                            "\n\nAppProfiler configuration cancelled. Using default settings."
                        )
                        logger.info(
                            "Interactive AppProfiler configuration cancelled by user, using defaults"
                        )
                else:
                    logger.info(
                        "Non-interactive session detected, using default hook configuration and AppProfiler settings"
                    )

                # Snapshot creation prompt (SPAWN mode only, BEFORE session creation)
                # This ensures a clean state to revert to after analysis
                if Toolbox.is_spawn_mode() and sys.stdin.isatty() and sys.stdout.isatty():
                    from colorama import Fore, Style
                    import click
                    from datetime import datetime

                    print(f"\n{Fore.CYAN}╔{'═' * 76}╗")
                    print(_box_line(
                        f"{Style.BRIGHT}Snapshot Management{Style.RESET_ALL}",
                        border_color=Fore.CYAN
                    ))
                    print(_box_line(
                        f"{Fore.YELLOW}Create snapshot before analysis? (Allows revert after){Style.RESET_ALL}",
                        border_color=Fore.CYAN
                    ))
                    print(f"╠{'═' * 76}╣")
                    print(_box_line(
                        f"{Fore.WHITE}This will save the current AVD state before spawning the app.{Style.RESET_ALL}",
                        align="left",
                        border_color=Fore.CYAN
                    ))
                    print(_box_line(
                        f"{Fore.WHITE}You can revert to this state when stopping the analysis.{Style.RESET_ALL}",
                        align="left",
                        border_color=Fore.CYAN
                    ))
                    print(f"╠{'═' * 76}╣")
                    print(_box_line(
                        f"{Fore.GREEN}[Y]{Style.RESET_ALL} Yes (recommended)  "
                        f"{Fore.RED}[N]{Style.RESET_ALL} No",
                        align="left",
                        border_color=Fore.CYAN
                    ))
                    print(f"╚{'═' * 76}╝{Style.RESET_ALL}")

                    print(f"\n{Fore.CYAN}Create snapshot? [{Fore.GREEN}Y{Fore.CYAN}/{Fore.RED}n{Fore.CYAN}]:{Style.RESET_ALL} ", end="", flush=True)
                    choice = click.getchar().lower()
                    print(f"{Fore.YELLOW}{choice}{Style.RESET_ALL}")

                    if choice != 'n':  # Default to yes (Enter, y, or any other key)
                        # Generate snapshot name with timestamp
                        timestamp = datetime.now().strftime("%Y%m%d_%H%M%S")
                        self.snapshot_name = f"sandroid_malware_{timestamp}".encode()

                        print(f"\n{Fore.CYAN}Creating snapshot: {Fore.YELLOW}{self.snapshot_name.decode()}{Style.RESET_ALL}")
                        Toolbox.create_snapshot(self.snapshot_name)
                        self.should_revert_snapshot = True
                        print(f"{Fore.GREEN}✓ Snapshot created successfully{Style.RESET_ALL}\n")
                        logger.info(f"Created snapshot: {self.snapshot_name.decode()}")
                    else:
                        print(f"\n{Fore.YELLOW}Skipping snapshot creation{Style.RESET_ALL}\n")
                        logger.info("User declined snapshot creation")

                # NOW get the unified session (supports both spawn and attach modes)
                session, mode, app_info = Toolbox.get_frida_session_for_spotlight()

                self.dexray_mode = mode
                self.process_session = session
                self.app_package = app_info["package_name"]
                self.frida_device = app_info["device"]

                # Get result folder path for dexray dumps
                # Use RESULTS_PATH to put dexray_intercept at root level (sibling to raw/)
                results_path = os.getenv("RESULTS_PATH", "./results/")
                dexray_base_path = os.path.join(results_path, "dexray_intercept")
                os.makedirs(dexray_base_path, exist_ok=True)

                # Make fritap_output_dir relative to dexray_intercept folder
                fritap_full_path = os.path.join(dexray_base_path, self.fritap_output_dir)

                # Create profiler with configured settings from interactive configuration
                # DUAL mode provides terminal output like standalone dexray-intercept while collecting JSON
                self.profiler = AppProfiler(
                    self.process_session,
                    self.debug_mode,
                    output_format="DUAL",
                    base_path=dexray_base_path,
                    deactivate_unlink=self.deactivate_unlink,  # From configuration
                    hook_config=self.hook_config,
                    enable_stacktrace=self.enable_full_stacktrace,  # From configuration
                    enable_fritap=self.enable_fritap,  # From configuration
                    fritap_output_dir=fritap_full_path,  # Full path relative to dexray_intercept
                    target_name=self.app_package,
                    spawn_mode=self.dexray_mode,
                    custom_scripts=self.custom_script  # From configuration
                )

                # Start profiling
                script = self.profiler.start_profiling()
                self.running = True
                Toolbox.malware_monitor_running = True

                # Resume spawned process now that hooks are installed
                if mode == "spawn":
                    Toolbox.resume_spawned_process_after_hooks(
                        self.frida_device,
                        app_info['pid']
                    )

                logger.info(
                    f"MalwareMonitor started in {mode.upper()} mode for package: {self.app_package} "
                    f"(PID: {app_info['pid']})"
                )

            except Exception as e:
                logger.error(f"Error starting MalwareMonitor: {e}")
                self.running = False
                Toolbox.malware_monitor_running = False

    def has_new_results(self):
        """Check if there are new results available.

        :returns: True if there are new results, False otherwise.
        :rtype: bool
        """
        if self.running:
            return False
        return self._new_results_available  # Updated variable name

    def return_data(self):
        """Return the last profiling data.

        This method returns the last profiling data and resets the new results flag.

        :returns: The last profiling data in JSON format.
        :rtype: str
        """
        self._new_results_available = False  # Updated variable name
        return self.last_output

    def pretty_print(self):
        """Not implemented"""
