# -*- coding: utf-8 -*-
# cumulative_return.py
import numpy as np
import pandas as pd


def cumulative_return(
    df: pd.DataFrame, column: str = "close", fillna: bool = False
) -> pd.Series:
    """
    Calculate the Cumulative Return (CR) of a specified column in a DataFrame.

    Parameters:
    - df (pandas.DataFrame): Input DataFrame containing the specified column.
    - column (str): The column on which the cumulative return is to be calculated. Default is "close".
    - fillna (bool): If True, fill NaN values. Default is False.

    Call with:
        df['cumulative_return'] = bta.cumulative_return(df)

    Returns:
    - pd.Series: Series of cumulative return values.
    """
    # Create a copy of the DataFrame to avoid modifying the original
    df_copy = df.copy()

    # Calculate cumulative return
    df_copy["cum_ret"] = (df_copy[column] / df_copy[column].iloc[0]) - 1
    df_copy["cum_ret"] *= 100  # Expressing cumulative return as a percentage

    # Handle NaN values if requested
    if fillna:
        df_copy["cum_ret"] = df_copy["cum_ret"].fillna(-1)

    return df_copy["cum_ret"].rename("cumulative_return")


cumulative_return.__doc__ = """
Name:
    Cumulative Return

Description:
    Cumulative Return measures the aggregate percentage return of an asset over a specific period.
    It calculates how much an initial investment would have grown or declined over time,
    expressed as a percentage. This is a fundamental metric in performance analysis to evaluate
    the total return generated by an asset or strategy.

More info:
    The cumulative return is calculated as: (Current Price / Initial Price - 1) * 100
    This provides the percentage change from the initial price to the current price.

Parameters:
    - df (pandas.DataFrame): Input DataFrame containing the specified column.
    - column (str): The column on which the cumulative return is to be calculated. Default is "close".
    - fillna (bool): If True, fill NaN values with -1. Default is False.

Call with:
    df['cum_return'] = bta.cumulative_return(df)
    
    # To use a different price column:
    df['cum_return'] = bta.cumulative_return(df, column='adjusted_close')

Returns:
    pd.Series: Series of cumulative return values expressed as percentages.
"""


def test():
    """
    Test function for the cumulative_return indicator.
    
    This function uses the generic test_indicator function from bamboo_ta.py
    to test the cumulative_return indicator.
    
    Returns:
        None: Displays the results to the console
    """
    try:
        # Import the test_indicator function from bamboo_ta
        from bamboo_ta.bamboo_ta import test_indicator
        
        # Test the indicator
        test_indicator(cumulative_return)
        
    except ImportError:
        print("Error: Could not import test_indicator from bamboo_ta.bamboo_ta")
    except Exception as e:
        print(f"Error during testing: {e}")

# Execute the test if this file is run directly
if __name__ == "__main__":
    test()
