from datetime import datetime
from dataclasses import dataclass
from typing import Optional, List, Dict, Any

"""
Forms
{'id': 'nrZJxv', 'name': 'Lead generation form', 'isNameModifiedByUser': False, 'workspaceId': 'wMbMJX', 
    'organizationId': 'mV87xl', 'status': 'PUBLISHED', 'hasDraftBlocks': False, 'numberOfSubmissions': 1, 
    'createdAt': '2025-08-07T09:26:56.000Z', 'updatedAt': '2025-08-07T09:31:53.000Z', 'index': 0, 'isClosed': False}
"""


class FormStatus:
    DRAFT = "DRAFT"
    PUBLISHED = "PUBLISHED"
    DELETED = "DELETED"


@dataclass
class Form:
    id: str
    name: str
    isNameModifiedByUser: bool
    workspaceId: str
    organizationId: str
    status: FormStatus
    hasDraftBlocks: bool
    numberOfSubmissions: int
    createdAt: datetime
    updatedAt: datetime
    index: int
    isClosed: bool

    @classmethod
    def from_api_response(cls, data: Dict[str, Any]) -> 'Form':
        """Crea un'istanza Form dai dati dell'API"""
        return cls(
            id=data['id'],
            name=data['name'],
            isNameModifiedByUser=data['isNameModifiedByUser'],
            workspaceId=data['workspaceId'],
            organizationId=data['organizationId'],
            status=data['status'],
            hasDraftBlocks=data['hasDraftBlocks'],
            numberOfSubmissions=data['numberOfSubmissions'],
            createdAt=datetime.fromisoformat(
                data['createdAt'].replace('Z', '+00:00')),
            updatedAt=datetime.fromisoformat(
                data['updatedAt'].replace('Z', '+00:00')),
            index=data['index'],
            isClosed=data['isClosed'],
        )

    def to_dict(self) -> Dict[str, Any]:
        """Converte l'oggetto in dizionario"""
        return {
            'id': self.id,
            'name': self.name,
            'isNameModifiedByUser': self.isNameModifiedByUser,
            'workspaceId': self.workspaceId,
            'organizationId': self.organizationId,
            'status': self.status,
            'hasDraftBlocks': self.hasDraftBlocks,
            'numberOfSubmissions': self.numberOfSubmissions,
            'createdAt': self.createdAt,
            'updatedAt': self.updatedAt,
            'index': self.index,
            'isClosed': self.isClosed
        }


class FieldType:
    INPUT_FIELD = "INPUT_FIELD"
    CHECKBOXES = "CHECKBOXES"


class QuestionType:
    INPUT_FIELD = "INPUT_FIELD"
    CHECKBOXES = "CHECKBOXES"
    DROPDOWN = "DROPDOWN"


@dataclass
class Field:
    uuid: str
    type: FieldType
    questionType: str
    blockGroupUuid: str
    title: str

    @classmethod
    def from_api_response(cls, data: Dict[str, Any]) -> 'Field':
        """Crea un'istanza Field dai dati dell'API"""
        return cls(
            uuid=data['uuid'],
            type=data['type'],
            questionType=data['questionType'],
            blockGroupUuid=data['blockGroupUuid'],
            title=data['title'],
        )

    def to_dict(self) -> Dict[str, Any]:
        """Converte l'oggetto in dizionario"""
        return {
            'uuid': self.uuid,
            'type': self.type,
            'questionType': self.questionType,
            'blockGroupUuid': self.blockGroupUuid,
            'title': self.title
        }


@dataclass
class Question:
    id: str
    type: QuestionType
    title: str
    isTitleModifiedByUser: bool
    formId: str
    isDeleted: bool
    numberOfResponses: int
    createdAt: datetime
    fields: Field

    @classmethod
    def from_api_response(cls, data: Dict[str, Any]) -> 'Question':
        """Crea un'istanza Question dai dati dell'API"""
        return cls(
            id=data['id'],
            type=data['type'],
            title=data['title'],
            isTitleModifiedByUser=data['isTitleModifiedByUser'],
            formId=data['formId'],
            isDeleted=data['isDeleted'],
            numberOfResponses=data['numberOfResponses'],
            createdAt=datetime.fromisoformat(
                data['createdAt'].replace('Z', '+00:00')),
            fields=data['fields'],
        )

    def to_dict(self) -> Dict[str, Any]:
        """Converte l'oggetto in dizionario"""
        return {
            'id': self.id,
            'type': self.type,
            'title': self.title,
            'isTitleModifiedByUser': self.isTitleModifiedByUser,
            'formId': self.formId,
            'isDeleted': self.isDeleted,
            'numberOfResponses': self.numberOfResponses,
            'createdAt': self.createdAt,
            'fields': self.fields
        }


@dataclass
class FormResponse:
    id: str
    formId: str
    questionId: str
    respondentId: str
    submissionId: str
    sessionUuid: str
    answer: str | Any
    createdAt: datetime
    updatedAt: datetime

    @classmethod
    def from_api_response(cls, data: Dict[str, Any]) -> 'FormResponse':
        """Crea un'istanza FormResponse dai dati dell'API"""
        return cls(
            id=data['id'],
            formId=data['formId'],
            questionId=data['questionId'],
            respondentId=data['respondentId'],
            submissionId=data['submissionId'],
            sessionUuid=data['sessionUuid'],
            answer=data['answer'],
            createdAt=datetime.fromisoformat(
                data['createdAt'].replace('Z', '+00:00')),
            updatedAt=datetime.fromisoformat(
                data['updatedAt'].replace('Z', '+00:00')),
        )

    def to_dict(self) -> Dict[str, Any]:
        """Converte l'oggetto in dizionario"""
        return {
            'id': self.id,
            'formId': self.formId,
            'questionId': self.questionId,
            'respondentId': self.respondentId,
            'submissionId': self.submissionId,
            'sessionUuid': self.sessionUuid,
            'answer': self.answer,
            'createdAt': self.createdAt,
            'updatedAt': self.updatedAt
        }


@dataclass
class Submission:
    id: str
    formId: str
    respondentId: str
    isCompleted: bool
    submittedAt: datetime
    createdAt: datetime
    updatedAt: datetime
    responses: List[FormResponse]

    @classmethod
    def from_api_response(cls, data: Dict[str, Any]) -> 'Submission':
        """Crea un'istanza FormResponse dai dati dell'API"""
        return cls(
            id=data['id'],
            formId=data['formId'],
            respondentId=data['respondentId'],
            isCompleted=data['isCompleted'],
            submittedAt=data['submittedAt'],
            createdAt=datetime.fromisoformat(
                data['createdAt'].replace('Z', '+00:00')),
            updatedAt=datetime.fromisoformat(
                data['updatedAt'].replace('Z', '+00:00')),
            responses=[FormResponse.from_api_response(
                item) for item in data['responses']],
        )

    def to_dict(self) -> Dict[str, Any]:
        """Converte l'oggetto in dizionario"""
        return {
            'id': self.id,
            'formId': self.formId,
            'respondentId': self.respondentId,
            'isCompleted': self.isCompleted,
            'submittedAt': self.submittedAt,
            'responses': self.responses,
            'createdAt': self.createdAt,
            'updatedAt': self.updatedAt
        }


@dataclass
class FormSubmission:
    questions: List[Question]
    submission: List[Submission] | Submission

    @classmethod
    def from_api_response(cls, data: Dict[str, Any]) -> 'FormSubmission':
        """Crea un'istanza FormResponse dai dati dell'API"""
        return cls(
            questions=[Question.from_api_response(
                item) for item in data['questions']],
            submission=Submission.from_api_response(data['submission'])
        )

    def to_dict(self) -> Dict[str, Any]:
        """Converte l'oggetto in dizionario"""
        return {
            'questions': self.questions,
            'submission': self.submission,
        }


@dataclass
class PaginatedResponse:
    """Classe generica per risposte paginate"""
    items: List[Any]
    page: int
    limit: int
    total: int
    hasMore: bool

    @classmethod
    def from_api_response(cls, data: Dict[str, Any], item_class=None) -> 'PaginatedResponse':
        items = data['items']
        if item_class:
            items = [item_class.from_api_response(item) for item in items]

        return cls(
            items=items,
            page=data['page'],
            limit=data['limit'],
            total=data['total'],
            hasMore=data['hasMore']
        )
