from tally.core.models import PaginatedResponse, Form, Question, FormSubmission


class BaseResource:
    def __init__(self, client):
        self.client = client
        self.base_path = None  # Override in subclasses

    def _build_url(self, *parts, **params):
        """Construct URLs with proper escaping and query params"""
        path_parts = [self.base_path] + list(parts)
        url = "/".join(str(p) for p in path_parts if p is not None)

        if params:
            query_string = "&".join(
                f"{k}={v}" for k, v in params.items() if v is not None)
            url += f"?{query_string}"

        return url


class UsersResource(BaseResource):
    def __init__(self, client):
        super().__init__(client)
        self.base_path = "users"

    def get(self):
        url = self._build_url('me')  # "users/me"
        return self.client._request("GET", url)


class FormResources(BaseResource):
    def __init__(self, client):
        super().__init__(client)
        self.base_path = "forms"

    def list(self, page=1) -> PaginatedResponse:
        url = self._build_url(page=page)
        # "forms?page=1&limit=20&status=active"
        return PaginatedResponse.from_api_response(self.client._request("GET", url), Form)

    def retrieve(self, form_id) -> Form:
        url = self._build_url(form_id)
        # "forms/123"
        return Form.from_api_response(self.client._request("GET", url))

    def form_questions(self, form_id) -> PaginatedResponse:
        url = self._build_url(form_id, "questions")
        # "forms/123/questions"
        return PaginatedResponse(self.client._request("GET", url), Question)

    def list_submissions(self, form_id) -> FormSubmission:
        url = self._build_url(form_id, "submissions")
        # "forms/123/submissions"
        return FormSubmission.from_api_response(self.client._request("GET", url))

    def retrieve_submission(self, form_id, sub_id) -> FormSubmission:
        url = self._build_url(form_id, "submissions", sub_id)
        # "forms/123/submissions/456"
        return FormSubmission.from_api_response(self.client._request("GET", url))
