import * as d3 from 'd3';
import { BasePlot } from './baseplot';
import { BaseModel, BaseView } from '../base/base';
import { 
    ClickSelectButton,   
    SideBar,
    DeselectAllButton 
} from "./tools/tools";
import {
    HorizonChartParams,
    HorizonDataPoint
} from './interface';

const MARGIN = { top: 30, right: 10, bottom: 10, left: 10 };

export class HorizonChart extends BasePlot {
    // Extrae la lógica de extracción de datos originales
    private extractOriginalDataFromSelected(container: d3.Selection<any, any, any, any>): any[] {
        const selectedSeries: any[] = [];
        container.selectAll('.horizon-series.selected').each(function(d: any) {
            const [, values] = d as [string, HorizonDataPoint[]];
            const originalData = values.map(point => point.originalData);
            selectedSeries.push(...originalData);
        });
        return selectedSeries;
    }
    // Extrae el manejo del click de serie
    private createSeriesClickHandler(
        clickSelectButton: ClickSelectButton<SVGGElement> | null,
        setSelectedValues: ((values: any[]) => void) | undefined,
        GG: d3.Selection<SVGGElement, unknown, null, undefined>
    ) {
        return (event: MouseEvent, [seriesKey, seriesValues]: [string, HorizonDataPoint[]]) => {
            const canSelect = Boolean(clickSelectButton?.isSelected);
            if (!canSelect) return;
            
            setTimeout(() => {
                const selectedSeries = this.extractOriginalDataFromSelected(GG);
                if (setSelectedValues) {
                    setSelectedValues(selectedSeries);
                }
            }, 0);
        };
    }
    // Extrae la función de actualización desde botones
    private createUpdateSelectedFromButtons(
        GG: d3.Selection<SVGGElement, unknown, null, undefined>,
        setSelectedValues: ((values: any[]) => void) | undefined
    ) {
        return () => {
            const selectedSeries = this.extractOriginalDataFromSelected(GG);
            setSelectedValues?.(selectedSeries);
        };
    }
    /**
     * Procesa y transforma los datos crudos en el formato interno del Horizon Chart
     * @param data - Array de objetos con los datos originales
     * @param xValue - Nombre de la columna que contiene las fechas
     * @param yValue - Nombre de la columna que contiene los valores numéricos
     * @param seriesValue - (Opcional) Nombre de la columna para agrupar en series
     * @returns Map donde la clave es el nombre de la serie y el valor es un array de puntos ordenados
     */
    private processData(data: any[], xValue: string, yValue: string, seriesValue?: string): Map<string, HorizonDataPoint[]> {
        // Parsers de tiempo: Convierten strings de fecha a objetos Date
        // Soporta formatos: "2025-10-22T14:30:00" o "2025-10-22"
        const parseTime = d3.timeParse("%Y-%m-%dT%H:%M:%S") || d3.timeParse("%Y-%m-%d");

        // Transforma cada fila de datos al formato interno
        const preparedData = data.map((d, index) => ({
            x_: d[xValue] ? parseTime(d[xValue]) || new Date(d[xValue]) : null,  // Convierte string a Date
            y_: +d[yValue],  // Convierte a número con el operador unario +
            series: seriesValue ? String(d[seriesValue]) : 'default',  // Identifica la serie o usa 'default'
            id: index,  // Agrega ID único
            originalData: { ...d }  // Almacena copia de los datos originales
        })).filter(d => d.x_ instanceof Date && !Number.isNaN(d.x_.getTime()) && Number.isFinite(d.y_));  // Filtra datos inválidos

        // Agrupa los datos por serie y ordena cada grupo cronológicamente
        // d3.rollup crea un Map<string, HorizonDataPoint[]>
        const series = d3.rollup(
            preparedData,
            (values) => d3.sort(values, d => d.x_!.getTime()),  // Ordena por timestamp
            d => d.series  // Clave de agrupación: nombre de la serie
        );
        
        return series as Map<string, HorizonDataPoint[]>;
    }

    /**
     * Método principal que renderiza el Horizon Chart
     * @param params - Parámetros de configuración del gráfico
     */
    plot(params: HorizonChartParams): void {
        // Desestructura los parámetros recibidos
        const { data, xValue, yValue, seriesValue, bands, height: containerHeight, noSideBar, setSelectedValues } = params;
        
        // Calcula el ancho disponible (resta el espacio del sidebar si está visible)
        let width = params.width ?? 0;
        if (!noSideBar) width = width ? width - SideBar.SIDE_BAR_WIDTH : 0;
        
        // Procesa los datos crudos y los agrupa por serie
        const seriesData = this.processData(data, xValue, yValue, seriesValue);
        const seriesKeys = Array.from(seriesData.keys());
        
        // Calcula la altura por serie basada en la altura total del contenedor
        const availableHeight = (containerHeight ?? 400) - MARGIN.top - MARGIN.bottom;
        const singleHeight = seriesKeys.length > 0 
            ? Math.max(40, availableHeight / seriesKeys.length)  // Mínimo 40px por serie
            : 100;  // Valor por defecto si no hay series
        // Usa la altura total del contenedor
        const totalHeight = containerHeight ?? 400;

        // Inicializa el SVG con las dimensiones calculadas
        this.init(width, totalHeight);
        const GG = this.gGrid;  // Referencia al grupo SVG principal
        
        // Genera un ID único para los elementos SVG (clip-paths y paths)
        const uid = `horizon-${Math.random().toString(16).slice(2)}`;
        let clickSelectButton: ClickSelectButton<SVGGElement> | null = null;
        // --- CÁLCULO DE DOMINIOS (Rangos de datos) ---

        // 1. Aplana todos los datos de todas las series en un solo array
        const allDataPoints = Array.from(seriesData.values()).flat();
        
        // 2. Calcula el dominio X (rango de fechas)
        let xDomain: [Date, Date];
        const xExtent = d3.extent(allDataPoints, d => d.x_);  // Encuentra [fecha_mínima, fecha_máxima]
        
        // Si no hay datos válidos, usa un rango por defecto (ayer - hoy)
        if (xExtent[0] === undefined || xExtent[1] === undefined) {
            const now = new Date();
            xDomain = [d3.timeDay.offset(now, -1), now];
        } else {
            xDomain = [xExtent[0], xExtent[1]];
        }

        // 3. Calcula el dominio Y (rango de valores)
        const yMax = d3.max(allDataPoints, d => d.y_);  // Encuentra el valor máximo
        const yDomain = [0, yMax ?? 0];  // Dominio desde 0 hasta el máximo

        // --- FIN DE CÁLCULO DE DOMINIOS ---

        // --- ESCALAS (Mapean datos a posiciones en píxeles) ---
        
        // Escala X: Convierte fechas a posiciones horizontales en el SVG
        const xScale = d3.scaleTime()
            .domain(xDomain)  // Rango de fechas [fecha_inicio, fecha_fin]
            .range([MARGIN.left, width - MARGIN.right]);  // Espacio físico disponible en píxeles

        // Escala Y: Convierte valores numéricos a posiciones verticales
        // La multiplicación por 'bands' crea el efecto de superposición del Horizon Chart
        const yScale = d3.scaleLinear()
            .domain(yDomain)
            .range([singleHeight, singleHeight - bands * singleHeight]);

        // --- GENERADOR DE ÁREA (Crea las formas SVG) ---
        
        // Define cómo convertir datos en paths SVG de área rellena
        const area = d3.area<HorizonDataPoint>()
            .defined(d => !Number.isNaN(d.y_))
            .x(d => xScale(d.x_))
            .y0(singleHeight)
            .y1(d => yScale(d.y_));

        // --- ESQUEMA DE COLORES ---
        
        // Selecciona una paleta de azules de D3, ajustada al número de bandas
        const colorScheme = d3.schemeBlues[Math.max(3, bands)] || d3.schemeBlues[3];
        // Toma los colores más intensos (últimos N colores de la paleta)
        const colors = colorScheme.slice(0, bands);

        // --- EJE TEMPORAL (Opcional) ---
        
        
        // Crea un eje temporal en la parte superior del gráfico
        GG.append('g')
            .attr('transform', `translate(0, ${MARGIN.top})`)  // Posiciona el eje arriba
            .call(d3.axisTop(xScale).ticks(width / 80).tickSizeOuter(0))  // Crea eje con marcas de tiempo
            .call(g => g.select('.domain').remove());  // Remueve la línea base del eje
    

        // --- CREACIÓN DE GRUPOS POR SERIE ---
        
        // Grupos por serie: Ajustados a la nueva altura
        const seriesGroup = GG.selectAll('.horizon-series')
            .data(seriesData)
            .enter().append('g')
            .attr('class', 'horizon-series')
            .attr('transform', (d, i) => `translate(0, ${i * singleHeight + MARGIN.top})`);

        const handleSeriesClick = this.createSeriesClickHandler(clickSelectButton, setSelectedValues, GG);

        seriesGroup
            .on('mouseenter.hover', function () {
                const enabled = Boolean(clickSelectButton?.isSelected);
                d3.select(this)
                    .classed('click-select-enabled', enabled)
                    .style('cursor', enabled ? 'pointer' : 'default');
            })
            .on('mouseleave.hover', function () {
                d3.select(this)
                    .classed('click-select-enabled', false)
                    .style('cursor', 'default');
            });
        // --- DEFINICIONES SVG (Clip Paths y Paths reutilizables) ---
        
        seriesGroup.append('defs').each(function([key, values], i) {
            const defs = d3.select(this);
            
            defs.append('clipPath')
                .attr('id', `${uid}-clip-${i}`)
                .append('rect')
                .attr('y', 0)
                .attr('width', width)
                .attr('height', singleHeight);
            
            defs.append('path')
                .attr('id', `${uid}-path-${i}`)
                .attr('d', area(values));
        });
        
        // Bandas superpuestas: Ajustadas a la nueva altura
        seriesGroup.append('g')
            .attr('clip-path', (d, i) => `url(#${uid}-clip-${i})`)
            .selectAll('use')
            .data((d, i) => new Array(bands).fill(i))
            .enter().append('use')
            .attr('xlink:href', (i) => `#${uid}-path-${i}`)
            .attr('fill', (_, i) => colors[i]) // CORRECCIÓN: Usa el índice directo, no invertido
            .attr('transform', (_, i) => `translate(0, ${i * singleHeight})`); // Desplazamiento dinámico
      
        // FUNCIONAMIENTO:
        // - Banda 0 (más clara):  transform(0, 0)    - Sin desplazamiento, muestra parte superior
        // - Banda 1:              transform(0, 30)   - Desplazada 30px, muestra segunda franja
        // - Banda 2:              transform(0, 60)   - Desplazada 60px, muestra tercera franja
        // - Banda 3 (más oscura): transform(0, 90)   - Desplazada 90px, muestra parte inferior
        // El clip-path recorta cada banda para que solo sea visible su porción
        // Resultado: Valores altos = más bandas superpuestas = color más intenso

        // --- ETIQUETAS DE SERIE ---
        
        // Etiquetas: Centradas en la nueva altura
        seriesGroup.append('text')
            .attr('x', MARGIN.left + 4)
            .attr('y', (singleHeight + 1) / 2) // Centrado con el padding
            .attr('dy', '0.35em')
            .attr('fill', 'currentColor')
            .style('font-size', '12px')
            .text(([key]) => key);

        // --- BARRA LATERAL DE HERRAMIENTAS (Opcional) ---

        if (!noSideBar) {
            const updateSelectedFromButtons = this.createUpdateSelectedFromButtons(GG, setSelectedValues);

            clickSelectButton = new ClickSelectButton<SVGGElement>(false);
            clickSelectButton.addWhenSelectedCallback(() => {
                seriesGroup.style('cursor', 'pointer');
            });
            clickSelectButton.addWhenUnselectedCallback(() => {
                seriesGroup.style('cursor', 'default');
            });

            const deselectAllButton = new DeselectAllButton(seriesGroup, updateSelectedFromButtons);

            seriesGroup.on('click.clickSelect', (event, d) => {
                if (!clickSelectButton?.isSelected) return;
                clickSelectButton.selectionClickEffect(d3.select(event.currentTarget as SVGGElement));
                handleSeriesClick(event, d);
            });

            const sideBar = new SideBar(this.element, clickSelectButton, deselectAllButton);
            sideBar.inicializar();
        }

    }
}

export class HorizonChartModel extends BaseModel {
    defaults() {
        return {
            ...super.defaults(),
            _model_name: HorizonChartModel.model_name,
            _view_name: HorizonChartModel.view_name,
            dataRecords: [],
            x: String,
            y: String,
            series: String,
            bands: 4,
            mode: 'offset',
            selectedValuesRecords: [],
        };
    }

    static readonly model_name = "HorizonChartModel";
    static readonly view_name = "HorizonChartView";
}

export class HorizonChartView extends BaseView<HorizonChart> {
    params(): HorizonChartParams {
        return {
            data: this.model.get("dataRecords"),
            xValue: this.model.get("x"),
            yValue: this.model.get("y"),
            seriesValue: this.model.get("series"),
            bands: this.model.get("bands"),
            mode: this.model.get("mode"),
            setSelectedValues: this.setSelectedValues.bind(this),
            width: this.width,
            height: this.height,
            noSideBar: false,
        };
    }

    plot(element: HTMLElement) {
        this.widget = new HorizonChart(element);
        this.model.on("change:dataRecords", () => this.replot(), this);
        this.model.on("change:x", () => this.replot(), this);
        this.model.on("change:y", () => this.replot(), this);
        this.model.on("change:series", () => this.replot(), this);
        this.model.on("change:bands", () => this.replot(), this);
        window.addEventListener("resize", () => this.replot());
        this.widget.plot(this.params());
    }

    setSelectedValues(values: any[]) {
        this.model.set({ selectedValuesRecords: values });
        this.model.save_changes();
    }
}