import ipywidgets as widgets
import pandas as pd
from traitlets import Bool, Float, List, Unicode, Int

from vizpro.base_widget import BaseWidget

@widgets.register
class BarPlot(BaseWidget):
    _view_name = Unicode("BarPlotView").tag(sync=True)
    _model_name = Unicode("BarPlotModel").tag(sync=True)

    dataRecords = List([]).tag(sync=True)
    direction = Unicode().tag(sync=True)
    x = Unicode().tag(sync=True)
    y = Unicode().tag(sync=True)
    hue = Unicode().tag(sync=True)
    selectedValuesRecords = List([]).tag(sync=True)

    def __init__(self, data, direction="vertical", **kwargs):
        self.data = data
        self.direction = direction
        self.selectedValues = pd.DataFrame()
        super().__init__(**kwargs)

    @property
    def data(self):
        return pd.DataFrame.from_records(self.dataRecords)

    @data.setter
    def data(self, val):
        self.dataRecords = val.to_dict(orient="records")

    @property
    def selectedValues(self):
        return pd.DataFrame.from_records(self.selectedValuesRecords)

    @selectedValues.setter
    def selectedValues(self, val):
        self.selectedValuesRecords = val.to_dict(orient="records")

    def on_select_values(self, callback):
        self.observe(callback, names=["selectedValuesRecords"])


#Scatter
@widgets.register
class ScatterPlot(BaseWidget):
    _view_name = Unicode("ScatterPlotView").tag(sync=True)
    _model_name = Unicode("ScatterPlotModel").tag(sync=True)

    dataRecords = List([]).tag(sync=True)
    x = Unicode().tag(sync=True)
    y = Unicode().tag(sync=True)
    hue = Unicode().tag(sync=True)
    size = Unicode().tag(sync=True)
    pointSize = Float(5.0).tag(sync=True)
    opacity = Float(0.7).tag(sync=True)
    selectedValuesRecords = List([]).tag(sync=True)

    def __init__(self, data, point_size = 5.0, opacity = 0.7, **kwargs):
        self.data = data
        self.pointSize = point_size
        self.opacity = opacity
        self.selectedValues = pd.DataFrame()
        super().__init__(**kwargs)

    @property
    def data(self):
        return pd.DataFrame.from_records(self.dataRecords)
    
    @data.setter
    def data(self, val):
        self.dataRecords = val.to_dict(orient="records")

    @property
    def selectedValues(self):
        return pd.DataFrame.from_records(self.selectedValuesRecords)
    
    @selectedValues.setter
    def selectedValues(self, val):
        self.selectedValuesRecords = val.to_dict(orient="records")
        
    def on_select_values(self, callback):
        self.observe(callback, names=["selectedValuesRecords"])

# Horizon Chart
@widgets.register
class HorizonChart(BaseWidget):
    """Widget para visualizar gráficos de horizonte."""
    _view_name = Unicode("HorizonChartView").tag(sync=True)
    _model_name = Unicode("HorizonChartModel").tag(sync=True)

    # -- Parámetros sincronizados con el frontend --
    dataRecords = List([]).tag(sync=True)
    x = Unicode().tag(sync=True)
    y = Unicode().tag(sync=True)
    series = Unicode().tag(sync=True) # Columna para agrupar las series
    bands = Int(4).tag(sync=True)      # Número de bandas
    mode = Unicode("offset").tag(sync=True) # 'offset' o 'mirror'
    selectedValuesRecords = List([]).tag(sync=True)

    def __init__(self, data, **kwargs):
        """
        Inicializa el widget del Horizon Chart.
        
        Args:
            data (pd.DataFrame): DataFrame que contiene los datos a visualizar.
            **kwargs: Otros argumentos para el BaseWidget.
        """
        self.data = data
        self.selectedValues = pd.DataFrame()
        super().__init__(**kwargs)

    @property
    def data(self):
        """Obtiene los datos como un DataFrame de pandas."""
        return pd.DataFrame.from_records(self.dataRecords)

    @data.setter
    def data(self, val):
        """Establece los datos desde un DataFrame de pandas."""
        self.dataRecords = val.to_dict(orient="records")

    @property
    def selectedValues(self):
        """Obtiene los valores seleccionados como un DataFrame de pandas."""
        return pd.DataFrame.from_records(self.selectedValuesRecords)

    @selectedValues.setter
    def selectedValues(self, val):
        self.selectedValuesRecords = val.to_dict(orient="records")

    def on_select_values(self, callback):
        self.observe(callback, names=["selectedValuesRecords"])

    def on_select_values(self, callback):
        """Registra una función de callback para cuando los valores seleccionados cambian."""
        self.observe(callback, names=["selectedValuesRecords"])

class RadViz(BaseWidget):
    _view_name = Unicode("RadVizView").tag(sync=True)
    _model_name = Unicode("RadVizModel").tag(sync=True)

    dataRecords = List([]).tag(sync=True)
    dimensions = List([]).tag(sync=True)
    hue = Unicode().tag(sync=True)
    selectedValuesRecords = List([]).tag(sync=True)

    def __init__(self, data, dimensions, hue, **kwargs):
        self.data = data
        self.dimensions = dimensions
        self.hue = hue
        self.selectedValues = pd.DataFrame()
        super().__init__(**kwargs)

    @property
    def data(self):
        return pd.DataFrame.from_records(self.dataRecords)

    @data.setter
    def data(self, val):
        self.dataRecords = val.to_dict(orient="records")

    @property
    def selectedValues(self):
        return pd.DataFrame.from_records(self.selectedValuesRecords)

    @selectedValues.setter
    def selectedValues(self, val):
        self.selectedValuesRecords = val.to_dict(orient="records")

    def on_select_values(self, callback):
        self.observe(callback, names=["selectedValuesRecords"])


@widgets.register
class StarCoordinates(BaseWidget):
    _view_name = Unicode("StarCoordinatesView").tag(sync=True)
    _model_name = Unicode("StarCoordinatesModel").tag(sync=True)

    dataRecords = List([]).tag(sync=True)
    dimensions = List([]).tag(sync=True)
    hue = Unicode().tag(sync=True)
    selectedValuesRecords = List([]).tag(sync=True)

    def __init__(self, data, dimensions, hue, **kwargs):
        self.data = data
        self.dimensions = dimensions
        self.hue = hue
        self.selectedValues = pd.DataFrame()
        super().__init__(**kwargs)

    @property
    def data(self):
        return pd.DataFrame.from_records(self.dataRecords)

    @data.setter
    def data(self, val):
        self.dataRecords = val.to_dict(orient="records")

    @property
    def selectedValues(self):
        return pd.DataFrame.from_records(self.selectedValuesRecords)

    @selectedValues.setter
    def selectedValues(self, val):
        self.selectedValuesRecords = val.to_dict(orient="records")

    def on_select_values(self, callback):
        self.observe(callback, names=["selectedValuesRecords"])
