import ipywidgets as widgets
import pandas as pd
from traitlets import Bool, Float, List, Unicode, Int
from vizpro.colors import colors
from vizpro.base_widget import BaseWidget

class TextBaseWidget(BaseWidget):
    value = Unicode().tag(sync=True)
    placeholder = Unicode().tag(sync=True)
    description = Unicode().tag(sync=True)
    disabled = Bool().tag(sync=True)

@widgets.register
class Button(BaseWidget):
    _view_name = Unicode("ButtonView").tag(sync=True)
    _model_name = Unicode("ButtonModel").tag(sync=True)

    description = Unicode().tag(sync=True)
    disabled = Bool().tag(sync=True)
    _clicked = Bool().tag(sync=True)

    def on_click(self, callback):
        self.observe(callback, names=["_clicked"])

@widgets.register
class Checkbox(BaseWidget):
    _view_name = Unicode("CheckboxView").tag(sync=True)
    _model_name = Unicode("CheckboxModel").tag(sync=True)

    description = Unicode().tag(sync=True)
    checked = Bool().tag(sync=True)

    def on_check(self, callback):
        self.observe(callback, names=["checked"])

@widgets.register
class Dropdown(BaseWidget):
    _view_name = Unicode("DropdownView").tag(sync=True)
    _model_name = Unicode("DropdownModel").tag(sync=True)

    dataRecords = List([]).tag(sync=True)
    variable = Unicode().tag(sync=True)
    description = Unicode().tag(sync=True)
    options = List().tag(sync=True)
    value = Unicode().tag(sync=True)
    disabled = Bool().tag(sync=True)
    _clicked = Bool().tag(sync=True)

    def __init__(self, data=pd.DataFrame(), **kwargs):
        self.data = data
        super().__init__(**kwargs)

    @property
    def data(self):
        return pd.DataFrame.from_records(self.dataRecords)

    @data.setter
    def data(self, val):
        self.dataRecords = val.to_dict(orient="records")

    def on_select(self, callback):
        self.observe(callback, names=["value"])

@widgets.register
class Input(TextBaseWidget):
    _view_name = Unicode("InputView").tag(sync=True)
    _model_name = Unicode("InputModel").tag(sync=True)

    def on_text_changed(self, callback):
        self.observe(callback, names=["value"])

@widgets.register
class RangeSlider(BaseWidget):
    _view_name = Unicode("RangeSliderView").tag(sync=True)
    _model_name = Unicode("RangeSliderModel").tag(sync=True)

    dataRecords = List([]).tag(sync=True)
    variable = Unicode().tag(sync=True)
    step = Float().tag(sync=True)
    description = Unicode().tag(sync=True)
    fromValue = Float().tag(sync=True)
    toValue = Float().tag(sync=True)
    minValue = Float().tag(sync=True)
    maxValue = Float().tag(sync=True)

    def __init__(self, data=pd.DataFrame(), **kwargs):
        self.data = data
        super().__init__(**kwargs)

    @property
    def data(self):
        return pd.DataFrame.from_records(self.dataRecords)

    @data.setter
    def data(self, val):
        self.dataRecords = val.to_dict(orient="records")

    def on_drag(self, callback):
        self.observe(callback, names=["fromValue", "toValue"])



@widgets.register
class TextArea(TextBaseWidget):
    _view_name = Unicode("TextAreaView").tag(sync=True)
    _model_name = Unicode("TextAreaModel").tag(sync=True)

    def on_text_changed(self, callback):
        self.observe(callback, names=["value"])


@widgets.register
class Text(TextBaseWidget):
    _view_name = Unicode("TextView").tag(sync=True)
    _model_name = Unicode("TextModel").tag(sync=True)

@widgets.register
class SelectColor(BaseWidget):
    _view_name = Unicode("SelectColorView").tag(sync=True)
    _model_name = Unicode("SelectColorModel").tag(sync=True)

    description = Unicode().tag(sync=True)
    selectedPalette = Unicode("Blues").tag(sync=True)
    selectedNumColors = Int(5).tag(sync=True)
    disabled = Bool().tag(sync=True)
    _paletteChanged = Bool().tag(sync=True)

    def on_palette_changed(self, callback):
        self.observe(callback, names=["selectedPalette", "selectedNumColors"])

    def get_colors(self):
        """Returns the currently selected color palette as a list of hex colors"""
        
        palette = colors.get(self.selectedPalette, {})
        return palette.get(self.selectedNumColors, [])
