# /// script
# dependencies = ["click", "requests"]
# ///
#
# You can run this script directly without installing Dallinger,
# for example:
#    uv run generate_constraints.py --update-existing
# or
#    curl -s https://raw.githubusercontent.com/Dallinger/Dallinger/master/dallinger/constraints.py | uv run -
"""
Constraints file utilities for Dallinger.

Dallinger experiment directories contain a requirements.txt file specifying the requested Python dependencies.
A constraints.txt file is generated by running `dallinger constraints generate` in the experiment directory.
This file records a full list of all dependencies, both explicit and implicit, compiled with reference to three files:
- requirements.txt;
- dev-requirements.txt, a file sourced from the Dallinger GitHub repository which contains pinned versions for all
  Dallinger dependencies, as tested in the CI;
- .python-version, a file specifying the Python version to use.

When the experiment is deployed, packages are installed from the constraints.txt file.

Originally the constraints.txt file was generated by pip-compile. Now by default we use uv,
which is quite a lot faster and also has some useful features such as following .python-version files.
We still support pip-compile as a fallback if uv is not available, but if the new logic works well we should
consider dropping pip-compile entirely.
"""
import contextlib
import logging
import os
import platform
import re
import subprocess
import tempfile
from hashlib import md5
from pathlib import Path
from typing import Optional

import click
import requests

logger = logging.getLogger(__name__)

python_version_path = Path(".python-version")
requirements_path = Path("requirements.txt")
pyproject_path = Path("pyproject.toml")
constraints_path = Path("constraints.txt")


@click.group()
def constraints_cli():
    """Dallinger constraints file utilities."""


@constraints_cli.command()
def check():
    """
    Check the working directory to see whether a constraints.txt file exists and is up to date. Raises a ValueError if not.
    """
    check_constraints()


def check_constraints():
    """
    Check the working directory to see whether a constraints.txt file exists and is up to date. Raises a ValueError if not.
    """
    assert_python_version_file_presence()
    assert_constraints_file_presence()

    input_path = _find_input_path()

    if not _constraints_up_to_date(input_path):
        raise ValueError(
            f"constraints.txt file in {Path.cwd()} is not up to date with {input_path.name}."
        )
    logger.info(
        f"constraints.txt file in {Path.cwd()} exists and is up to date with {input_path.name}."
    )


def assert_python_version_file_presence():
    if not python_version_path.exists():
        raise FileNotFoundError(
            f"The experiment directory ({Path.cwd()}) is missing a .python-version file. "
            "Please create a text file called .python-version in your experiment directory, "
            "and write the Python version you wish to use in that file (just as numbers, e.g. 3.12.10). "
            f"Note: you're currently using Python version {platform.python_version()}."
        )


def ensure_python_version_file_presence():
    if not python_version_path.exists():
        python_version_path.write_text(
            platform.python_version() + "\n", encoding="utf-8"
        )


def assert_constraints_file_presence():
    if not constraints_path.exists():
        raise FileNotFoundError(
            f"{constraints_path.name} file not found in {Path.cwd()}. "
            "You can create one by running `dallinger constraints generate` in your experiment directory."
        )


@constraints_cli.command()
def generate():
    """
    Generate a constraints.txt file for the current directory.
    """
    generate_constraints()


def generate_constraints():
    """
    Generate a constraints.txt file for the current working directory.

    The command looks for an input file in the current working directory,
    either requirements.txt or pyproject.toml.
    If neither are present, a requirements.txt file is created with
    dallinger as its only dependency.

    This input file is processed to identify the requested version of Dallinger.
    The corresponding dev-requirements.txt file for that version of Dallinger is then
    sourced from the Dallinger GitHub repository. This file specifies pinned versions
    for all Dallinger dependencies, validated in the CI.

    The constraints.txt file is then generated by reference to both the input file
    and the dev-requirements.txt file. It contains a list of versions for all dependencies
    (explicit and implicit).

    This generation process uses ``uv pip-compile`` if uv is available, otherwise the slower
    legacy ``pip-compile`` is used.

    If the constraints.txt file exists already it will be overwritten.
    For a version of this function that does not automatically overwrite the constraints.txt file,
    see ``ensure_constraints_file_presence``.

    """
    input_path = _find_input_path()
    output_path = Path("constraints.txt")

    ensure_python_version_file_presence()

    dallinger_reference = _get_dallinger_reference(input_path)
    dallinger_dev_requirements_path = _get_dallinger_dev_requirements_path(
        dallinger_reference
    )
    _test_dallinger_dev_requirements_path(dallinger_dev_requirements_path)

    print(
        f"Compiling constraints.txt file from {input_path} and {dallinger_dev_requirements_path}"
    )

    _pip_compile(
        input_path,
        output_path,
        constraints=[dallinger_dev_requirements_path],
    )

    _make_constraints_paths_relative()


@constraints_cli.command()
def ensure():
    """
    Ensure that a constraints.txt file exists for the specified directory,
    preserving the existing constraints.txt file if it exists and is up to date,
    and updating it if it is out of date.
    """
    ensure_constraints_file_presence(Path.cwd())


def ensure_constraints_file_presence(directory):
    """
    Ensures that a ``constraints.txt`` file exists in the specified directory.

    - If the environment variable SKIP_DEPENDENCY_CHECK is set, no action will be performed.
    - If no ``constraints.txt`` file exists, one will be automatically generated using ``generate_constraints``.
    - If a manually written ``constraints.txt`` file exists already, then no action will be taken.
    - If an automatically generated ``constraints.txt`` file exists already,
      but it seems up-to-date with the input file (i.e. the MD5 hash of the input file is present in the constraints.txt file),
      then no action will be taken.
    - If an automatically generated ``constraints.txt`` file exists already, but it seems out-of-date,
      then ``constraints.txt`` will be automatically updated.
    """
    if os.environ.get("SKIP_DEPENDENCY_CHECK"):
        return

    input_path = _find_input_path()

    if constraints_path.exists():
        if not _constraints_autogenerated():
            logger.info(
                "%s was written manually, no attempt will be made to update it.",
                constraints_path,
            )
            return
        if _constraints_up_to_date(input_path):
            logger.info("%s is up to date with %s.", constraints_path, input_path)
            return
        logger.info(
            "%s is out of date with %s, updating.", constraints_path, input_path
        )

    with working_directory(directory):
        generate_constraints()


def _find_input_path() -> Path:
    if requirements_path.exists():
        input_path = requirements_path
    elif pyproject_path.exists():
        input_path = pyproject_path
    else:
        logger.warning(
            "No requirements.txt or pyproject.toml file found, will autogenerate a requirements.txt"
        )
        requirements_path.write_text("dallinger\n")
        input_path = requirements_path

    return input_path


def _get_md5sum(input_path: Path) -> str:
    with open(input_path, "rb") as f:
        file_bytes = f.read()
    return md5(file_bytes).hexdigest()


def _constraints_autogenerated() -> bool:
    text = constraints_path.read_text()
    return bool(re.search(r"This file (is|was) autogenerated", text))


def _constraints_up_to_date(input_path: Path) -> bool:
    assert_python_version_file_presence()
    python_version = _get_requested_python_version()
    return (
        f"Python {python_version}\n" in constraints_path.read_text()
        and _get_md5sum(input_path) in constraints_path.read_text()
    )


def _get_requested_python_version() -> str:
    assert_python_version_file_presence()
    return python_version_path.read_text().strip()


def _get_dallinger_reference(input_path: Path) -> str:
    explicit_reference = _get_explicit_dallinger_reference(input_path)
    if explicit_reference:
        return explicit_reference
    else:
        return _get_implied_dallinger_reference(input_path)


def _get_explicit_dallinger_reference(input_path: Path) -> Optional[str]:
    release = _get_explicit_dallinger_numbered_release(input_path)
    if release:
        return f"v{release}"
    else:
        return _get_explicit_dallinger_github_requirement(input_path)


def _get_explicit_dallinger_numbered_release(input_path: Path) -> Optional[str]:
    # Should catch patterns like dallinger[docker,test]==11.5.0
    pattern = re.compile(r"dallinger(?:\[[^\]]+\])?==([0-9]+\.[0-9]+\.[0-9]+)")
    with open(input_path, "r") as f:
        for line in f:
            match = pattern.search(line)
            if match:
                return match.group(1)
    return None


def _get_explicit_dallinger_github_requirement(input_path: Path) -> Optional[str]:
    # e.g. dallinger[docker]@git+https://github.com/Dallinger/Dallinger.git@my-branch#egg=dallinger
    pattern = re.compile(
        r"dallinger(?:\[[^\]]+\])?\s*@\s*git\+https://github\.com/Dallinger/Dallinger(?:\.git)?@([^\s#]+)(?:#.*)?"
    )
    with open(input_path, "r") as f:
        for line in f:
            match = pattern.search(line)
            if match:
                return match.group(1)
    return None


def _get_implied_dallinger_reference(input_path: Path) -> str:
    with tempfile.NamedTemporaryFile(suffix=".txt") as tmpfile:
        _pip_compile(input_path, tmpfile.name, constraints=None)
        retrieved = _get_explicit_dallinger_reference(Path(tmpfile.name))
        if retrieved is None:
            with open(tmpfile.name, "r", encoding="utf-8") as f:
                for line in f:
                    print(f"  {line.rstrip()}")
            raise ValueError(
                f"Failed to retrieve an implied Dallinger reference from {input_path} "
                "(see above for the indirect dependency list that was searched). "
                "Consider specifying Dallinger explicitly in the requirements.txt file."
            )
    return retrieved


def _get_dallinger_dev_requirements_path(dallinger_reference: str) -> str:
    return f"https://raw.githubusercontent.com/Dallinger/Dallinger/{dallinger_reference}/dev-requirements.txt"


def _test_dallinger_dev_requirements_path(url: str):
    try:
        response = requests.get(url, timeout=10)
    except requests.exceptions.ConnectionError as e:
        raise RuntimeError(
            """It looks like you're offline. Dallinger can't generate constraints
To get a valid constraints.txt file you can copy the requirements.txt file:
cp requirements.txt constraints.txt"""
        ) from e
    if response.status_code != 200:
        raise ValueError(
            f"{url} not found. Please make sure your specified Dallinger "
            "version exists in the Dallinger repository. "
        )


def _pip_compile(in_file, out_file, constraints: Optional[list] = None):
    in_file = Path(in_file)
    out_file = Path(out_file)

    if not in_file.resolve().parent == Path.cwd():
        # `uv pip compile` intelligently infers the Python version from the working directory, looking for example
        # for a `.python-version` file or a `.venv` directory. We therefore should make sure we are running from the
        # experiment directory.
        raise ValueError(
            f"The input file ({in_file}) is not in the current working directory ({Path.cwd()}). "
            "Please rerun this command from the same directory as the input file."
        )

    use_uv = uv_available()
    requested_python_version = _get_requested_python_version()
    md5sum = _get_md5sum(in_file)
    compile_info = f"dallinger constraints generate\n#\n# Compiled from a {Path(in_file).name} file with md5sum {md5sum} and a .python-version file requesting Python {requested_python_version}"

    if use_uv:
        cmd = [
            "uv",
            "pip",
            "compile",
            "--python-version",
            requested_python_version,
        ]
    else:
        logger.info(
            "Calling `pip-compile` (consider installing uv for faster compilation)..."
        )
        cmd = ["pip-compile"]
    cmd += [
        str(in_file),
        "--output-file",
        str(out_file),
    ]
    if constraints:
        for constraint in constraints:
            cmd += ["--constraint", constraint]

    env = dict(os.environ)
    if compile_info:
        if use_uv:
            env["UV_CUSTOM_COMPILE_COMMAND"] = compile_info
        else:
            env["CUSTOM_COMPILE_COMMAND"] = compile_info
    try:
        subprocess.check_output(
            cmd,
            env=env,
            stderr=subprocess.STDOUT,
        )
        if not use_uv:
            # Verify that pip used the correct Python version
            _check_pip_compile_output(out_file)

    except subprocess.CalledProcessError as e:
        message = e.output.decode()
        if "No solution found when resolving dependencies" in message:
            msg = (
                "An error occurred when compiling the constraints.txt file. "
                f"See if you can adjust {in_file.name} to resolve the dependency conflicts, "
                "e.g. by selecting different package versions or removing version pins entirely.\n\n"
            )
            if len(constraints) > 0:
                msg += f"Note that these packages need to be compatible with the pins specified in {constraints}.\n\n"
            msg += (
                "If you can't resolve the issue, but you want to proceed anyway, you can manually create an empty constraints.txt file "
                f"at {out_file}. See below output for more details:\n\n{message}"
            )
            raise RuntimeError(msg) from e
        else:
            print(message)
            raise e


def _check_pip_compile_output(out_file):
    assert out_file.exists()
    requested_python_version = _get_requested_python_version()
    obtained_python_version = _extract_python_version_from_pip_compile_output(out_file)

    if not _python_versions_consistent(
        requested_python_version, obtained_python_version
    ):
        raise ValueError(
            f"pip-compile produced a constraints.txt file with the wrong Python version "
            f"(the .python-version file requested {requested_python_version}, "
            f"but pip-compile used {obtained_python_version}). "
            "Make sure the Python version in your local environment "
            "matches the one specified in the .python-version file, "
            "or alternatively install uv (`pip install uv`) and then try again "
            "(uv follows .python-version automatically, whereas pip-compile does not)."
        )


def _python_versions_consistent(v1, v2):
    for a, b in zip(v1.split("."), v2.split(".")):
        if int(a) != int(b):
            return False
    return True


def _extract_python_version_from_pip_compile_output(out_file):
    assert out_file.exists()
    match = re.search(r"Python (\d+\.\d+(\.\d+)?)", out_file.read_text())
    if match:
        return match.group(1)
    raise logger.warning(
        "Could not find a Python version in %s. This points to a bug in dallinger.constraints._extract_python_version_from_pip_compile_output.",
        {out_file.name},
    )


def uv_available() -> bool:
    """
    Check whether uv is available for use.
    """
    try:
        subprocess.check_output(["uv", "--version"])
        return True
    except subprocess.CalledProcessError:
        return False


def _make_constraints_paths_relative():
    constraints_contents = constraints_path.read_text()
    constraints_contents_amended = re.sub(
        "via -r .*requirements.txt", "via -r requirements.txt", constraints_contents
    )
    constraints_path.write_text(constraints_contents_amended)


@contextlib.contextmanager
def working_directory(path):
    start_dir = os.getcwd()
    try:
        os.chdir(path)
        yield
    finally:
        os.chdir(start_dir)


if __name__ == "__main__":
    logging.basicConfig(
        level=logging.INFO,
        format="%(message)s",
    )
    constraints_cli()
