import os
import platform
import re
from pathlib import Path
from unittest.mock import patch

import pytest

from dallinger.constraints import (
    _constraints_autogenerated,
    _get_explicit_dallinger_numbered_release,
    _pip_compile,
    _python_versions_consistent,
    check_constraints,
    ensure_constraints_file_presence,
    generate_constraints,
    working_directory,
)
from dallinger.utils import check_output


@pytest.fixture
def python_version_file(tempdir):
    with open(Path(tempdir) / ".python-version", "w") as f:
        f.write(platform.python_version())


@pytest.mark.parametrize("uv_available_value", [True, False])
@pytest.mark.slow
class TestGenerateConstraints:
    @pytest.fixture(autouse=True)
    def mock_uv_available(self, monkeypatch, uv_available_value):
        # We test both uv and pip-compile (the fallback if uv is not available)
        monkeypatch.setattr(
            "dallinger.constraints.uv_available", lambda: uv_available_value
        )

    def test_constraints_dallinger_release(self, in_tempdir, python_version_file):
        with open("requirements.txt", "w") as f:
            f.write("dallinger==11.4.0")
        generate_constraints()
        assert Path("constraints.txt").exists()
        assert _constraints_autogenerated()

        # see https://raw.githubusercontent.com/Dallinger/Dallinger/v11.4.0/dev-requirements.txt
        # # to verify that flask==3.1.1 is indeed the correct version
        assert "flask==3.1.1" in Path("constraints.txt").read_text()

        # Verify that the Python version is included in the constraints.txt file
        # (irrespective of whether uv or pip-compile is used)
        content = Path("constraints.txt").read_text()
        python_version_pattern = r"Python \d+\.\d+(\.\d+)?"
        assert re.search(
            python_version_pattern, content
        ), "constraints.txt missing Python version information"

    def test_constraints_dallinger_commit(self, in_tempdir, python_version_file):
        # Note: pip-compile under Python 3.13 failed to compile requirements for a Python 3.12 version of Dallinger.
        # This might cause problems again when we upgrade to Python 3.14.
        # uv is more robust; by that point, perhaps we should consider dropping pip-compile entirely.
        with open("requirements.txt", "w") as f:
            f.write(
                "dallinger@git+https://github.com/Dallinger/Dallinger.git@e18ce87985dc9cef64fe4ebcddfc1027f41c3398#egg=dallinger"
            )
        generate_constraints()

        # see https://raw.githubusercontent.com/Dallinger/Dallinger/e18ce87985dc9cef64fe4ebcddfc1027f41c3398/dev-requirements.txt
        # to verify that 3.1.1 is indeed the correct version
        assert "flask==3.1.1" in Path("constraints.txt").read_text()

    def test_constraints_psynet(self, in_tempdir, python_version_file):
        with open("requirements.txt", "w") as f:
            f.write("psynet==12.0.0")
        generate_constraints()
        assert Path("constraints.txt").exists()
        dallinger_version = _get_explicit_dallinger_numbered_release(
            Path("constraints.txt")
        )
        major, minor, patch = dallinger_version.split(".")

        # To verify the correct version see https://gitlab.com/PsyNetDev/PsyNet/-/blob/v12.0.0/pyproject.toml?ref_type=tags
        assert major == "11"
        assert int(minor) >= 1
        if int(minor) == 1:
            assert int(patch) >= 1

    def test_caching(self, tempdir, in_tempdir, python_version_file):
        requirements_path = Path("requirements.txt")
        requirements_path.write_text("dallinger==11.4.0\n")
        constraints_path = Path("constraints.txt")

        ensure_constraints_file_presence(tempdir)

        assert constraints_path.exists()
        assert "dallinger==11.4.0" in constraints_path.read_text()

        original_mtime = os.path.getmtime(constraints_path)

        ensure_constraints_file_presence(tempdir)

        # No change to requirements.txt so no change to constraints.txt
        assert os.path.getmtime(constraints_path) == original_mtime

        with open(requirements_path, "a") as f:
            f.write("scipy")

        # Change to requirements.txt so constraints.txt should be updated
        original_mtime = os.path.getmtime(constraints_path)
        ensure_constraints_file_presence(tempdir)
        assert os.path.getmtime(constraints_path) > original_mtime

    def test_constraints_written_manually(
        self, in_tempdir, tempdir, python_version_file
    ):
        with open("constraints.txt", "w") as f:
            f.write("dallinger==11.4.0")

        original_mtime = os.path.getmtime("constraints.txt")
        ensure_constraints_file_presence(tempdir)

        # Manually written constraints.txt files should not be updated
        assert os.path.getmtime(Path("constraints.txt")) == original_mtime

    def test_check_constraints(self, in_tempdir, tempdir, python_version_file):
        with open("requirements.txt", "w") as f:
            f.write("dallinger==11.4.0")
        with pytest.raises(FileNotFoundError):
            check_constraints()
        generate_constraints()
        check_constraints()


def test_check_constraints_wrong_python_version(in_tempdir):
    with open(".python-version", "w") as f:
        f.write("3.15.0")

    with open("requirements.txt", "w") as f:
        f.write("dallinger==11.4.0")

    with patch("dallinger.constraints.uv_available", return_value=False):
        with pytest.raises(
            ValueError,
            match="pip-compile produced a constraints.txt file with the wrong Python version",
        ):
            generate_constraints()


@pytest.mark.slow
def test_constraints_script_works_independently(tempdir, python_version_file):
    """Test that constraints.py runs when used as a self-contained script."""
    script_path = Path("dallinger/constraints.py").resolve()
    with working_directory(tempdir):
        check_output(["uv", "run", str(script_path), "generate"])
        check_output(["uv", "run", str(script_path), "check"])
        assert Path("constraints.txt").exists()
        check_output(["uv", "run", str(script_path), "ensure"])
        assert Path("constraints.txt").exists()


@pytest.mark.slow
def test_legacy_interface(in_tempdir, python_version_file):
    """Test that the legacy interface works."""
    check_output(["dallinger", "generate-constraints"])
    assert Path("constraints.txt").exists()


@pytest.mark.slow
def test_impossible_resolution(in_tempdir, python_version_file):
    with open("requirements.txt", "w") as f:
        f.write("dallinger==10.0.0\n")
        f.write("dallinger==10.0.1\n")
    with pytest.raises(
        RuntimeError,
        match="An error occurred when compiling the constraints.txt file. See if you can adjust",
    ):
        generate_constraints()


def test_constraints_run_from_wrong_directory(tempdir, python_version_file):
    with open(Path(tempdir) / "requirements.txt", "w") as f:
        f.write("dallinger==11.4.0\n")
    with pytest.raises(
        ValueError,
        match="not in the current working directory",
    ):
        _pip_compile(
            Path(tempdir) / "requirements.txt", Path(tempdir) / "constraints.txt"
        )

    with working_directory(tempdir):
        _pip_compile(Path("requirements.txt"), Path("constraints.txt"))


def test_python_versions_consistent():
    assert _python_versions_consistent("3.15.0", "3.15.0")
    assert _python_versions_consistent("3.15", "3.15.1")
    assert not _python_versions_consistent("3.15", "3.16")
    assert not _python_versions_consistent("3.15", "3.14.1")
