"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const path_1 = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_autoscaling_1 = require("aws-cdk-lib/aws-autoscaling");
const aws_ec2_1 = require("aws-cdk-lib/aws-ec2");
const nag_rules_1 = require("../../nag-rules");
/**
 * EBS volumes have encryption enabled
 * @param node the CfnResource to check
 */
exports.default = Object.defineProperty((node) => {
    if (node instanceof aws_ec2_1.CfnVolume) {
        const encryption = nag_rules_1.NagRules.resolveIfPrimitive(node, node.encrypted);
        if (encryption !== true) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        return nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else if (node instanceof aws_ec2_1.CfnInstance) {
        const instanceEBSState = InstanceEBSState(node);
        const launchTemplate = aws_cdk_lib_1.Stack.of(node).resolve(node.launchTemplate);
        for (const child of aws_cdk_lib_1.Stack.of(node).node.findAll()) {
            if (child instanceof aws_ec2_1.CfnLaunchTemplate) {
                if (isMatchingLaunchTemplate(child, launchTemplate.launchTemplateName, launchTemplate.launchTemplateId)) {
                    const instanceLaunchTemplateState = InstanceLaunchTemplateState(child);
                    if (instanceEBSState === BlockDevicesState.Absent &&
                        instanceLaunchTemplateState === BlockDevicesState.Absent) {
                        return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
                    }
                    if (instanceEBSState === BlockDevicesState.Unencrypted ||
                        instanceLaunchTemplateState === BlockDevicesState.Unencrypted) {
                        return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
                    }
                    return nag_rules_1.NagRuleCompliance.COMPLIANT;
                }
            }
        }
        if (instanceEBSState === BlockDevicesState.Encrypted) {
            return nag_rules_1.NagRuleCompliance.COMPLIANT;
        }
        return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
    }
    else if (node instanceof aws_autoscaling_1.CfnLaunchConfiguration) {
        const blockDeviceMappings = aws_cdk_lib_1.Stack.of(node).resolve(node.blockDeviceMappings);
        if (blockDeviceMappings == undefined) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        else {
            for (const blockDeviceMapping of blockDeviceMappings) {
                const encryption = nag_rules_1.NagRules.resolveIfPrimitive(node, blockDeviceMapping.ebs.encrypted);
                if (encryption !== true) {
                    return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
                }
            }
        }
        return nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else if (node instanceof aws_autoscaling_1.CfnAutoScalingGroup) {
        const launchTemplate = aws_cdk_lib_1.Stack.of(node).resolve(node.launchTemplate);
        for (const child of aws_cdk_lib_1.Stack.of(node).node.findAll()) {
            if (child instanceof aws_ec2_1.CfnLaunchTemplate) {
                if (isMatchingLaunchTemplate(child, launchTemplate.launchTemplateName, launchTemplate.launchTemplateId)) {
                    if (InstanceLaunchTemplateState(child) === BlockDevicesState.Encrypted) {
                        return nag_rules_1.NagRuleCompliance.COMPLIANT;
                    }
                    return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
                }
            }
        }
        return nag_rules_1.NagRuleCompliance.NOT_APPLICABLE;
    }
    else {
        return nag_rules_1.NagRuleCompliance.NOT_APPLICABLE;
    }
}, 'name', { value: (0, path_1.parse)(__filename).name });
var BlockDevicesState;
(function (BlockDevicesState) {
    BlockDevicesState[BlockDevicesState["Encrypted"] = 0] = "Encrypted";
    BlockDevicesState[BlockDevicesState["Unencrypted"] = 1] = "Unencrypted";
    BlockDevicesState[BlockDevicesState["Absent"] = 2] = "Absent";
})(BlockDevicesState || (BlockDevicesState = {}));
function isMatchingLaunchTemplate(node, launchTemplateName, launchTemplateId) {
    return (launchTemplateName === node.launchTemplateName ||
        launchTemplateId === nag_rules_1.NagRules.resolveResourceFromIntrinsic(node, node.ref));
}
function InstanceLaunchTemplateState(launchTemplate) {
    const launchTemplateData = aws_cdk_lib_1.Stack.of(launchTemplate).resolve(launchTemplate.launchTemplateData);
    if (launchTemplateData.blockDeviceMappings === undefined ||
        launchTemplateData.blockDeviceMappings.length === 0) {
        return BlockDevicesState.Absent;
    }
    else {
        const launchTemplateBlockDeviceMappings = aws_cdk_lib_1.Stack.of(launchTemplate).resolve(launchTemplateData.blockDeviceMappings);
        const devicesAllEncrypted = launchTemplateBlockDeviceMappings.every((blockDeviceMapping) => {
            const encryption = nag_rules_1.NagRules.resolveIfPrimitive(launchTemplate, blockDeviceMapping.ebs.encrypted);
            return encryption === true;
        });
        return devicesAllEncrypted
            ? BlockDevicesState.Encrypted
            : BlockDevicesState.Unencrypted;
    }
}
function InstanceEBSState(node) {
    const blockDeviceMappings = aws_cdk_lib_1.Stack.of(node).resolve(node.blockDeviceMappings);
    if (blockDeviceMappings === undefined || blockDeviceMappings.length === 0) {
        return BlockDevicesState.Absent;
    }
    else {
        const devicesAllEncrypted = blockDeviceMappings.every((blockDeviceMapping) => {
            const encryption = nag_rules_1.NagRules.resolveIfPrimitive(node, blockDeviceMapping.ebs.encrypted);
            return encryption === true;
        });
        return devicesAllEncrypted
            ? BlockDevicesState.Encrypted
            : BlockDevicesState.Unencrypted;
    }
}
//# sourceMappingURL=data:application/json;base64,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