"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const path_1 = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_elasticloadbalancing_1 = require("aws-cdk-lib/aws-elasticloadbalancing");
const nag_rules_1 = require("../../nag-rules");
/**
 * CLB listeners are configured for secure (HTTPs or SSL) protocols for client communication
 * @param node the CfnResource to check
 */
exports.default = Object.defineProperty((node) => {
    if (node instanceof aws_elasticloadbalancing_1.CfnLoadBalancer) {
        const listeners = aws_cdk_lib_1.Stack.of(node).resolve(node.listeners);
        for (const listener of listeners) {
            const resolvedListener = aws_cdk_lib_1.Stack.of(node).resolve(listener);
            const protocol = nag_rules_1.NagRules.resolveIfPrimitive(node, resolvedListener.protocol);
            const instanceProtocol = nag_rules_1.NagRules.resolveIfPrimitive(node, resolvedListener.instanceProtocol);
            if (protocol.toLowerCase() == 'ssl') {
                if (!(instanceProtocol == undefined ||
                    instanceProtocol.toLowerCase() == 'ssl')) {
                    return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
                }
            }
            else if (protocol.toLowerCase() == 'https') {
                if (!(instanceProtocol == undefined ||
                    instanceProtocol.toLowerCase() == 'https')) {
                    return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
                }
            }
        }
        return nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else {
        return nag_rules_1.NagRuleCompliance.NOT_APPLICABLE;
    }
}, 'name', { value: (0, path_1.parse)(__filename).name });
//# sourceMappingURL=data:application/json;base64,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