"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const path_1 = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_glue_1 = require("aws-cdk-lib/aws-glue");
const nag_rules_1 = require("../../nag-rules");
/**
 * Glue crawlers and jobs have CloudWatch Log encryption enabled
 * @param node the CfnResource to check
 */
exports.default = Object.defineProperty((node) => {
    if (node instanceof aws_glue_1.CfnCrawler || node instanceof aws_glue_1.CfnJob) {
        let securityConfigurationId;
        if (node instanceof aws_glue_1.CfnCrawler) {
            securityConfigurationId = nag_rules_1.NagRules.resolveResourceFromIntrinsic(node, node.crawlerSecurityConfiguration);
        }
        else {
            securityConfigurationId = nag_rules_1.NagRules.resolveResourceFromIntrinsic(node, node.securityConfiguration);
        }
        if (securityConfigurationId == undefined) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        let found = false;
        for (const child of aws_cdk_lib_1.Stack.of(node).node.findAll()) {
            if (child instanceof aws_glue_1.CfnSecurityConfiguration) {
                if (isMatchingSecurityConfig(child, securityConfigurationId)) {
                    found = true;
                    break;
                }
            }
        }
        if (!found) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        return nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else {
        return nag_rules_1.NagRuleCompliance.NOT_APPLICABLE;
    }
}, 'name', { value: (0, path_1.parse)(__filename).name });
/**
 * Helper function to check whether the referenced Security Configuration encrypts CloudWatch Logs
 * @param node the CfnSecurityConfiguration to check
 * @param securityConfigurationId the Cfn Logical ID of the security configuration
 * returns whether the CfnSecurityConfiguration encrypts CloudWatch Logs
 */
function isMatchingSecurityConfig(node, securityConfigurationId) {
    const resolvedSecurityConfigurationLogicalId = nag_rules_1.NagRules.resolveResourceFromIntrinsic(node, node.ref);
    if (resolvedSecurityConfigurationLogicalId === securityConfigurationId ||
        node.name === securityConfigurationId) {
        const encryptionConfiguration = aws_cdk_lib_1.Stack.of(node).resolve(node.encryptionConfiguration);
        const cloudWatchEncryption = encryptionConfiguration.cloudWatchEncryption;
        if (cloudWatchEncryption !== undefined) {
            const cloudWatchEncryptionMode = nag_rules_1.NagRules.resolveIfPrimitive(node, cloudWatchEncryption.cloudWatchEncryptionMode);
            if (cloudWatchEncryptionMode === 'SSE-KMS') {
                return true;
            }
        }
    }
    return false;
}
//# sourceMappingURL=data:application/json;base64,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