"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const path_1 = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_elasticsearch_1 = require("aws-cdk-lib/aws-elasticsearch");
const aws_opensearchservice_1 = require("aws-cdk-lib/aws-opensearchservice");
const nag_rules_1 = require("../../nag-rules");
/**
 * OpenSearch Service domains only grant access via allowlisted IP addresses
 * @param node the CfnResource to check
 */
exports.default = Object.defineProperty((node) => {
    if (node instanceof aws_elasticsearch_1.CfnDomain || node instanceof aws_opensearchservice_1.CfnDomain) {
        const accessPolicies = getPolicyFromCR() ?? aws_cdk_lib_1.Stack.of(node).resolve(node.accessPolicies);
        if (accessPolicies === undefined) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        const statements = accessPolicies?.Statement;
        if (statements === undefined || statements.length === 0) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        for (const statement of statements) {
            if (statement.Effect === 'Allow') {
                const allowList = statement?.Condition?.IpAddress?.['aws:sourceIp'];
                if (allowList === undefined || allowList.length === 0) {
                    return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
                }
            }
        }
        return nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else {
        return nag_rules_1.NagRuleCompliance.NOT_APPLICABLE;
    }
    function getPolicyFromCR() {
        const crPath = node.node.path.split('/Resource').slice(0, -1).join('/Resource') +
            (node instanceof aws_opensearchservice_1.CfnDomain ? '/AccessPolicy' : '/ESAccessPolicy') +
            '/Resource/Default';
        const cr = aws_cdk_lib_1.Stack.of(node)
            .node.findAll()
            .find((c) => c.node.path === crPath);
        if (cr) {
            const props = aws_cdk_lib_1.Stack.of(cr).resolve(cr._cfnProperties);
            const update = props?.Update?.['Fn::Join']?.[1];
            return JSON.parse(JSON.parse(update.join('')).parameters.AccessPolicies);
        }
    }
}, 'name', { value: (0, path_1.parse)(__filename).name });
//# sourceMappingURL=data:application/json;base64,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