"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const path_1 = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_elasticsearch_1 = require("aws-cdk-lib/aws-elasticsearch");
const aws_opensearchservice_1 = require("aws-cdk-lib/aws-opensearchservice");
const nag_rules_1 = require("../../nag-rules");
/**
 * OpenSearch Service domains do not allow for unsigned requests or anonymous access
 * @param node the CfnResource to check
 */
exports.default = Object.defineProperty((node) => {
    if (node instanceof aws_elasticsearch_1.CfnDomain || node instanceof aws_opensearchservice_1.CfnDomain) {
        const accessPolicies = getPolicyFromCR() ?? aws_cdk_lib_1.Stack.of(node).resolve(node.accessPolicies);
        if (accessPolicies == undefined) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        const statements = accessPolicies?.Statement;
        if (statements === undefined || statements.length === 0) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        for (const statement of statements) {
            if (statement.Effect == 'Allow') {
                const awsString = statement.Principal
                    ? JSON.stringify(statement.Principal)
                    : '';
                if (awsString.includes('*')) {
                    return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
                }
            }
        }
        return nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else {
        return nag_rules_1.NagRuleCompliance.NOT_APPLICABLE;
    }
    function getPolicyFromCR() {
        const crPath = node.node.path.split('/Resource').slice(0, -1).join('/Resource') +
            (node instanceof aws_opensearchservice_1.CfnDomain ? '/AccessPolicy' : '/ESAccessPolicy') +
            '/Resource/Default';
        const cr = aws_cdk_lib_1.Stack.of(node)
            .node.findAll()
            .find((c) => c.node.path === crPath);
        if (cr) {
            const props = aws_cdk_lib_1.Stack.of(cr).resolve(cr._cfnProperties);
            const update = props?.Update?.['Fn::Join']?.[1];
            return JSON.parse(JSON.parse(update.join('')).parameters.AccessPolicies);
        }
    }
}, 'name', { value: (0, path_1.parse)(__filename).name });
//# sourceMappingURL=data:application/json;base64,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