"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const path_1 = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_elasticsearch_1 = require("aws-cdk-lib/aws-elasticsearch");
const aws_opensearchservice_1 = require("aws-cdk-lib/aws-opensearchservice");
const nag_rules_1 = require("../../nag-rules");
/**
 * OpenSearch Service domains minimally publish SEARCH_SLOW_LOGS and INDEX_SLOW_LOGS to CloudWatch Logs
 * @param node the CfnResource to check
 */
exports.default = Object.defineProperty((node) => {
    if (node instanceof aws_elasticsearch_1.CfnDomain || node instanceof aws_opensearchservice_1.CfnDomain) {
        const logPublishingOptions = aws_cdk_lib_1.Stack.of(node).resolve(node.logPublishingOptions);
        const requiredSlowLogs = ['SEARCH_SLOW_LOGS', 'INDEX_SLOW_LOGS'];
        const findings = [];
        for (const log of requiredSlowLogs) {
            const resolvedLog = aws_cdk_lib_1.Stack.of(node).resolve(logPublishingOptions?.[log]);
            const enabled = nag_rules_1.NagRules.resolveIfPrimitive(node, resolvedLog?.enabled);
            if (!enabled) {
                findings.push(`LogExport::${log}`);
            }
        }
        return findings.length ? findings : nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else {
        return nag_rules_1.NagRuleCompliance.NOT_APPLICABLE;
    }
}, 'name', { value: (0, path_1.parse)(__filename).name });
//# sourceMappingURL=data:application/json;base64,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