//!
//! Rule MD005: Inconsistent indentation for list items at the same level
//!
//! See [docs/md005.md](../../docs/md005.md) for full documentation, configuration, and examples.

use crate::utils::range_utils::{LineIndex, calculate_match_range};

use crate::rule::{Fix, LintError, LintResult, LintWarning, Rule, RuleCategory, Severity};
use crate::utils::document_structure::DocumentStructure;
// No regex patterns needed for this rule
use std::collections::HashMap;
use toml;

/// Rule MD005: Inconsistent indentation for list items at the same level
#[derive(Clone)]
pub struct MD005ListIndent;

impl MD005ListIndent {
    // Determine the expected indentation for a list item
    // Each nested item should align with the text content of its parent
    #[inline]
    fn get_expected_indent(level: usize, parent_text_position: Option<usize>) -> usize {
        if level == 1 {
            0 // Top level items should be at the start of the line
        } else if let Some(pos) = parent_text_position {
            // Align with parent's text content
            pos
        } else {
            // Fallback to standard nested indentation: 2 spaces per level
            2 * (level - 1)
        }
    }

    /// Get parent info for any list item to determine proper text alignment
    /// Returns parent_text_position where the child should align
    fn get_parent_text_position(
        &self,
        ctx: &crate::lint_context::LintContext,
        current_line: usize,
        current_indent: usize,
    ) -> Option<usize> {
        // Look backward from current line to find parent item
        for line_idx in (1..current_line).rev() {
            if let Some(line_info) = ctx.line_info(line_idx) {
                if let Some(list_item) = &line_info.list_item {
                    // Found a list item - check if it's at a lower indentation (parent level)
                    if list_item.marker_column < current_indent {
                        // This is a parent item - calculate where child should align
                        if list_item.is_ordered {
                            // For ordered lists, align with text start
                            let text_start_pos = list_item.marker_column + list_item.marker.len() + 1; // +1 for space after marker
                            return Some(text_start_pos);
                        } else {
                            // For unordered lists, align with text start
                            let text_start_pos = list_item.marker_column + 2; // "* " or "- " or "+ "
                            return Some(text_start_pos);
                        }
                    }
                }
                // If we encounter non-blank, non-list content at column 0, stop looking
                else if !line_info.is_blank && line_info.indent == 0 {
                    break;
                }
            }
        }
        None
    }

    /// Group related list blocks that should be treated as one logical list structure
    fn group_related_list_blocks<'a>(
        &self,
        list_blocks: &'a [crate::lint_context::ListBlock],
    ) -> Vec<Vec<&'a crate::lint_context::ListBlock>> {
        if list_blocks.is_empty() {
            return Vec::new();
        }

        let mut groups = Vec::new();
        let mut current_group = vec![&list_blocks[0]];

        for i in 1..list_blocks.len() {
            let prev_block = &list_blocks[i - 1];
            let current_block = &list_blocks[i];

            // Check if blocks are consecutive (no significant gap between them)
            let line_gap = current_block.start_line.saturating_sub(prev_block.end_line);

            // Group blocks if they are close together (within 2 lines)
            // This handles cases where mixed list types are split but should be treated together
            if line_gap <= 2 {
                current_group.push(current_block);
            } else {
                // Start a new group
                groups.push(current_group);
                current_group = vec![current_block];
            }
        }
        groups.push(current_group);

        groups
    }

    /// Check a group of related list blocks as one logical list structure
    fn check_list_block_group(
        &self,
        ctx: &crate::lint_context::LintContext,
        group: &[&crate::lint_context::ListBlock],
        warnings: &mut Vec<LintWarning>,
    ) -> Result<(), LintError> {
        let line_index = LineIndex::new(ctx.content.to_string());

        // Collect all list items from all blocks in the group
        let mut all_list_items = Vec::new();

        for list_block in group {
            for &item_line in &list_block.item_lines {
                if let Some(line_info) = ctx.line_info(item_line) {
                    if let Some(list_item) = &line_info.list_item {
                        // Calculate the effective indentation (considering blockquotes)
                        let effective_indent = if let Some(blockquote) = &line_info.blockquote {
                            // For blockquoted lists, use relative indentation within the blockquote
                            list_item.marker_column.saturating_sub(blockquote.nesting_level * 2)
                        } else {
                            // For normal lists, use the marker column directly
                            list_item.marker_column
                        };

                        all_list_items.push((item_line, effective_indent, line_info, list_item));
                    }
                }
            }
        }

        if all_list_items.is_empty() {
            return Ok(());
        }

        // Sort by line number to process in order
        all_list_items.sort_by_key(|(line_num, _, _, _)| *line_num);

        // Determine levels based on indentation progression (like the original algorithm)
        let mut indent_to_level: HashMap<usize, usize> = HashMap::new();

        // Process items to establish level mapping based on nesting structure
        for (_line_num, indent, _line_info, _list_item) in &all_list_items {
            let _level = if indent_to_level.is_empty() {
                // First item establishes level 1
                indent_to_level.insert(*indent, 1);
                1
            } else if let Some(&existing_level) = indent_to_level.get(indent) {
                // This indentation already has a level
                existing_level
            } else {
                // Determine level based on relative indentation and parent-child relationships
                let mut level = 1;
                for (&existing_indent, &existing_level) in &indent_to_level {
                    if existing_indent < *indent {
                        level = level.max(existing_level + 1);
                    }
                }
                indent_to_level.insert(*indent, level);
                level
            };
        }

        // Group items by level and check for consistency within each level
        let mut level_groups: HashMap<usize, Vec<(usize, usize, &crate::lint_context::LineInfo)>> = HashMap::new();
        for (line_num, indent, line_info, _list_item) in &all_list_items {
            let level = indent_to_level[indent];
            level_groups
                .entry(level)
                .or_default()
                .push((*line_num, *indent, *line_info));
        }

        // Process each level group
        for (level, mut group) in level_groups {
            // Sort by line number to process in order
            group.sort_by_key(|(line_num, _, _)| *line_num);

            // Get parent text position for proper alignment
            let parent_text_position = if level > 1 {
                // Get parent info from the first item in the group
                if let Some((line_num, indent, _)) = group.first() {
                    self.get_parent_text_position(ctx, *line_num, *indent)
                } else {
                    None
                }
            } else {
                None
            };

            let expected_indent = Self::get_expected_indent(level, parent_text_position);

            // Check if items in this level have consistent indentation
            let indents: std::collections::HashSet<usize> = group.iter().map(|(_, indent, _)| *indent).collect();

            if indents.len() > 1 {
                // Multiple different indentations at the same level - flag all as inconsistent
                for (line_num, indent, line_info) in &group {
                    let inconsistent_message = format!(
                        "Expected indentation of {} {}, found {}",
                        expected_indent,
                        if expected_indent == 1 { "space" } else { "spaces" },
                        indent
                    );

                    let (start_line, start_col, end_line, end_col) = if *indent > 0 {
                        calculate_match_range(*line_num, &line_info.content, 0, *indent)
                    } else {
                        calculate_match_range(*line_num, &line_info.content, 0, 1)
                    };

                    let fix_range = if *indent > 0 {
                        let start_byte = line_index.line_col_to_byte_range(*line_num, 1).start;
                        let end_byte = line_index.line_col_to_byte_range(*line_num, *indent + 1).start;
                        start_byte..end_byte
                    } else {
                        let byte_pos = line_index.line_col_to_byte_range(*line_num, 1).start;
                        byte_pos..byte_pos
                    };

                    let replacement = if expected_indent > 0 {
                        " ".repeat(expected_indent)
                    } else {
                        String::new()
                    };

                    warnings.push(LintWarning {
                        rule_name: Some(self.name()),
                        line: start_line,
                        column: start_col,
                        end_line,
                        end_column: end_col,
                        message: inconsistent_message,
                        severity: Severity::Warning,
                        fix: Some(Fix {
                            range: fix_range,
                            replacement,
                        }),
                    });
                }
            } else {
                // Single indentation at this level - check if it matches expected
                let actual_indent = indents.iter().next().unwrap();
                if *actual_indent != expected_indent {
                    for (line_num, indent, line_info) in &group {
                        let inconsistent_message = format!(
                            "Expected indentation of {} {}, found {}",
                            expected_indent,
                            if expected_indent == 1 { "space" } else { "spaces" },
                            indent
                        );

                        let (start_line, start_col, end_line, end_col) = if *indent > 0 {
                            calculate_match_range(*line_num, &line_info.content, 0, *indent)
                        } else {
                            calculate_match_range(*line_num, &line_info.content, 0, 1)
                        };

                        let fix_range = if *indent > 0 {
                            let start_byte = line_index.line_col_to_byte_range(*line_num, 1).start;
                            let end_byte = line_index.line_col_to_byte_range(*line_num, *indent + 1).start;
                            start_byte..end_byte
                        } else {
                            let byte_pos = line_index.line_col_to_byte_range(*line_num, 1).start;
                            byte_pos..byte_pos
                        };

                        let replacement = if expected_indent > 0 {
                            " ".repeat(expected_indent)
                        } else {
                            String::new()
                        };

                        warnings.push(LintWarning {
                            rule_name: Some(self.name()),
                            line: start_line,
                            column: start_col,
                            end_line,
                            end_column: end_col,
                            message: inconsistent_message,
                            severity: Severity::Warning,
                            fix: Some(Fix {
                                range: fix_range,
                                replacement,
                            }),
                        });
                    }
                }
            }
        }

        Ok(())
    }

    /// Migrated to use centralized list blocks for better performance and accuracy
    fn check_optimized(&self, ctx: &crate::lint_context::LintContext) -> LintResult {
        let content = ctx.content;

        // Early returns for common cases
        if content.is_empty() {
            return Ok(Vec::new());
        }

        // Quick check for any list blocks before processing
        if ctx.list_blocks.is_empty() {
            return Ok(Vec::new());
        }

        let mut warnings = Vec::new();

        // Group consecutive list blocks that should be treated as one logical structure
        // This is needed because mixed list types (ordered/unordered) get split into separate blocks
        let block_groups = self.group_related_list_blocks(&ctx.list_blocks);

        for group in block_groups {
            self.check_list_block_group(ctx, &group, &mut warnings)?;
        }

        Ok(warnings)
    }
}

impl Default for MD005ListIndent {
    fn default() -> Self {
        Self
    }
}

impl Rule for MD005ListIndent {
    fn name(&self) -> &'static str {
        "MD005"
    }

    fn description(&self) -> &'static str {
        "List indentation should be consistent"
    }

    fn check(&self, ctx: &crate::lint_context::LintContext) -> LintResult {
        // Use optimized version
        self.check_optimized(ctx)
    }

    fn fix(&self, ctx: &crate::lint_context::LintContext) -> Result<String, LintError> {
        let warnings = self.check(ctx)?;
        if warnings.is_empty() {
            return Ok(ctx.content.to_string());
        }

        // Sort warnings by position (descending) to apply from end to start
        let mut warnings_with_fixes: Vec<_> = warnings
            .into_iter()
            .filter_map(|w| w.fix.clone().map(|fix| (w, fix)))
            .collect();
        warnings_with_fixes.sort_by_key(|(_, fix)| std::cmp::Reverse(fix.range.start));

        // Apply fixes to content
        let mut content = ctx.content.to_string();
        for (_, fix) in warnings_with_fixes {
            if fix.range.start <= content.len() && fix.range.end <= content.len() {
                content.replace_range(fix.range, &fix.replacement);
            }
        }

        Ok(content)
    }

    fn category(&self) -> RuleCategory {
        RuleCategory::List
    }

    /// Check if this rule should be skipped
    fn should_skip(&self, ctx: &crate::lint_context::LintContext) -> bool {
        // Skip if content is empty or has no list items
        ctx.content.is_empty() || !ctx.lines.iter().any(|line| line.list_item.is_some())
    }

    /// Optimized check using document structure
    fn check_with_structure(
        &self,
        ctx: &crate::lint_context::LintContext,
        structure: &DocumentStructure,
    ) -> LintResult {
        // If no lists in structure, return early
        if structure.list_lines.is_empty() {
            return Ok(Vec::new());
        }

        // Use optimized check - it's already efficient enough
        self.check_optimized(ctx)
    }

    fn as_any(&self) -> &dyn std::any::Any {
        self
    }

    fn as_maybe_document_structure(&self) -> Option<&dyn crate::rule::MaybeDocumentStructure> {
        Some(self)
    }

    fn default_config_section(&self) -> Option<(String, toml::Value)> {
        None
    }

    fn from_config(_config: &crate::config::Config) -> Box<dyn Rule>
    where
        Self: Sized,
    {
        Box::new(MD005ListIndent)
    }
}

impl crate::utils::document_structure::DocumentStructureExtensions for MD005ListIndent {
    fn has_relevant_elements(
        &self,
        _ctx: &crate::lint_context::LintContext,
        doc_structure: &crate::utils::document_structure::DocumentStructure,
    ) -> bool {
        !doc_structure.list_lines.is_empty()
    }
}

#[cfg(test)]
mod tests {
    use super::*;
    use crate::lint_context::LintContext;
    use crate::utils::document_structure::DocumentStructureExtensions;

    #[test]
    fn test_valid_unordered_list() {
        let rule = MD005ListIndent;
        let content = "\
* Item 1
* Item 2
  * Nested 1
  * Nested 2
* Item 3";
        let ctx = LintContext::new(content);
        let result = rule.check(&ctx).unwrap();
        assert!(result.is_empty());
    }

    #[test]
    fn test_valid_ordered_list() {
        let rule = MD005ListIndent;
        let content = "\
1. Item 1
2. Item 2
   1. Nested 1
   2. Nested 2
3. Item 3";
        let ctx = LintContext::new(content);
        let result = rule.check(&ctx).unwrap();
        // With dynamic alignment, nested items should align with parent's text content
        // Ordered items starting with "1. " have text at column 3, so nested items need 3 spaces
        assert!(result.is_empty());
    }

    #[test]
    fn test_invalid_unordered_indent() {
        let rule = MD005ListIndent;
        let content = "\
* Item 1
 * Item 2
   * Nested 1";
        let ctx = LintContext::new(content);
        let result = rule.check(&ctx).unwrap();
        // With dynamic alignment, line 3 correctly aligns with line 2's text position
        // Only line 2 is incorrectly indented
        assert_eq!(result.len(), 1);
        let fixed = rule.fix(&ctx).unwrap();
        assert_eq!(fixed, "* Item 1\n  * Item 2\n   * Nested 1");
    }

    #[test]
    fn test_invalid_ordered_indent() {
        let rule = MD005ListIndent;
        let content = "\
1. Item 1
 2. Item 2
    1. Nested 1";
        let ctx = LintContext::new(content);
        let result = rule.check(&ctx).unwrap();
        assert_eq!(result.len(), 1);
        let fixed = rule.fix(&ctx).unwrap();
        // With dynamic alignment, ordered items align with parent's text content
        // Line 1 text starts at col 3, so line 2 should have 3 spaces
        // Line 3 already correctly aligns with line 2's text position
        assert_eq!(fixed, "1. Item 1\n   2. Item 2\n    1. Nested 1");
    }

    #[test]
    fn test_mixed_list_types() {
        let rule = MD005ListIndent;
        let content = "\
* Item 1
  1. Nested ordered
  * Nested unordered
* Item 2";
        let ctx = LintContext::new(content);
        let result = rule.check(&ctx).unwrap();
        assert!(result.is_empty());
    }

    #[test]
    fn test_multiple_levels() {
        let rule = MD005ListIndent;
        let content = "\
* Level 1
   * Level 2
      * Level 3";
        let ctx = LintContext::new(content);
        let result = rule.check(&ctx).unwrap();
        assert_eq!(result.len(), 2);
        let fixed = rule.fix(&ctx).unwrap();
        // With dynamic alignment:
        // Level 2 aligns with Level 1's text (2 spaces)
        // Level 3 aligns with Level 2's text (5 spaces: 2 + "* " + 1)
        assert_eq!(
            fixed,
            "\
* Level 1
  * Level 2
     * Level 3"
        );
    }

    #[test]
    fn test_empty_lines() {
        let rule = MD005ListIndent;
        let content = "\
* Item 1

  * Nested 1

* Item 2";
        let ctx = LintContext::new(content);
        let result = rule.check(&ctx).unwrap();
        assert!(result.is_empty());
    }

    #[test]
    fn test_no_lists() {
        let rule = MD005ListIndent;
        let content = "\
Just some text
More text
Even more text";
        let ctx = LintContext::new(content);
        let result = rule.check(&ctx).unwrap();
        assert!(result.is_empty());
    }

    #[test]
    fn test_complex_nesting() {
        let rule = MD005ListIndent;
        let content = "\
* Level 1
  * Level 2
    * Level 3
  * Back to 2
    1. Ordered 3
    2. Still 3
* Back to 1";
        let ctx = LintContext::new(content);
        let result = rule.check(&ctx).unwrap();
        assert!(result.is_empty());
    }

    #[test]
    fn test_invalid_complex_nesting() {
        let rule = MD005ListIndent;
        let content = "\
* Level 1
   * Level 2
     * Level 3
   * Back to 2
      1. Ordered 3
     2. Still 3
* Back to 1";
        let ctx = LintContext::new(content);
        let result = rule.check(&ctx).unwrap();
        // With dynamic alignment, fewer items need correction
        // Lines 2,4: should align with Level 1's text (2 spaces)
        // Line 5: should align with "Back to 2"'s text (5 spaces)
        assert_eq!(result.len(), 3);
        let fixed = rule.fix(&ctx).unwrap();
        assert_eq!(
            fixed,
            "* Level 1\n  * Level 2\n     * Level 3\n  * Back to 2\n     1. Ordered 3\n     2. Still 3\n* Back to 1"
        );
    }

    #[test]
    fn test_with_document_structure() {
        let rule = MD005ListIndent;

        // Test with consistent list indentation
        let content = "* Item 1\n* Item 2\n  * Nested item\n  * Another nested item";
        let structure = DocumentStructure::new(content);
        let ctx = LintContext::new(content);
        let result = rule.check_with_structure(&ctx, &structure).unwrap();
        assert!(result.is_empty());

        // Test with inconsistent list indentation
        let content = "* Item 1\n* Item 2\n * Nested item\n  * Another nested item";
        let structure = DocumentStructure::new(content);
        let ctx = LintContext::new(content);
        let result = rule.check_with_structure(&ctx, &structure).unwrap();
        assert!(!result.is_empty()); // Should have at least one warning

        // Test with different level indentation issues
        let content = "* Item 1\n  * Nested item\n * Another nested item with wrong indent";
        let structure = DocumentStructure::new(content);
        let ctx = LintContext::new(content);
        let result = rule.check_with_structure(&ctx, &structure).unwrap();
        assert!(!result.is_empty()); // Should have at least one warning
    }

    // Additional comprehensive tests
    #[test]
    fn test_list_with_continuations() {
        let rule = MD005ListIndent;
        let content = "\
* Item 1
  This is a continuation
  of the first item
  * Nested item
    with its own continuation
* Item 2";
        let ctx = LintContext::new(content);
        let result = rule.check(&ctx).unwrap();
        assert!(result.is_empty());
    }

    #[test]
    fn test_list_in_blockquote() {
        let rule = MD005ListIndent;
        let content = "\
> * Item 1
>   * Nested 1
>   * Nested 2
> * Item 2";
        let ctx = LintContext::new(content);
        let result = rule.check(&ctx).unwrap();

        // Blockquoted lists should have correct indentation within the blockquote context
        assert!(
            result.is_empty(),
            "Expected no warnings for correctly indented blockquote list, got: {result:?}"
        );
    }

    #[test]
    fn test_list_with_code_blocks() {
        let rule = MD005ListIndent;
        let content = "\
* Item 1
  ```
  code block
  ```
  * Nested item
* Item 2";
        let ctx = LintContext::new(content);
        let result = rule.check(&ctx).unwrap();
        assert!(result.is_empty());
    }

    #[test]
    fn test_list_with_tabs() {
        let rule = MD005ListIndent;
        let content = "* Item 1\n\t* Tab indented\n  * Space indented";
        let ctx = LintContext::new(content);
        let result = rule.check(&ctx).unwrap();
        // Should detect inconsistent indentation
        assert!(!result.is_empty());
    }

    #[test]
    fn test_inconsistent_at_same_level() {
        let rule = MD005ListIndent;
        let content = "\
* Item 1
  * Nested 1
  * Nested 2
   * Wrong indent for same level
  * Nested 3";
        let ctx = LintContext::new(content);
        let result = rule.check(&ctx).unwrap();
        assert!(!result.is_empty());
        // Should flag the inconsistent item
        assert!(result.iter().any(|w| w.line == 4));
    }

    #[test]
    fn test_zero_indent_top_level() {
        let rule = MD005ListIndent;
        let content = "\
 * Wrong indent
* Correct
  * Nested";
        let ctx = LintContext::new(content);
        let result = rule.check(&ctx).unwrap();

        // The current implementation accepts lists that start indented
        // It treats the first item as establishing the base indent level
        // This is reasonable behavior - not all lists must start at column 0
        assert_eq!(result.len(), 0);
    }

    #[test]
    fn test_fix_preserves_content() {
        let rule = MD005ListIndent;
        let content = "\
* Item with **bold** and *italic*
 * Wrong indent with `code`
   * Also wrong with [link](url)";
        let ctx = LintContext::new(content);
        let fixed = rule.fix(&ctx).unwrap();
        assert!(fixed.contains("**bold**"));
        assert!(fixed.contains("*italic*"));
        assert!(fixed.contains("`code`"));
        assert!(fixed.contains("[link](url)"));
    }

    #[test]
    fn test_deeply_nested_lists() {
        let rule = MD005ListIndent;
        let content = "\
* L1
  * L2
    * L3
      * L4
        * L5
          * L6";
        let ctx = LintContext::new(content);
        let result = rule.check(&ctx).unwrap();
        assert!(result.is_empty());
    }

    #[test]
    fn test_fix_multiple_issues() {
        let rule = MD005ListIndent;
        let content = "\
* Item 1
 * Wrong 1
   * Wrong 2
    * Wrong 3
  * Correct
   * Wrong 4";
        let ctx = LintContext::new(content);
        let fixed = rule.fix(&ctx).unwrap();
        // With dynamic alignment, items align with their parent's text content
        let lines: Vec<&str> = fixed.lines().collect();
        assert_eq!(lines[0], "* Item 1");
        assert_eq!(lines[1], "  * Wrong 1");
        assert_eq!(lines[2], "   * Wrong 2"); // Aligns with line 2's text
        assert_eq!(lines[3], "     * Wrong 3"); // Aligns with line 3's text
        assert_eq!(lines[4], "   * Correct"); // Back to level 2, aligns with line 1's text
        assert_eq!(lines[5], "   * Wrong 4"); // Same level as "Correct"
    }

    #[test]
    fn test_performance_large_document() {
        let rule = MD005ListIndent;
        let mut content = String::new();
        for i in 0..100 {
            content.push_str(&format!("* Item {i}\n"));
            content.push_str(&format!("  * Nested {i}\n"));
        }
        let ctx = LintContext::new(&content);
        let result = rule.check(&ctx).unwrap();
        assert!(result.is_empty());
    }

    #[test]
    fn test_column_positions() {
        let rule = MD005ListIndent;
        let content = " * Wrong indent";
        let ctx = LintContext::new(content);
        let result = rule.check(&ctx).unwrap();
        assert_eq!(result.len(), 1);
        assert_eq!(result[0].column, 1);
        assert_eq!(result[0].end_column, 2);
    }

    #[test]
    fn test_should_skip() {
        let rule = MD005ListIndent;

        // Empty content should skip
        let ctx = LintContext::new("");
        assert!(rule.should_skip(&ctx));

        // Content without lists should skip
        let ctx = LintContext::new("Just plain text");
        assert!(rule.should_skip(&ctx));

        // Content with lists should not skip
        let ctx = LintContext::new("* List item");
        assert!(!rule.should_skip(&ctx));

        let ctx = LintContext::new("1. Ordered list");
        assert!(!rule.should_skip(&ctx));
    }

    #[test]
    fn test_has_relevant_elements() {
        let rule = MD005ListIndent;
        let content = "* List item";
        let ctx = LintContext::new(content);
        let doc_structure = DocumentStructure::new(content);
        assert!(rule.has_relevant_elements(&ctx, &doc_structure));

        let content = "No lists here";
        let ctx = LintContext::new(content);
        let doc_structure = DocumentStructure::new(content);
        assert!(!rule.has_relevant_elements(&ctx, &doc_structure));
    }

    #[test]
    fn test_edge_case_single_space_indent() {
        let rule = MD005ListIndent;
        let content = "\
* Item 1
 * Single space - wrong
  * Two spaces - correct";
        let ctx = LintContext::new(content);
        let result = rule.check(&ctx).unwrap();
        // Both the single space and two space items get warnings
        // because they establish inconsistent indentation at the same level
        assert_eq!(result.len(), 2);
        assert!(result.iter().any(|w| w.line == 2 && w.message.contains("found 1")));
    }

    #[test]
    fn test_edge_case_three_space_indent() {
        let rule = MD005ListIndent;
        let content = "\
* Item 1
   * Three spaces - wrong
  * Two spaces - correct";
        let ctx = LintContext::new(content);
        let result = rule.check(&ctx).unwrap();
        // Both items get warnings due to inconsistent indentation
        assert_eq!(result.len(), 2);
        assert!(result.iter().any(|w| w.line == 2 && w.message.contains("found 3")));
    }

    #[test]
    fn test_nested_bullets_under_numbered_items() {
        let rule = MD005ListIndent;
        let content = "\
1. **Active Directory/LDAP**
   - User authentication and directory services
   - LDAP for user information and validation

2. **Oracle Unified Directory (OUD)**
   - Extended user directory services
   - Verification of project account presence and changes";
        let ctx = LintContext::new(content);
        let result = rule.check(&ctx).unwrap();
        // Should have no warnings - 3 spaces is correct for bullets under numbered items
        assert!(
            result.is_empty(),
            "Expected no warnings for bullets with 3 spaces under numbered items, got: {result:?}"
        );
    }

    #[test]
    fn test_nested_bullets_under_numbered_items_wrong_indent() {
        let rule = MD005ListIndent;
        let content = "\
1. **Active Directory/LDAP**
  - Wrong: only 2 spaces
   - Correct: 3 spaces";
        let ctx = LintContext::new(content);
        let result = rule.check(&ctx).unwrap();
        // Should flag the 2-space indentation as wrong
        assert_eq!(result.len(), 2); // Both items flagged due to inconsistency
        assert!(result.iter().any(|w| w.line == 2 && w.message.contains("found 2")));
    }

    #[test]
    fn test_regular_nested_bullets_still_work() {
        let rule = MD005ListIndent;
        let content = "\
* Top level
  * Second level (2 spaces is correct for bullets under bullets)
    * Third level (4 spaces)";
        let ctx = LintContext::new(content);
        let result = rule.check(&ctx).unwrap();
        // Should have no warnings - regular bullet nesting still uses 2-space increments
        assert!(
            result.is_empty(),
            "Expected no warnings for regular bullet nesting, got: {result:?}"
        );
    }

    #[test]
    fn test_fix_range_accuracy() {
        let rule = MD005ListIndent;
        let content = " * Wrong indent";
        let ctx = LintContext::new(content);
        let result = rule.check(&ctx).unwrap();
        assert_eq!(result.len(), 1);

        let fix = result[0].fix.as_ref().unwrap();
        // Fix should replace the single space with nothing (0 indent for level 1)
        assert_eq!(fix.replacement, "");
    }
}
